/*
 *  $Id: cmap_curvefilter.c 29039 2025-12-26 17:22:49Z yeti-dn $
 *  Copyright (C) 2021 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwynlfit.h>
#include <libgwyddion/gwythreads.h>
#include <libprocess/lawn.h>
#include <libprocess/gwyprocesstypes.h>
#include <libgwydgets/gwydataview.h>
#include <libgwydgets/gwygraph.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-cmap.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "libgwyddion/gwyomp.h"

#define RUN_MODES (GWY_RUN_INTERACTIVE)

enum {
    PARAM_KEEP_CURVES,
};

typedef struct {
    GwyParams *params;
    GwyLawn *lawn;
    GwyLawn *result;
    const GwyEnum *segnames;
    guint nresults;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GSList *keep_curves;
    GtkWidget *keep_curves_label;
    GwyContainer *data;
    GwyEnum *curve_enum;
} ModuleGUI;

static gboolean         module_register         (void);
static GwyParamDef*     define_module_params    (void);
static void             cmap_curvefilter             (GwyContainer *data,
                                                 GwyRunType runtype);
static gboolean         execute                 (ModuleArgs *args);
static GwyDialogOutcome run_gui                 (ModuleArgs *args,
                                                 GwyContainer *data,
                                                 gint id);
static GtkWidget*       create_keep_curves      (ModuleGUI *gui);
static void             keep_curves_changed     (GtkRadioButton *button,
                                                 ModuleGUI *gui);
static void             sanitise_params         (ModuleArgs *args);


static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    /* XXX: Why the hell is ‘Curve Map’ in titlecase here? */
    N_("Removes some curves from Curve Map"),
    "Petr KLapetek <klapetek@gwyddion.net>",
    "1.0",
    "Petr Klapetek & David Nečas (Yeti)",
    "2025",
};

GWY_MODULE_QUERY2(module_info, cmap_curvefilter)

static gboolean
module_register(void)
{
    gwy_curve_map_func_register("cmap_curvefilter",
                                (GwyCurveMapFunc)&cmap_curvefilter,
                                /* XXX: Wrong capitalisation, Curves should be in Titlecase. */
                                N_("/_Basic Operations/_Filter curves..."),
                                NULL,
                                RUN_MODES,
                                GWY_MENU_FLAG_CURVE_MAP,
                                /* XXX: Why the hell is ‘Curve Map’ in titlecase here? */
                                N_("Remove some curves from Curve Map"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_curve_map_func_current());
    gwy_param_def_add_int(paramdef, PARAM_KEEP_CURVES, "keep_curves", _("Keep curves"), 0, G_MAXINT, 1);
    return paramdef;
}

static void
cmap_curvefilter(GwyContainer *data, GwyRunType runtype)
{
    ModuleArgs args;
    GwyLawn *lawn = NULL;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    gint oldid, newid;

    g_return_if_fail(runtype & RUN_MODES);
    g_return_if_fail(g_type_from_name("GwyLayerPoint"));

    gwy_clear(&args, 1);
    gwy_app_data_browser_get_current(GWY_APP_LAWN, &lawn,
                                     GWY_APP_LAWN_ID, &oldid,
                                     0);
    g_return_if_fail(GWY_IS_LAWN(lawn));
    args.lawn = lawn;
    args.params = gwy_params_new_from_settings(define_module_params());
    sanitise_params(&args);

    if (runtype == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, oldid);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (execute(&args)) {
        if (args.result) {
            newid = gwy_app_data_browser_add_lawn(args.result, NULL, data, TRUE);
            gwy_container_set_const_string(data, gwy_app_get_lawn_title_key_for_id(newid), "output"/*args.segnames[i].name*/);
            GWY_OBJECT_UNREF(args.result);
        }
    }

end:
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, G_GNUC_UNUSED GwyContainer *data, G_GNUC_UNUSED gint id)
{
    GtkWidget *kclist;
    GwyDialog *dialog;
    ModuleGUI gui;
    GwyDialogOutcome outcome;

    gwy_clear(&gui, 1);
    gui.args = args;
    gui.data = gwy_container_new();

    gui.dialog = gwy_dialog_new(_("Cut to Segments"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    kclist = create_keep_curves(&gui);
    gwy_dialog_add_content(GWY_DIALOG(gui.dialog), kclist, FALSE, FALSE, 0);

    outcome = gwy_dialog_run(dialog);

    g_object_unref(gui.data);
    g_free(gui.curve_enum);

    return outcome;
}

static GtkWidget*
create_keep_curves(ModuleGUI *gui)
{
    GtkTable *table;
    GwyLawn *lawn = gui->args->lawn;
    gint keep_curves = gwy_params_get_int(gui->args->params, PARAM_KEEP_CURVES);
    const gchar *name;
    gint ncurves, i;

    ncurves = gwy_lawn_get_n_curves(lawn);
    gui->curve_enum = g_new(GwyEnum, ncurves);
    for (i = 0; i < ncurves; i++) {
        name = gwy_lawn_get_curve_label(lawn, i);
        gui->curve_enum[i].value = (1 << i);
        gui->curve_enum[i].name = name ? name : _("Untitled");
    }
    gui->keep_curves = gwy_check_boxes_create(gui->curve_enum, ncurves, G_CALLBACK(keep_curves_changed), gui,
                                              keep_curves);

    table = GTK_TABLE(gtk_table_new(ncurves+1, 2, FALSE));
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_table_set_row_spacings(table, 2);
    gtk_table_set_col_spacings(table, 6);
    gui->keep_curves_label = gtk_label_new(_("Keep curves:"));
    gtk_misc_set_alignment(GTK_MISC(gui->keep_curves_label), 0.0, 0.5);
    gtk_table_attach(table, gui->keep_curves_label, 0, 2, 0, 1, 0, 0, 0, 0);
    gwy_check_boxes_attach_to_table(gui->keep_curves, table, 2, 1);

    return GTK_WIDGET(table);
}

static void
keep_curves_changed(G_GNUC_UNUSED GtkRadioButton *button, ModuleGUI *gui)
{
    gwy_params_set_int(gui->args->params, PARAM_KEEP_CURVES, gwy_check_boxes_get_selected(gui->keep_curves));
}

static gboolean
execute(ModuleArgs *args)
{
    GwyLawn *lawn = args->lawn, *result;
    GwyParams *params = args->params;
    guint keep_curves = gwy_params_get_int(params, PARAM_KEEP_CURVES);
    gint xres = gwy_lawn_get_xres(lawn), yres = gwy_lawn_get_yres(lawn);
    gint ncurves = gwy_lawn_get_n_curves(lawn);
    const gdouble *cd;
    gdouble *ncd;
    gint i, k, m, n, ndata, nc, col, row;

    nc = 0;
    for (m = 0; m < ncurves; m++) {
        if (keep_curves & (1 << m))
            nc++;
    }

    if (!nc) {
        args->result = NULL;
        return FALSE;
    }

    result = gwy_lawn_new(xres, yres, gwy_lawn_get_xreal(lawn), gwy_lawn_get_yreal(lawn), 
                          nc, gwy_lawn_get_n_segments(lawn));
    gwy_lawn_set_xoffset(result, gwy_lawn_get_xoffset(lawn));
    gwy_lawn_set_yoffset(result, gwy_lawn_get_yoffset(lawn));
    gwy_si_unit_assign(gwy_lawn_get_si_unit_xy(result), gwy_lawn_get_si_unit_xy(lawn));
    for (m = n = 0; m < ncurves; m++) {
        if (keep_curves & (1 << m)) { //keep mth curve as nth one
            gwy_si_unit_assign(gwy_lawn_get_si_unit_curve(result, n), gwy_lawn_get_si_unit_curve(lawn, m));
            gwy_lawn_set_curve_label(result, n, gwy_lawn_get_curve_label(lawn, m));
            n++;
        }
    }

    for (k = 0; k < xres*yres; k++) {
        col = k % xres;
        row = k/xres;
        ndata = gwy_lawn_get_curve_length(lawn, col, row);
        ncd = g_new(gdouble, nc*ndata);

        for (m = n = 0; m < ncurves; m++) {
            if (keep_curves & (1 << m)) {
               cd = gwy_lawn_get_curve_data_const(lawn, col, row, m, NULL);
               for (i = 0; i < ndata; i++)
                   ncd[n*ndata + i] = cd[i];
               n++;
            }
        }
        gwy_lawn_set_curves(result, col, row, ndata, ncd, gwy_lawn_get_segments(lawn, col, row, NULL));
    }
    for (n = 0; n < gwy_lawn_get_n_segments(result); n++)
        gwy_lawn_set_segment_label(result, n, gwy_lawn_get_segment_label(lawn, n));


    args->result = result;

    return TRUE;
}

static void
sanitise_params(ModuleArgs *args)
{
    GwyParams *params = args->params;
    GwyLawn *lawn = args->lawn;
    gint ncurves, keep_curves;

    ncurves = gwy_lawn_get_n_curves(lawn);
    keep_curves = gwy_params_get_int(params, PARAM_KEEP_CURVES);
    gwy_params_set_int(params, PARAM_KEEP_CURVES, keep_curves & ((1 << ncurves) - 1));
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
