/*
 *  $Id: cmap_filter.c 29036 2025-12-22 19:19:15Z yeti-dn $
 *  Copyright (C) 2021 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libprocess/lawn.h>
#include <libprocess/stats.h>
#include <libprocess/correct.h>
#include <libprocess/gwyprocesstypes.h>
#include <libgwydgets/gwydataview.h>
#include <libgwydgets/gwygraph.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-cmap.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "libgwyddion/gwyomp.h"

#define RUN_MODES (GWY_RUN_INTERACTIVE)

enum {
    PREVIEW_SIZE = 360,
};

enum {
    PARAM_RANGE_FROM,
    PARAM_RANGE_TO,
    PARAM_ABSCISSA,
    PARAM_ORDINATE,
    PARAM_SEGMENT,
    PARAM_ENABLE_SEGMENT,
    PARAM_XPOS,
    PARAM_YPOS,
    PARAM_REMOVE,
};

typedef enum {
    REMOVE_MARKED    = 0,
    REMOVE_UNMARKED  = 1,
} FilterRemove;


typedef struct {
    GwyParams *params;
    GwyLawn *lawn;
    GwyLawn *result;
    GwyDataField *field;
    GwySIValueFormat *vf;
    gint nsegments;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
    GwyParamTable *table_output;
    GwyContainer *data;
    GwySelection *selection;
    GwySelection *graph_selection;
    GwyGraphModel *gmodel;
} ModuleGUI;

static gboolean         module_register         (void);
static GwyParamDef*     define_module_params    (void);
static void             filter                  (GwyContainer *data,
                                                 GwyRunType runtype);
static void             execute                 (ModuleArgs *args);
static GwyDialogOutcome run_gui                 (ModuleArgs *args,
                                                 GwyContainer *data,
                                                 gint id);
static void             param_changed           (ModuleGUI *gui,
                                                 gint id);
static void             set_selection           (ModuleGUI *gui);
static void             point_selection_changed (ModuleGUI *gui,
                                                 gint id,
                                                 GwySelection *selection);
static void             graph_selected          (GwySelection* selection,
                                                 gint i,
                                                 ModuleGUI *gui);
static void             extract_one_curve       (ModuleGUI *gui);
static void             update_graph_model_props(GwyGraphModel *gmodel,
                                                 ModuleArgs *args);
static void             sanitise_params         (ModuleArgs *args);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Filters curve data on basis of abscissa."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2025",
};

GWY_MODULE_QUERY2(module_info, cmap_filter)

static gboolean
module_register(void)
{
    gwy_curve_map_func_register("cmap_filter",
                                (GwyCurveMapFunc)&filter,
                                N_("/Filter by _Value..."),
                                NULL,
                                RUN_MODES,
                                GWY_MENU_FLAG_CURVE_MAP,
                                N_("Filter all curves by abscissa values"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;
    static const GwyEnum removes[] = {
        { N_("Marked"),   REMOVE_MARKED,    },
        { N_("Unmarked"), REMOVE_UNMARKED,  },
    };

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_curve_map_func_current());
    gwy_param_def_add_lawn_curve(paramdef, PARAM_ABSCISSA, "abscissa", _("Abscissa"));
    gwy_param_def_add_lawn_curve(paramdef, PARAM_ORDINATE, "ordinate", _("Ordinate"));
    gwy_param_def_add_int(paramdef, PARAM_XPOS, "xpos", NULL, -1, G_MAXINT, -1);
    gwy_param_def_add_int(paramdef, PARAM_YPOS, "ypos", NULL, -1, G_MAXINT, -1);
    gwy_param_def_add_double(paramdef, PARAM_RANGE_FROM, "from", _("_From"), -G_MAXDOUBLE, G_MAXDOUBLE, 0.0);
    gwy_param_def_add_double(paramdef, PARAM_RANGE_TO, "to", _("_To"), -G_MAXDOUBLE, G_MAXDOUBLE, 1.0);
    gwy_param_def_add_lawn_segment(paramdef, PARAM_SEGMENT, "segment", NULL);
    gwy_param_def_add_boolean(paramdef, PARAM_ENABLE_SEGMENT, "enable_segment", NULL, FALSE);
    gwy_param_def_add_gwyenum(paramdef, PARAM_REMOVE, "remove", _("Remove"),
                              removes, G_N_ELEMENTS(removes), (1 << REMOVE_MARKED));

    return paramdef;
}

static void
filter(GwyContainer *data, GwyRunType runtype)
{
    ModuleArgs args;
    GwyLawn *lawn = NULL;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    gint id, newid;

    g_return_if_fail(runtype & RUN_MODES);
    g_return_if_fail(g_type_from_name("GwyLayerPoint"));

    gwy_clear(&args, 1);
    gwy_app_data_browser_get_current(GWY_APP_LAWN, &lawn,
                                     GWY_APP_LAWN_ID, &id,
                                     0);
    g_return_if_fail(GWY_IS_LAWN(lawn));

    args.lawn = lawn;
    args.nsegments = gwy_lawn_get_n_segments(lawn);
    args.params = gwy_params_new_from_settings(define_module_params());
    sanitise_params(&args);

    args.vf = gwy_lawn_get_value_format_curve(lawn, 0, GWY_SI_UNIT_FORMAT_VFMARKUP, NULL);
    args.field = gwy_data_field_new(gwy_lawn_get_xres(lawn), gwy_lawn_get_yres(lawn),
                                    gwy_lawn_get_xreal(lawn), gwy_lawn_get_yreal(lawn), TRUE);
    gwy_data_field_set_xoffset(args.field, gwy_lawn_get_xoffset(lawn));
    gwy_data_field_set_yoffset(args.field, gwy_lawn_get_yoffset(lawn));
    gwy_si_unit_assign(gwy_data_field_get_si_unit_xy(args.field), gwy_lawn_get_si_unit_xy(lawn));

    if (runtype == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, id);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    execute(&args);

    if (args.result) {
        newid = gwy_app_data_browser_add_lawn(args.result, NULL, data, TRUE);
        gwy_container_set_const_string(data, gwy_app_get_lawn_title_key_for_id(newid), "filtered");
    }
    gwy_app_curve_map_log_add(data, -1, id, "cmap::cmap_filter", NULL);

end:
    GWY_OBJECT_UNREF(args.result);
    g_object_unref(args.field);
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyContainer *data, gint id)
{
    GtkWidget *hbox, *graph, *dataview, *align, *area;
    GwyParamTable *table;
    GwyDialog *dialog;
    ModuleGUI gui;
    GwyDataField *field;
    GwyDialogOutcome outcome;
    GwyGraphCurveModel *gcmodel;
    const guchar *gradient;

    gwy_clear(&gui, 1);
    gui.args = args;
    gui.data = gwy_container_new();
    field = gwy_container_get_object(data, gwy_app_get_lawn_preview_key_for_id(id));
    gwy_container_set_object(gui.data, gwy_app_get_data_key_for_id(0), field);
    if (gwy_container_gis_string(data, gwy_app_get_lawn_palette_key_for_id(id), &gradient))
        gwy_container_set_const_string(gui.data, gwy_app_get_data_palette_key_for_id(0), gradient);

    gui.dialog = gwy_dialog_new(_("Filter by Abscissa Value"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    hbox = gwy_hbox_new(0);
    gwy_dialog_add_content(GWY_DIALOG(gui.dialog), hbox, TRUE, TRUE, 0);

    align = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 0);

    dataview = gwy_create_preview(gui.data, 0, PREVIEW_SIZE, FALSE);
    gtk_container_add(GTK_CONTAINER(align), dataview);
    gui.selection = gwy_create_preview_vector_layer(GWY_DATA_VIEW(dataview), 0, "Point", 1, TRUE);
    set_selection(&gui);

    gui.gmodel = gwy_graph_model_new();
    gcmodel = gwy_graph_curve_model_new();
    g_object_set(gcmodel,
                 "mode", GWY_GRAPH_CURVE_LINE,
                 "color", gwy_graph_get_preset_color(0),
                 "description", g_strdup(_("data")),
                 NULL);
    gwy_graph_model_add_curve(gui.gmodel, gcmodel);
    g_object_unref(gcmodel);

    graph = gwy_graph_new(gui.gmodel);
    area = gwy_graph_get_area(GWY_GRAPH(graph));
    gwy_graph_enable_user_input(GWY_GRAPH(graph), FALSE);
    gwy_graph_area_set_status(GWY_GRAPH_AREA(area), GWY_GRAPH_STATUS_XSEL);
    gwy_graph_area_set_selection_editable(GWY_GRAPH_AREA(area), TRUE);
    gui.graph_selection = gwy_graph_area_get_selection(GWY_GRAPH_AREA(area), GWY_GRAPH_STATUS_XSEL);
    gtk_widget_set_size_request(graph, PREVIEW_SIZE, PREVIEW_SIZE);
    gtk_box_pack_start(GTK_BOX(hbox), graph, TRUE, TRUE, 0);

    hbox = gwy_hbox_new(20);
    gwy_dialog_add_content(GWY_DIALOG(gui.dialog), hbox, TRUE, TRUE, 4);

    table = gui.table = gwy_param_table_new(args->params);
    gwy_param_table_append_lawn_curve(table, PARAM_ABSCISSA, args->lawn);
    gwy_param_table_append_lawn_curve(table, PARAM_ORDINATE, args->lawn);
    gwy_param_table_append_slider(table, PARAM_RANGE_FROM);
    gwy_param_table_set_unitstr(table, PARAM_RANGE_FROM, args->vf->units);
    gwy_param_table_append_slider(table, PARAM_RANGE_TO);
    gwy_param_table_set_unitstr(table, PARAM_RANGE_TO, args->vf->units);
    if (args->nsegments) {
        gwy_param_table_append_lawn_segment(table, PARAM_SEGMENT, args->lawn);
        gwy_param_table_add_enabler(table, PARAM_ENABLE_SEGMENT, PARAM_SEGMENT);
    }

    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    table = gui.table_output = gwy_param_table_new(args->params);
    gwy_param_table_append_radio(table, PARAM_REMOVE);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    g_signal_connect_swapped(gui.table, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.selection, "changed", G_CALLBACK(point_selection_changed), &gui);
    g_signal_connect(gui.graph_selection, "changed", G_CALLBACK(graph_selected), &gui);

    outcome = gwy_dialog_run(dialog);

    g_object_unref(gui.gmodel);
    g_object_unref(gui.data);

    return outcome;
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;
    GwyLawn *lawn = args->lawn;
    gdouble min, max, sel[2];
    gint start, end, i, xres, yres, col, row, curve, ndata;
    const gdouble *cdata;
    const gint *segments;

    if (id < 0 || !(id == PARAM_RANGE_FROM || id == PARAM_RANGE_TO)) {
       extract_one_curve(gui);
    }

    if (id < 0 || id == PARAM_RANGE_FROM || id == PARAM_RANGE_TO) {
        sel[0] = gwy_params_get_double(params, PARAM_RANGE_FROM)*args->vf->magnitude;
        sel[1] = gwy_params_get_double(params, PARAM_RANGE_TO)*args->vf->magnitude;
        //printf("sel: %g %g\n", sel[0], sel[1]);
        gwy_selection_set_data(gui->graph_selection, 1, sel);
    }

    if (id < 0 || id == PARAM_ABSCISSA || id == PARAM_SEGMENT || id == PARAM_ENABLE_SEGMENT) {
        gboolean segment_enabled = args->nsegments ? gwy_params_get_boolean(params, PARAM_ENABLE_SEGMENT) : FALSE;
        gint segment = segment_enabled ? gwy_params_get_int(params, PARAM_SEGMENT) : -1;
        curve = gwy_params_get_int(params, PARAM_ABSCISSA);
        args->vf = gwy_lawn_get_value_format_curve(args->lawn, curve,
                                                   GWY_SI_UNIT_FORMAT_VFMARKUP,
                                                   args->vf);
        min = G_MAXDOUBLE;
        max = -G_MAXDOUBLE;

        xres = gwy_lawn_get_xres(lawn);
        yres = gwy_lawn_get_yres(lawn);
        for (row = 0; row < yres; row++) {
            for (col = 0; col < xres; col++) {
                cdata = gwy_lawn_get_curve_data_const(lawn, col, row, curve, &ndata);
                if (segment >= 0) {
                    segments = gwy_lawn_get_segments(lawn, col, row, NULL);
                    start = segments[2*segment];
                    end = segments[2*segment + 1];
                } else {
                    start = 0;
                    end = ndata;
                }
                for (i = start; i < end; i++) {
                    max = fmax(max, cdata[i]);
                    min = fmin(min, cdata[i]);
                }
            }
        }
        if (min == G_MAXDOUBLE && max == -G_MAXDOUBLE) {
            min = 0;
            max = 0;
        }

        gwy_param_table_slider_restrict_range(gui->table, PARAM_RANGE_FROM,
                                              min/args->vf->magnitude, max/args->vf->magnitude);
        gwy_param_table_slider_restrict_range(gui->table, PARAM_RANGE_TO,
                                              min/args->vf->magnitude, max/args->vf->magnitude);

        gwy_param_table_set_unitstr(gui->table, PARAM_RANGE_FROM, args->vf->units);
        gwy_param_table_set_unitstr(gui->table, PARAM_RANGE_TO, args->vf->units);
    }

    gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
set_selection(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    gint col = gwy_params_get_int(args->params, PARAM_XPOS);
    gint row = gwy_params_get_int(args->params, PARAM_YPOS);
    gdouble xy[2];

    xy[0] = (col + 0.5)*gwy_lawn_get_dx(args->lawn);
    xy[1] = (row + 0.5)*gwy_lawn_get_dy(args->lawn);
    gwy_selection_set_object(gui->selection, 0, xy);
}

static void
point_selection_changed(ModuleGUI *gui, gint id, GwySelection *selection)
{
    ModuleArgs *args = gui->args;
    GwyLawn *lawn = args->lawn;
    gint i, xres = gwy_lawn_get_xres(lawn), yres = gwy_lawn_get_yres(lawn);
    gdouble xy[2];

    gwy_selection_get_object(selection, id, xy);
    i = GWY_ROUND(floor(xy[0]/gwy_lawn_get_dx(lawn)));
    gwy_params_set_int(args->params, PARAM_XPOS, CLAMP(i, 0, xres-1));
    i = GWY_ROUND(floor(xy[1]/gwy_lawn_get_dy(lawn)));
    gwy_params_set_int(args->params, PARAM_YPOS, CLAMP(i, 0, yres-1));

    gwy_param_table_param_changed(gui->table, PARAM_XPOS);
    gwy_param_table_param_changed(gui->table, PARAM_YPOS);
}

static void
graph_selected(GwySelection* selection, gint i, ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    gdouble range[2];
    gdouble xfrom, xto;
    gboolean have_range = TRUE;

    g_return_if_fail(i <= 0);

    if (gwy_selection_get_data(selection, NULL) <= 0)
        have_range = FALSE;
    else {
        gwy_selection_get_object(selection, 0, range);
        if (range[0] == range[1])
            have_range = FALSE;
    }
    if (have_range) {
        xfrom = MIN(range[0], range[1]);
        xto = MAX(range[0], range[1]);

        gwy_param_table_set_double(gui->table, PARAM_RANGE_FROM, xfrom/args->vf->magnitude);
        gwy_param_table_set_double(gui->table, PARAM_RANGE_TO, xto/args->vf->magnitude);
    }
}

static gboolean
point_to_be_removed(const gdouble *cdx, gint n, gdouble from, gdouble to, gint segment, const gint *cs, gint remove)
{
    if (remove == REMOVE_MARKED) { //remove only marked points
        if (segment >= 0) {        //if we have segments, do not remove points from other segments
            if ((n < cs[2*segment]) || (n >= cs[2*segment + 1]))
                return FALSE;
        }
        if (cdx[n] >= from && cdx[n] < to) //if it is within the range, remove it
            return TRUE;
    }
    else if (remove == REMOVE_UNMARKED) { //remove unmarked points
        if (segment >= 0) {      //if we have segment, do not remove points from other segments
            if ((n < cs[2*segment]) || (n >= cs[2*segment + 1]))
                return FALSE;
        }
        if (cdx[n] < from || cdx[n] >= to) //if it is outside of the range, remove it
            return TRUE;
    }

    return FALSE;
}

static void
execute(ModuleArgs *args)
{
    GwyLawn *result, *lawn = args->lawn;
    GwyParams *params = args->params;
    gint abscissa = gwy_params_get_int(params, PARAM_ABSCISSA);
    gdouble from = gwy_params_get_double(params, PARAM_RANGE_FROM)*args->vf->magnitude;
    gdouble to = gwy_params_get_double(params, PARAM_RANGE_TO)*args->vf->magnitude;
    gboolean segment_enabled = args->nsegments ? gwy_params_get_boolean(params, PARAM_ENABLE_SEGMENT) : FALSE;
    gint segment = segment_enabled ? gwy_params_get_int(params, PARAM_SEGMENT) : -1;
    gint remove = gwy_params_get_enum(params, PARAM_REMOVE);
    gint xres = gwy_lawn_get_xres(lawn), yres = gwy_lawn_get_yres(lawn);
    gint ncurves = gwy_lawn_get_n_curves(lawn);
    const gdouble *cdx, *cdy;
    gdouble *ncd;
    gint ndata, i, k, n, col, row;
    gint nc, nn, nsegments, *segments, *ncs, *nsskip;
    const gint *cs;
    
    result = gwy_lawn_new(xres, yres, gwy_lawn_get_xreal(lawn), gwy_lawn_get_yreal(lawn), ncurves, 0);
    gwy_lawn_set_xoffset(result, gwy_lawn_get_xoffset(lawn));
    gwy_lawn_set_yoffset(result, gwy_lawn_get_yoffset(lawn));
    gwy_si_unit_assign(gwy_lawn_get_si_unit_xy(result), gwy_lawn_get_si_unit_xy(lawn));

    for (n = 0; n < ncurves; n++) {
        gwy_si_unit_assign(gwy_lawn_get_si_unit_curve(result, n), gwy_lawn_get_si_unit_curve(lawn, n));
        gwy_lawn_set_curve_label(result, n, gwy_lawn_get_curve_label(lawn, n));
    }

    nsegments = gwy_lawn_get_n_segments(lawn);
    segments = g_new0(gint, 2*nsegments*xres*yres); //full segmentation
    ncs = g_new0(gint, 2*nsegments);                //new curve segments
    nsskip = g_new(gint, 2*nsegments);              //array to count skipped values before the segment entry

    gwy_lawn_set_segments(result, nsegments, segments);
    for (n = 0; n < nsegments; n++)
       gwy_lawn_set_segment_label(result, n, gwy_lawn_get_segment_label(lawn, n));

    for (k = 0; k < xres*yres; k++) {
        col = k % xres;
        row = k/xres;

        cs = gwy_lawn_get_segments(lawn, col, row, NULL);
        cdx = gwy_lawn_get_curve_data_const(lawn, col, row, abscissa, &ndata);
        nc = 0;

        for (i = 0; i < (2*nsegments); i++)       //zero segment skipping information
            nsskip[i] = 0;

        //printf("seg0 %d %d   seg%d  %d %d\n", cs[0], cs[1], segment, cs[2*segment], cs[2*segment + 1]);

        for (n = 0; n < ndata; n++) {
            if (!point_to_be_removed(cdx, n, from, to, segment, cs, remove)) {
                nc++;                               //number of new curve data points
            } else {                                //add information about skipping for segments
                for (i = 0; i < (2*nsegments); i++) {
                    if (n < cs[i])                 //we have skipped something before this segment entry
                        nsskip[i]++;
                }
            }
        }

        //create new curve data
        ncd = g_new(gdouble, nc*ncurves);
        nn = 0;
        for (n = 0; n < ndata; n++) {
            if (!point_to_be_removed(cdx, n, from, to, segment, cs, remove)) {
                for (i = 0; i < ncurves; i++) {
                   cdy = gwy_lawn_get_curve_data_const(lawn, col, row, i, NULL);
                   ncd[i*nc + nn] = cdy[n];
                }
                nn++;
            }
        }

        //shift the segment entries if we have skipped data before them
        for (i = 0; i < (2*nsegments); i++) {
            ncs[i] = cs[i] - nsskip[i];
        //    printf("s %d  cs %d ncs %d   skip %d\n", i, cs[i], ncs[i], nsskip[i]);
        }

        gwy_lawn_set_curves(result, col, row, nc, ncd, ncs); //length, curvesdata, segments 
    }
    gwy_lawn_data_changed(result);

    args->result = result;

    g_free(nsskip);
    g_free(ncs);
}

static void
extract_one_curve(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;
    GwyLawn *lawn = args->lawn;
    gint abscissa = gwy_params_get_int(params, PARAM_ABSCISSA);
    gint ordinate = gwy_params_get_int(params, PARAM_ORDINATE);
    gint col = gwy_params_get_int(params, PARAM_XPOS);
    gint row = gwy_params_get_int(params, PARAM_YPOS);
    GwyGraphCurveModel *gcmodel = gwy_graph_model_get_curve(gui->gmodel, 0);
    gboolean segment_enabled = args->nsegments ? gwy_params_get_boolean(params, PARAM_ENABLE_SEGMENT) : FALSE;
    gint segment = segment_enabled ? gwy_params_get_int(params, PARAM_SEGMENT) : -1;
    gint start, end, ndata;
    const gdouble *xdata, *ydata;
    const gint *segments;

    ydata = gwy_lawn_get_curve_data_const(lawn, col, row, ordinate, &ndata);
    xdata = gwy_lawn_get_curve_data_const(lawn, col, row, abscissa, NULL);

    if (segment >= 0) {
        segments = gwy_lawn_get_segments(lawn, col, row, NULL);
        start = segments[2*segment];
        end = segments[2*segment + 1];
        xdata += start;
        ydata += start;
        ndata = end - start;
    }

    gwy_graph_curve_model_set_data(gcmodel, xdata, ydata, ndata);

    update_graph_model_props(gui->gmodel, args);
}

static void
update_graph_model_props(GwyGraphModel *gmodel, ModuleArgs *args)
{
    GwyLawn *lawn = args->lawn;
    GwyParams *params = args->params;
    gint abscissa = gwy_params_get_int(params, PARAM_ABSCISSA);
    gint ordinate = gwy_params_get_int(params, PARAM_ORDINATE);
    GwySIUnit *xunit, *yunit;
    const gchar *xlabel, *ylabel;

    xunit = gwy_lawn_get_si_unit_curve(lawn, abscissa);
    xlabel = gwy_lawn_get_curve_label(lawn, abscissa);
    yunit = gwy_lawn_get_si_unit_curve(lawn, ordinate);
    ylabel = gwy_lawn_get_curve_label(lawn, ordinate);

    g_object_set(gmodel,
                 "si-unit-x", xunit,
                 "si-unit-y", yunit,
                 "axis-label-bottom", xlabel ? xlabel : _("Untitled"),
                 "axis-label-left", ylabel ? ylabel : _("Untitled"),
                 NULL);
}

static void
sanitise_one_param(GwyParams *params, gint id, gint min, gint max, gint defval)
{
    gint v;

    v = gwy_params_get_int(params, id);
    if (v >= min && v <= max) {
        gwy_debug("param #%d is %d, i.e. within range [%d..%d]", id, v, min, max);
        return;
    }
    gwy_debug("param #%d is %d, setting it to the default %d", id, v, defval);
    gwy_params_set_int(params, id, defval);
}

static void
sanitise_params(ModuleArgs *args)
{
    GwyParams *params = args->params;
    GwyLawn *lawn = args->lawn;

    sanitise_one_param(params, PARAM_XPOS, 0, gwy_lawn_get_xres(lawn)-1, gwy_lawn_get_xres(lawn)/2);
    sanitise_one_param(params, PARAM_YPOS, 0, gwy_lawn_get_yres(lawn)-1, gwy_lawn_get_yres(lawn)/2);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
