/*
 *  $Id: cmap_peaks.c 29036 2025-12-22 19:19:15Z yeti-dn $
 *  Copyright (C) 2021 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libprocess/lawn.h>
#include <libprocess/correct.h>
#include <libprocess/stats.h>
#include <libprocess/linestats.h>
#include <libprocess/peaks.h>
#include <libprocess/gwyprocesstypes.h>
#include <libgwydgets/gwydataview.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-cmap.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "libgwyddion/gwyomp.h"

#define RUN_MODES (GWY_RUN_INTERACTIVE)

enum {
    PREVIEW_SIZE = 360,
    MAX_PEAKS = 1000,
};

enum {
    PARAM_QUANTITY,
    PARAM_WIDTH,
    PARAM_THRESHOLD,
    PARAM_INDIVIDUAL,
    PARAM_DATA_TYPE,
    PARAM_ENABLE_ABSCISSA,
    PARAM_ABSCISSA,
    PARAM_ORDINATE,
    PARAM_SEGMENT,
    PARAM_ENABLE_SEGMENT,
    PARAM_XPOS,
    PARAM_YPOS,
    PARAM_UPDATE,
    LABEL_VALUE,
    LABEL_INTERPOLATED,
};

typedef enum {
    DATA_RAW   = 0,
    DATA_DER   = 1,
    DATA_NDER  = 2,
    NDATAS
} PeaksData;

typedef enum {
    QUANTITY_HEIGHT     = 0,
    QUANTITY_AREA       = 1,
    QUANTITY_PROMINENCE = 2,
    NQUANTITIES
} PeaksQuantity;


typedef struct {
    GwyParams *params;
    GwyLawn *lawn;
    GwyDataField *result;
    GwyDataField *mask;
    GwyDataField *thfield;
    /* Cached input data properties. */
    gint nsegments;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table_quantity;
    GwyParamTable *table_options;
    GwyContainer *data;
    GwySelection *image_selection;
    GwySelection *graph_selection;
    GwyGraphModel *gmodel;
} ModuleGUI;

static gboolean                    module_register        (void);
static GwyParamDef*                define_module_params   (void);
static void                        cmap_peaks             (GwyContainer *data,
                                                           GwyRunType runtype);
static void                        execute                (ModuleArgs *args,
                                                           gboolean normalize_only);
static GwyDialogOutcome            run_gui                (ModuleArgs *args,
                                                           GwyContainer *data,
                                                           gint id);
static void                        param_changed          (ModuleGUI *gui,
                                                           gint id);
static void                        preview                (gpointer user_data);
static void                        set_image_selection    (ModuleGUI *gui);
static void                        point_selection_changed(ModuleGUI *gui,
                                                           gint id,
                                                           GwySelection *selection);
static void                        update_graph_curve     (ModuleGUI *gui);
static void                        update_current_value   (ModuleGUI *gui,
                                                           gdouble v);
static gint                        extract_data_lines     (GwyLawn *lawn,
                                                           GwyDataLine *atarget,
                                                           GwyDataLine *otarget,
                                                           gint col,
                                                           gint row,
                                                           gint abscissa,
                                                           gint ordinate,
                                                           gint segment);
static void                        dline_derive           (GwyDataLine *adline,
                                                           GwyDataLine *odline,
                                                           gboolean have_abscissa,
                                                           gint width,
                                                           gboolean invert);
static void                        sanitise_params        (ModuleArgs *args);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Calculate number of peaks in curve map."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.0",
    "Petr Klapetek & David Nečas (Yeti)",
    "2025",
};

GWY_MODULE_QUERY2(module_info, cmap_peaks)

static gboolean
module_register(void)
{
    gwy_curve_map_func_register("cmap_peaks",
                                (GwyCurveMapFunc)&cmap_peaks,
                                N_("/Count _Peaks..."),
                                NULL,
                                RUN_MODES,
                                GWY_MENU_FLAG_CURVE_MAP,
                                N_("Count peaks"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum data_types[] = {
        { N_("_Raw data"),            DATA_RAW,   },
        { N_("_Gradient"),            DATA_DER,   },
        { N_("_Inverted gradient"),   DATA_NDER,  },
    };
    /* XXX: The underscores are inside a combo. */
    static const GwyEnum quantities[] = {
        { N_("_Height"),         QUANTITY_HEIGHT,     },
        { N_("_Area"),           QUANTITY_AREA,       },
        { N_("_Prominence"),     QUANTITY_PROMINENCE, },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_curve_map_func_current());
    gwy_param_def_add_gwyenum(paramdef, PARAM_QUANTITY, "quantity", _("_Threshold by"),
                              quantities, G_N_ELEMENTS(quantities), DATA_RAW);
    gwy_param_def_add_gwyenum(paramdef, PARAM_DATA_TYPE, "data_type", _("Data"),
                              data_types, G_N_ELEMENTS(data_types), QUANTITY_HEIGHT);
    gwy_param_def_add_percentage(paramdef, PARAM_THRESHOLD, "threshold", _("_Above"), 0.5);
    gwy_param_def_add_boolean(paramdef, PARAM_INDIVIDUAL, "individual", _("Apply _individually"), FALSE);
    gwy_param_def_add_lawn_curve(paramdef, PARAM_ABSCISSA, "abscissa", _("Abscissa"));
    gwy_param_def_add_lawn_curve(paramdef, PARAM_ORDINATE, "ordinate", _("Ordinate"));
    gwy_param_def_add_boolean(paramdef, PARAM_ENABLE_ABSCISSA, "enable_abscissa", NULL, FALSE);
    gwy_param_def_add_lawn_segment(paramdef, PARAM_SEGMENT, "segment", NULL);
    gwy_param_def_add_boolean(paramdef, PARAM_ENABLE_SEGMENT, "enable_segment", NULL, FALSE);
    gwy_param_def_add_int(paramdef, PARAM_WIDTH, "width", _("Window size"), 1, 100, 1);
    gwy_param_def_add_int(paramdef, PARAM_XPOS, "xpos", NULL, -1, G_MAXINT, -1);
    gwy_param_def_add_int(paramdef, PARAM_YPOS, "ypos", NULL, -1, G_MAXINT, -1);
    gwy_param_def_add_instant_updates(paramdef, PARAM_UPDATE, "update", NULL, TRUE);
    return paramdef;
}

static void
cmap_peaks(GwyContainer *data, GwyRunType runtype)
{
    ModuleArgs args;
    GwyLawn *lawn = NULL;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    gint oldid, newid;

    g_return_if_fail(runtype & RUN_MODES);
    g_return_if_fail(g_type_from_name("GwyLayerPoint"));

    gwy_app_data_browser_get_current(GWY_APP_LAWN, &lawn,
                                     GWY_APP_LAWN_ID, &oldid,
                                     0);
    g_return_if_fail(GWY_IS_LAWN(lawn));
    args.lawn = lawn;
    args.nsegments = gwy_lawn_get_n_segments(lawn);
    args.params = gwy_params_new_from_settings(define_module_params());
    args.result = gwy_data_field_new(gwy_lawn_get_xres(lawn), gwy_lawn_get_yres(lawn),
                                     gwy_lawn_get_xreal(lawn), gwy_lawn_get_yreal(lawn), TRUE);
    gwy_data_field_set_xoffset(args.result, gwy_lawn_get_xoffset(lawn));
    gwy_data_field_set_yoffset(args.result, gwy_lawn_get_yoffset(lawn));
    gwy_si_unit_assign(gwy_data_field_get_si_unit_xy(args.result), gwy_lawn_get_si_unit_xy(lawn));
    args.mask = gwy_data_field_new_alike(args.result, TRUE);
    args.thfield = gwy_data_field_new_alike(args.result, TRUE);
    sanitise_params(&args);

    if (runtype == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, oldid);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (outcome != GWY_DIALOG_HAVE_RESULT)
        execute(&args, FALSE);

    newid = gwy_app_data_browser_add_data_field(args.result, data, TRUE);
    gwy_app_set_data_field_title(data, newid, _("Number of peaks"));
    if (gwy_data_field_get_max(args.mask) > 0.0)
        gwy_container_set_object(data, gwy_app_get_mask_key_for_id(newid), args.mask);
    gwy_app_channel_log_add(data, -1, newid, "cmap::cmap_peaks", NULL);

end:
    g_object_unref(args.result);
    g_object_unref(args.mask);
    g_object_unref(args.thfield);
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyContainer *data, gint id)
{
    GtkWidget *hbox, *graph, *dataview, *align, *area;
    GwyParamTable *table;
    GwyDialog *dialog;
    ModuleGUI gui;
    GwyDataField *field = args->result;
    GwyGraphCurveModel *gcmodel;
    GwyDialogOutcome outcome;
    const guchar *gradient;

    gwy_clear(&gui, 1);
    gui.args = args;
    gui.data = gwy_container_new();
    gwy_container_set_object(gui.data, gwy_app_get_data_key_for_id(0), field);
    if (gwy_container_gis_string(data, gwy_app_get_lawn_palette_key_for_id(id), &gradient))
        gwy_container_set_const_string(gui.data, gwy_app_get_data_palette_key_for_id(0), gradient);

    gui.dialog = gwy_dialog_new(_("Count Peaks"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_UPDATE, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    hbox = gwy_hbox_new(0);
    gwy_dialog_add_content(GWY_DIALOG(gui.dialog), hbox, TRUE, TRUE, 0);

    align = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 0);

    dataview = gwy_create_preview(gui.data, 0, PREVIEW_SIZE, FALSE);
    gtk_container_add(GTK_CONTAINER(align), dataview);
    gui.image_selection = gwy_create_preview_vector_layer(GWY_DATA_VIEW(dataview), 0, "Point", 1, TRUE);

    gui.gmodel = gwy_graph_model_new();
    g_object_set(gui.gmodel,
                 "label-visible", FALSE,
                 "axis-label-bottom", _("sample"),
                 NULL);

    gcmodel = gwy_graph_curve_model_new();
    g_object_set(gcmodel, "mode", GWY_GRAPH_CURVE_LINE, NULL);
    gwy_graph_model_add_curve(gui.gmodel, gcmodel);
    g_object_unref(gcmodel);

    graph = gwy_graph_new(gui.gmodel);
    gwy_graph_enable_user_input(GWY_GRAPH(graph), FALSE);
    gwy_graph_set_status(GWY_GRAPH(graph), GWY_GRAPH_STATUS_XLINES);
    area = gwy_graph_get_area(GWY_GRAPH(graph));
    gwy_graph_area_set_selection_editable(GWY_GRAPH_AREA(area), FALSE);
    gui.graph_selection = gwy_graph_area_get_selection(GWY_GRAPH_AREA(area), GWY_GRAPH_STATUS_XLINES);

    gtk_widget_set_size_request(graph, PREVIEW_SIZE, PREVIEW_SIZE);
    gtk_box_pack_start(GTK_BOX(hbox), graph, TRUE, TRUE, 0);

    hbox = gwy_hbox_new(20);
    gwy_dialog_add_content(GWY_DIALOG(gui.dialog), hbox, TRUE, TRUE, 4);

    table = gui.table_quantity = gwy_param_table_new(args->params);
    gwy_param_table_append_lawn_curve(table, PARAM_ABSCISSA, args->lawn);
    gwy_param_table_add_enabler(table, PARAM_ENABLE_ABSCISSA, PARAM_ABSCISSA);
    gwy_param_table_append_lawn_curve(table, PARAM_ORDINATE, args->lawn);
    if (args->nsegments) {
        gwy_param_table_append_lawn_segment(table, PARAM_SEGMENT, args->lawn);
        gwy_param_table_add_enabler(table, PARAM_ENABLE_SEGMENT, PARAM_SEGMENT);
    }
    gwy_param_table_append_radio(table, PARAM_DATA_TYPE);
    gwy_param_table_append_slider(table, PARAM_WIDTH);
    gwy_param_table_append_info(table, LABEL_VALUE, _("Value"));
    gwy_param_table_append_separator(table);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    table = gui.table_options = gwy_param_table_new(args->params);
    gwy_param_table_append_combo(table, PARAM_QUANTITY);
    gwy_param_table_append_slider(table, PARAM_THRESHOLD);
    gwy_param_table_append_checkbox(table, PARAM_INDIVIDUAL);
    gwy_param_table_append_separator(table);
    gwy_param_table_append_checkbox(table, PARAM_UPDATE);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    set_image_selection(&gui);
    g_signal_connect_swapped(gui.table_quantity, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.table_options, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.image_selection, "changed", G_CALLBACK(point_selection_changed), &gui);
    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_IMMEDIATE, preview, &gui, NULL);

    outcome = gwy_dialog_run(dialog);

    g_object_unref(gui.data);
    g_object_unref(gui.gmodel);
    return outcome;
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    GwyParams *params = gui->args->params;

    if (id != PARAM_UPDATE && id != PARAM_XPOS && id != PARAM_YPOS)
        gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
   
    if (id < 0 || ((id != PARAM_XPOS && id != PARAM_YPOS && id != PARAM_THRESHOLD)
                   && !gwy_params_get_boolean(params, PARAM_UPDATE)))
        execute(gui->args, TRUE);

    update_graph_curve(gui);

    if (id < 0 || id == PARAM_DATA_TYPE) 
        gwy_param_table_set_sensitive(gui->table_quantity,
                                      PARAM_WIDTH, 
                                      gwy_params_get_enum(gui->args->params, PARAM_DATA_TYPE) != DATA_RAW);
}

static void
set_image_selection(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    gint col = gwy_params_get_int(args->params, PARAM_XPOS);
    gint row = gwy_params_get_int(args->params, PARAM_YPOS);
    gdouble xy[2];

    xy[0] = (col + 0.5)*gwy_lawn_get_dx(args->lawn);
    xy[1] = (row + 0.5)*gwy_lawn_get_dy(args->lawn);
    gwy_selection_set_object(gui->image_selection, 0, xy);
}

static void
point_selection_changed(ModuleGUI *gui, G_GNUC_UNUSED gint id, GwySelection *selection)
{
    ModuleArgs *args = gui->args;
    GwyLawn *lawn = args->lawn;
    gint i, xres = gwy_lawn_get_xres(lawn), yres = gwy_lawn_get_yres(lawn);
    gdouble xy[2];

    if (!gwy_selection_get_object(selection, 0, xy)) {
        gwy_params_set_int(args->params, PARAM_XPOS, xres/2);
        gwy_params_set_int(args->params, PARAM_YPOS, yres/2);
    }
    else {
        i = GWY_ROUND(floor(xy[0]/gwy_lawn_get_dx(lawn)));
        gwy_params_set_int(args->params, PARAM_XPOS, CLAMP(i, 0, xres-1));
        i = GWY_ROUND(ceil(xy[1]/gwy_lawn_get_dy(lawn)));
        gwy_params_set_int(args->params, PARAM_YPOS, CLAMP(i, 0, yres-1));
    }
    gwy_param_table_param_changed(gui->table_quantity, PARAM_XPOS);
    gwy_param_table_param_changed(gui->table_quantity, PARAM_YPOS);
    update_graph_curve(gui);
}

static void
update_graph_curve(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;
    GwyLawn *lawn = args->lawn;
    GwyDataLine *adline = gwy_data_line_new(1, 1.0, FALSE);
    GwyDataLine *odline = gwy_data_line_new(1, 1.0, FALSE);
    gint col = gwy_params_get_int(params, PARAM_XPOS);
    gint row = gwy_params_get_int(params, PARAM_YPOS);
    gint ordinate = gwy_params_get_int(params, PARAM_ORDINATE);
    gdouble threshold = gwy_params_get_double(params, PARAM_THRESHOLD);
    gboolean individual = gwy_params_get_boolean(params, PARAM_INDIVIDUAL);
    gboolean abscissa_enabled = gwy_params_get_boolean(params, PARAM_ENABLE_ABSCISSA);
    gboolean segment_enabled = args->nsegments ? gwy_params_get_boolean(params, PARAM_ENABLE_SEGMENT) : FALSE;
    gint abscissa = abscissa_enabled ? gwy_params_get_int(params, PARAM_ABSCISSA) : -1;
    gint segment = segment_enabled ? gwy_params_get_int(params, PARAM_SEGMENT) : -1;
    gint width = gwy_params_get_int(params, PARAM_WIDTH);
    PeaksQuantity quantity = gwy_params_get_enum(params, PARAM_QUANTITY);
    PeaksData datatype = gwy_params_get_enum(params, PARAM_DATA_TYPE);
    GwyGraphCurveModel *gcmodel = gwy_graph_model_get_curve(gui->gmodel, 0);
    const gchar *label;
    GwyPeaks *analyser = gwy_peaks_new();
    gdouble *vals = g_new(gdouble, MAX_PEAKS);
    gdouble *poss = g_new(gdouble, MAX_PEAKS);
    gint i, n, nv;
    gdouble *sel;
    gdouble *tdata = gwy_data_field_get_data(args->thfield);
    gint xres = gwy_data_field_get_xres(args->thfield);

    if (extract_data_lines(lawn, adline, odline, col, row, abscissa, ordinate, segment)) {
        gwy_graph_curve_model_set_data(gcmodel,
                                       gwy_data_line_get_data(adline),
                                       gwy_data_line_get_data(odline),
                                       gwy_data_line_get_res(odline));
    }
    else
        gwy_graph_curve_model_set_data(gcmodel, NULL, NULL, 0);

    label = gwy_lawn_get_curve_label(lawn, ordinate);
    if (abscissa >= 0) {
        g_object_set(gui->gmodel,
                     "si-unit-x", gwy_lawn_get_si_unit_curve(lawn, abscissa),
                     NULL);
    }
    g_object_set(gui->gmodel,
                 "si-unit-y", gwy_lawn_get_si_unit_curve(lawn, ordinate),
                 "axis-label-left", label ? label : _("Untitled"),
                 NULL);

    if (datatype == DATA_DER)
        dline_derive(adline, odline, abscissa_enabled, width, FALSE);
    else if (datatype == DATA_NDER)
        dline_derive(adline, odline, abscissa_enabled, width, TRUE);

    if (datatype == DATA_DER || datatype == DATA_NDER)
        gwy_peaks_set_background(analyser, GWY_PEAK_BACKGROUND_ZERO);

    n = gwy_peaks_analyze(analyser,
                          gwy_data_line_get_data(adline),
                          gwy_data_line_get_data(odline),
                          gwy_data_line_get_res(odline),
                          MAX_PEAKS);
    nv = 0;

    gwy_peaks_get_quantity(analyser, GWY_PEAK_ABSCISSA, poss);
    if (quantity == QUANTITY_HEIGHT)
        gwy_peaks_get_quantity(analyser, GWY_PEAK_HEIGHT, vals);
    else if (quantity == QUANTITY_AREA)
        gwy_peaks_get_quantity(analyser, GWY_PEAK_AREA, vals);
    else
        gwy_peaks_get_quantity(analyser, GWY_PEAK_PROMINENCE, vals);

    if (individual)
        threshold *= tdata[row*xres + col];
    else
        threshold *= gwy_data_field_get_max(args->thfield);

    for (i = 0; i < n; i++) {
        if (vals[i] >= threshold)
            nv++;
    }

    sel = g_new(gdouble, nv);
    nv = 0;
    for (i = 0; i < n; i++) {
        if (vals[i] >= threshold)
            sel[nv++] = poss[i];
    }

    if (nv > 0) {
        gwy_selection_set_max_objects(gui->graph_selection, nv);
        gwy_selection_set_data(gui->graph_selection, nv, sel);
    } else
        gwy_selection_clear(gui->graph_selection);

    update_current_value(gui, nv);

    g_object_unref(adline);
    g_object_unref(odline);
    gwy_peaks_free(analyser);
    g_free(vals);
    g_free(poss);
    g_free(sel);
}

static void
update_current_value(ModuleGUI *gui, gdouble v)
{
    ModuleArgs *args = gui->args;
    gint col = gwy_params_get_int(args->params, PARAM_XPOS);
    gint row = gwy_params_get_int(args->params, PARAM_YPOS);
    gchar *s;

    if (!gwy_data_field_get_val(args->mask, col, row)) {
        s = g_strdup_printf("%d", (gint)v);
        gwy_param_table_info_set_valuestr(gui->table_quantity, LABEL_VALUE, s);
        g_free(s);
    } else {
        gwy_param_table_info_set_valuestr(gui->table_quantity, LABEL_VALUE, _("No data"));
    }
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    ModuleArgs *args = gui->args;

    execute(args, FALSE);
    gwy_data_field_data_changed(args->result);
    update_graph_curve(gui);
    gwy_dialog_have_result(GWY_DIALOG(gui->dialog));
}

static gdouble                                               
fit_slope(gdouble *ydata, gint res, gint pos, gint size)
{
    gdouble dsize = 2*size + 1;
    gdouble sumxi = 0.0;
    gdouble sumxixi = 0.0;
    gdouble sumsixi = 0.0;                         
    gdouble sumsi = 0.0;
    gint i;

    if (pos < size || (pos + size) >= res )
        return 0;

    for (i = (pos - size); i <= (pos + size); i++) {
        sumsi += ydata[i];
        sumsixi += ydata[i] * i;
        sumxi += i;
        sumxixi += i*i;
    }
    sumsi /= dsize;
    sumsixi /= dsize;
    sumxi /= dsize;
    sumxixi /= dsize;

    return (sumsixi - sumsi*sumxi)/(sumxixi - sumxi*sumxi);
}

static gdouble                                               
fit_slope_abscissa(gdouble *ydata, gdouble *xdata, gint res, gint pos, gint size)
{
    gint dsize = 2*size + 1;
    gdouble sumxi = 0.0;
    gdouble sumxixi = 0.0;
    gdouble sumsixi = 0.0;                         
    gdouble sumsi = 0.0;
    gint i;

    if (pos < size || (pos + size) >= res)
        return 0;

    for (i = (pos - size); i <= (pos + size); i++) {
        sumsi += ydata[i];
        sumsixi += ydata[i] * xdata[i];
        sumxi += xdata[i];
	    sumxixi += xdata[i] * xdata[i];
    }
    sumsi /= dsize;
    sumsixi /= dsize;
    sumxi /= dsize;
    sumxixi /= dsize;

    return (sumsixi - sumsi*sumxi)/(sumxixi - sumxi*sumxi);
}


static void
dline_derive(GwyDataLine *adline, GwyDataLine *odline, gboolean have_abscissa, gint width, gboolean invert)
{
    gint n = gwy_data_line_get_res(odline);
    gint i;
    GwyDataLine *buf = gwy_data_line_new_alike(odline, FALSE);
    gdouble *adata = gwy_data_line_get_data(adline);
    gdouble *bdata = gwy_data_line_get_data(buf);
    gdouble *odata = gwy_data_line_get_data(odline);

    if (n < 2)
        return;

    for (i = 0; i < n; i++)
        bdata[i] = odata[i];

    if (!have_abscissa) {
        for (i = 0; i < n; i++)
	        odata[i] = fit_slope(bdata, n, i, width);
            //odata[i] = gwy_data_line_get_der(buf, i);
    }
    else {
        //odata[0] = (bdata[1] - bdata[0])/(adata[1] - adata[0]);
        //odata[n-1] = (bdata[n-2] - bdata[n-1])/(adata[n-2] - adata[n-1]);
        //for (i = 1; i < (n-1); i++)
        //    odata[i] = (bdata[i+1] - bdata[i-1])/(adata[i+1] - adata[i-1]);
	    for (i = 0; i < n; i++)
	        odata[i] = fit_slope_abscissa(bdata, adata, n, i, width);
    }
    if (invert)
        gwy_data_line_invert(odline, FALSE, TRUE);

    gwy_object_unref(buf);
}
static void
execute(ModuleArgs *args, gboolean normalize_only)
{
    GwyParams *params = args->params;
    PeaksQuantity quantity = gwy_params_get_enum(params, PARAM_QUANTITY);
    PeaksData datatype = gwy_params_get_enum(params, PARAM_DATA_TYPE);
    gint ordinate = gwy_params_get_int(params, PARAM_ORDINATE);
    gboolean abscissa_enabled = gwy_params_get_boolean(params, PARAM_ENABLE_ABSCISSA);
    gdouble threshold = gwy_params_get_double(params, PARAM_THRESHOLD);
    gboolean individual = gwy_params_get_boolean(params, PARAM_INDIVIDUAL);
    gboolean segment_enabled = args->nsegments ? gwy_params_get_boolean(params, PARAM_ENABLE_SEGMENT) : FALSE;
    gint abscissa = abscissa_enabled ? gwy_params_get_int(params, PARAM_ABSCISSA) : -1;
    gint segment = segment_enabled ? gwy_params_get_int(params, PARAM_SEGMENT) : -1;
    gint width = gwy_params_get_int(params, PARAM_WIDTH);
    GwyLawn *lawn = args->lawn;
    GwyDataField *field = args->result, *mask = args->mask;
    GwyDataField *thfield = args->thfield;
    gint xres = gwy_lawn_get_xres(lawn), yres = gwy_lawn_get_yres(lawn);
    gdouble *data, *mdata, *tdata, thmax;

    gwy_data_field_clear(mask);
    data = gwy_data_field_get_data(field);
    mdata = gwy_data_field_get_data(mask);
    tdata = gwy_data_field_get_data(thfield);

#ifdef _OPENMP
#pragma omp parallel if(gwy_threads_are_enabled()) default(none) \
            shared(lawn,xres,yres,abscissa,ordinate,segment,data,mdata,tdata,threshold,quantity,datatype,abscissa_enabled,width)
#endif
    {
        GwyDataLine *adline = gwy_data_line_new(1, 1.0, FALSE);
        GwyDataLine *odline = gwy_data_line_new(1, 1.0, FALSE);
        GwyPeaks *analyser = gwy_peaks_new();
        guint kfrom = gwy_omp_chunk_start(xres*yres);
        guint kto = gwy_omp_chunk_end(xres*yres);
        guint k, i;
        gint n;
        gdouble mq;
        gdouble *vals = g_new(gdouble, MAX_PEAKS);

        for (k = kfrom; k < kto; k++) {
            if (extract_data_lines(lawn, adline, odline, k % xres, k/xres, abscissa, ordinate, segment)) {
                if (datatype == DATA_DER)
                    dline_derive(adline, odline, abscissa_enabled, width, FALSE);
                else if (datatype == DATA_NDER)
                    dline_derive(adline, odline, abscissa_enabled, width, TRUE);

                if (datatype == DATA_DER || datatype == DATA_NDER)
                    gwy_peaks_set_background(analyser, GWY_PEAK_BACKGROUND_ZERO);

                n = gwy_peaks_analyze(analyser,
                                      gwy_data_line_get_data(adline),
                                      gwy_data_line_get_data(odline),
                                      gwy_data_line_get_res(odline),
                                      MAX_PEAKS);
                if (quantity == QUANTITY_HEIGHT)
                    gwy_peaks_get_quantity(analyser, GWY_PEAK_HEIGHT, vals);
                else if (quantity == QUANTITY_AREA)
                    gwy_peaks_get_quantity(analyser, GWY_PEAK_AREA, vals);
                else
                    gwy_peaks_get_quantity(analyser, GWY_PEAK_PROMINENCE, vals);

                mq = -G_MAXDOUBLE;
                for (i = 0; i < n; i++) {
                    if (vals[i] > mq)
                        mq = vals[i];
                }
                if (n > 0)
                    tdata[k] = mq;
                else
                    tdata[k] = 0;
            } else {
                mdata[k] = 1.0;
                data[k] = 0;
                tdata[k] = 0;
            }
        }

        g_object_unref(adline);
        g_object_unref(odline);
        g_free(vals);
        gwy_peaks_free(analyser);
    }
    if (normalize_only)
        return;

    thmax = gwy_data_field_get_max(args->thfield);
 
#ifdef _OPENMP
#pragma omp parallel if(gwy_threads_are_enabled()) default(none) \
            shared(lawn,xres,yres,abscissa,ordinate,segment,data,mdata,tdata,threshold,quantity,individual,thmax,datatype,abscissa_enabled,width)
#endif
    {
        GwyDataLine *adline = gwy_data_line_new(1, 1.0, FALSE);
        GwyDataLine *odline = gwy_data_line_new(1, 1.0, FALSE);
        GwyPeaks *analyser = gwy_peaks_new();
        guint kfrom = 0;//gwy_omp_chunk_start(xres*yres);
        guint kto = xres*yres;//gwy_omp_chunk_end(xres*yres);
        guint k, i;
        gint n, nv;
        gdouble *vals = g_new(gdouble, MAX_PEAKS);
        gdouble thresh;

        for (k = kfrom; k < kto; k++) {
            if (extract_data_lines(lawn, adline, odline, k % xres, k/xres, abscissa, ordinate, segment)) {
                if (datatype == DATA_DER)
                    dline_derive(adline, odline, abscissa_enabled, width, FALSE);
                else if (datatype == DATA_NDER)
                    dline_derive(adline, odline, abscissa_enabled, width, TRUE);

                n = gwy_peaks_analyze(analyser,
                                      gwy_data_line_get_data(adline),
                                      gwy_data_line_get_data(odline),
                                      gwy_data_line_get_res(odline),
                                      MAX_PEAKS);
                if (quantity == QUANTITY_HEIGHT)
                    gwy_peaks_get_quantity(analyser, GWY_PEAK_HEIGHT, vals);
                else if (quantity == QUANTITY_AREA)
                    gwy_peaks_get_quantity(analyser, GWY_PEAK_AREA, vals);
                else
                    gwy_peaks_get_quantity(analyser, GWY_PEAK_PROMINENCE, vals);

                if (individual)
                    thresh = threshold*tdata[k];
                else
                    thresh = threshold*thmax;

                nv = 0;
                for (i = 0; i < n; i++) {
                    if (vals[i] >= thresh)
                        nv++;
                }
                data[k] = nv;
            }
        }

        g_object_unref(adline);
        g_object_unref(odline);
        g_free(vals);
        gwy_peaks_free(analyser);
    }
}

static gint
extract_data_lines(GwyLawn *lawn, GwyDataLine *atarget, GwyDataLine *otarget,
                   gint col, gint row, gint abscissa, gint ordinate, gint segment)
{
    gint pos = 0, len;
    const gdouble *xdata, *ydata;
    const gint *segments;
    gdouble *xldata, *yldata, *samplenodata = NULL;

    ydata = gwy_lawn_get_curve_data_const(lawn, col, row, ordinate, &len);
    if (abscissa >= 0)
        xdata = gwy_lawn_get_curve_data_const(lawn, col, row, abscissa, NULL);
    else
        xdata = samplenodata = gwy_math_linspace(NULL, len, 0, 1);

    if (!len)
        return 0;

    if (segment >= 0) {
        segments = gwy_lawn_get_segments(lawn, col, row, NULL);
        pos = segments[2*segment];
        len = segments[2*segment + 1] - pos;
        if (!len)
            return 0;
    }

    gwy_data_line_resample(atarget, len, GWY_INTERPOLATION_NONE);
    xldata = gwy_data_line_get_data(atarget);
    gwy_assign(xldata, xdata + pos, len);
    gwy_data_line_set_real(atarget, len);

    gwy_data_line_resample(otarget, len, GWY_INTERPOLATION_NONE);
    yldata = gwy_data_line_get_data(otarget);
    gwy_assign(yldata, ydata + pos, len);
    gwy_data_line_set_real(otarget, len);

    g_free(samplenodata);

    return len;
}


static void
sanitise_one_param(GwyParams *params, gint id, gint min, gint max, gint defval)
{
    gint v;

    v = gwy_params_get_int(params, id);
    if (v >= min && v <= max) {
        gwy_debug("param #%d is %d, i.e. within range [%d..%d]", id, v, min, max);
        return;
    }
    gwy_debug("param #%d is %d, setting it to the default %d", id, v, defval);
    gwy_params_set_int(params, id, defval);
}

static void
sanitise_params(ModuleArgs *args)
{
    GwyParams *params = args->params;
    GwyLawn *lawn = args->lawn;

    sanitise_one_param(params, PARAM_XPOS, 0, gwy_lawn_get_xres(lawn)-1, gwy_lawn_get_xres(lawn)/2);
    sanitise_one_param(params, PARAM_YPOS, 0, gwy_lawn_get_yres(lawn)-1, gwy_lawn_get_yres(lawn)/2);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
