/*
 *  $Id: lafmdmap.c 29045 2025-12-27 10:14:01Z klapetek $
 *  Copyright (C) 2025 Petr Klapetek, David Necas (Yeti).
 *  E-mail: klapetek@gwyddion.net, yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
 * [FILE-MAGIC-MISSING]
 * Export only.
 **/

#include "config.h"
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwyutils.h>
#include <libgwymodule/gwymodule-file.h>
#include <libprocess/datafield.h>
#include <libprocess/gwyprocesstypes.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwydgetutils.h>
#include <app/gwyapp.h>

#include "err.h"

#define EXTENSION ".afm"

enum {
    HEADER_USEFUL = 160,
    HEADER_ZEROS = 864,
    HEADER_SIZE = HEADER_USEFUL + HEADER_ZEROS,
};

static gboolean module_register(void);
static gint     detect_file    (const GwyFileDetectInfo *fileinfo,
                                gboolean only_name);
static gboolean export_file    (GwyContainer *data,
                                const gchar *filename,
                                GwyRunType mode,
                                GError **error);
static gboolean export_brick   (GwyBrick *brick,
                                gint32 nparticles,
                                gint32 ndetects,
                                gint32 bicubic,
                                gint32 density,
                                gdouble xps,
                                gdouble yps,
                                gdouble xrs,
                                gdouble yrs,
                                gdouble zr,
                                gdouble hwi,
                                gdouble hwd,
                                FILE *fh);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Exports volume data as LAFM density map."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.0",
    "Petr Klapetek & David Nečas (Yeti)",
    "2025",
};

GWY_MODULE_QUERY2(module_info, lafmdmap)

static gboolean
module_register(void)
{
    gwy_file_func_register("lafmdmap",
                           N_("LAFM density map (.afm)"),
                           (GwyFileDetectFunc)detect_file,
                           NULL,
                           NULL,
                           (GwyFileSaveFunc)export_file);

    return TRUE;
}

static gint
detect_file(const GwyFileDetectInfo *fileinfo,
            G_GNUC_UNUSED gboolean only_name)
{
    return g_str_has_suffix(fileinfo->name_lowercase, EXTENSION) ? 10 : 0;
}

static gdouble
parse_double_with_unit(const gchar *str)
{
    gchar *end;
    gint power10;
    gdouble value = g_ascii_strtod(str, &end);
    GwySIUnit *siunit = gwy_si_unit_new_parse(end, &power10);
    value *= pow10(power10);
    g_object_unref(siunit);

    return value;
}

static gboolean
export_file(GwyContainer *data,
            const gchar *filename,
            G_GNUC_UNUSED GwyRunType mode,
            GError **error)
{
    GwyContainer *meta;
    GwyBrick *brick;
    gint id;
    gint nparticles, ndetects, bicubic, density;
    gdouble xps, yps, xrs, yrs, zr, hwi, hwd;
    const guchar *title = NULL;
    const guchar *str;
    gchar *end;
    FILE *fh = NULL;
    gboolean metaok = TRUE, ok = FALSE;

    gwy_app_data_browser_get_current(GWY_APP_BRICK, &brick,
                                     GWY_APP_BRICK_ID, &id,
                                     0);

    if (!brick) {
        err_NO_CHANNEL_EXPORT(error);
        return FALSE;
    }

    meta = gwy_container_get_object(data, gwy_app_get_brick_meta_key_for_id(id));

    nparticles = ndetects = bicubic = density = 0;
    xps = yps = xrs = yrs = zr = hwi = hwd = 0;

    if (gwy_container_gis_string_by_name(meta, "X particle pixel size", &str))
        xps = parse_double_with_unit(str);
    else
        metaok = FALSE;

    if (gwy_container_gis_string_by_name(meta, "Y particle pixel size", &str))
        yps = parse_double_with_unit(str);
    else
        metaok = FALSE;

    if (gwy_container_gis_string_by_name(meta, "X particle real size", &str))
        xrs = parse_double_with_unit(str);
    else
        metaok = FALSE;

    if (gwy_container_gis_string_by_name(meta, "X particle real size", &str))
        yrs = parse_double_with_unit(str);
    else
        metaok = FALSE;

    if (gwy_container_gis_string_by_name(meta, "Z particle resolution", &str))
        zr = parse_double_with_unit(str);
    else
        metaok = FALSE;

    if (gwy_container_gis_string_by_name(meta, "Half-bit WL of image stack", &str))
        hwi = parse_double_with_unit(str);
    else
        metaok = FALSE;

    if (gwy_container_gis_string_by_name(meta, "Half-bit WL of density volume", &str))
        hwd = parse_double_with_unit(str);
    else
        metaok = FALSE;

    if (gwy_container_gis_string_by_name(meta, "Detections", &str))
        ndetects = (gint)g_ascii_strtod(str, &end);
    else
        metaok = FALSE;

    if (gwy_container_gis_string_by_name(meta, "Number of particles", &str))
        nparticles = (gint)g_ascii_strtod(str, &end);
    else
        metaok = FALSE;

    if (gwy_container_gis_string_by_name(meta, "Bicubic expansion scale", &str))
        bicubic = (gint)g_ascii_strtod(str, &end);
    else
        metaok = FALSE;

    if (gwy_container_gis_string_by_name(meta, "Density function", &str))
        density = (gint)g_ascii_strtod(str, &end);
    else
        metaok = FALSE;

    gwy_container_gis_string(data, gwy_app_get_brick_key_for_id(id), &title);
    if (!title)
        title = _("Untitled");

    if (!(fh = gwy_fopen(filename, "w"))) {
        err_OPEN_WRITE(error);
        goto fail;
    }

    if (!export_brick(brick, nparticles, ndetects, bicubic, density,
                      xps, yps, xrs, yrs, zr, hwi, hwd,
                      fh)) {
       err_WRITE(error);
       goto fail;
    }
    ok = TRUE;

fail:
    if (fh)
        fclose(fh);
    if (!ok)
        g_unlink(filename);

    return ok;
}

static inline void
append_float(guchar **target, const gfloat v)
{
    union { guchar pp[4]; float d; } u;
    guchar *p = *target;
    u.d = v;
#if (G_BYTE_ORDER == G_BIG_ENDIAN)
    p[0] = u.pp[3];
    p[1] = u.pp[2];
    p[2] = u.pp[1];
    p[3] = u.pp[0];
#else
    p[0] = u.pp[0];
    p[1] = u.pp[1];
    p[2] = u.pp[2];
    p[3] = u.pp[3];
#endif
    *target += sizeof(gfloat);
}

static inline void
append_gint32(guchar **target, gint32 v)
{
    v = GINT32_TO_LE(v);
    memcpy(*target, &v, sizeof(gint32));
    *target += sizeof(gint32);
}

// export to modified MRC2014 format, synthesis of these two tables:
// https://www.nature.com/articles/s41467-025-56760-7/tables/1
// https://www.ccpem.ac.uk/mrc-format/mrc2014/#8
static gboolean
export_brick(GwyBrick *brick,
             gint32 nparticles, gint32 ndetects, gint32 bicubic, gint32 density,
             gdouble xps, gdouble yps, gdouble xrs, gdouble yrs,
             gdouble zr, gdouble hwi, gdouble hwd,
             FILE *fh)
{
    gint i;
    gdouble *data = gwy_brick_get_data(brick);
    gint32 xres = gwy_brick_get_xres(brick), yres = gwy_brick_get_yres(brick), zres = gwy_brick_get_zres(brick);
    gdouble xreal = gwy_brick_get_xreal(brick), yreal = gwy_brick_get_yreal(brick), zreal = gwy_brick_get_zreal(brick);
    gdouble xspeed = 0;  //we have no metadata for this yet
    gdouble yspeed = 0;  //we have no metadata for this yet
    gint32 rdcode = 2;
    gint32 xsampling = 1; //?
    gint32 ysampling = 1; //?
    gint32 zsampling = 1; //?
    gint32 ispg = 0;
    gint32 sym = 0;
    gint32 nxstart = 0; //?
    gint32 nystart = 0;
    gint32 nzstart = 0;
    gint32 version = 20140;
    const gchar exttyp[] = "AFM1";
    gint32 zero = 0;
    gint n = xres*yres*zres;
    guchar *p, *buffer = g_new0(guchar, MAX(HEADER_SIZE, n*sizeof(gfloat)));
    gboolean ok = FALSE;

    p = buffer;
    append_gint32(&p, xres);          //1
    append_gint32(&p, yres);       //5
    append_gint32(&p, zres);       //9
    append_gint32(&p, rdcode);     //13
    append_gint32(&p, nxstart);    //17
    append_gint32(&p, nystart);    //21
    append_gint32(&p, nzstart);    //25
    append_gint32(&p, xsampling);  //29
    append_gint32(&p, ysampling);  //33
    append_gint32(&p, zsampling);  //37
    append_float(&p, xreal);      //41
    append_float(&p, yreal);      //45
    append_float(&p, zreal);      //49
    append_gint32(&p, zero);       //53
    append_gint32(&p, zero);       //57
    append_gint32(&p, zero);       //51
    append_gint32(&p, zero);       //65
    append_gint32(&p, zero);       //69
    append_gint32(&p, zero);       //73
    append_gint32(&p, zero);       //77
    append_gint32(&p, zero);       //71
    append_gint32(&p, zero);       //85
    append_gint32(&p, ispg);       //89
    append_gint32(&p, zero);       //93
    append_gint32(&p, nparticles); //97
    append_gint32(&p, sym);        //101
    memcpy(p, exttyp, strlen(exttyp)); // 105
    p += strlen(exttyp);
    append_gint32(&p, version);    //109
    append_float(&p, xps);        //113
    append_float(&p, yps);        //117
    append_float(&p, xrs*1e10);        //121
    append_float(&p, yrs*1e10);        //125
    append_float(&p, zr*1e10);         //129
    append_float(&p, xspeed);     //133
    append_float(&p, yspeed);     //137
    append_gint32(&p, bicubic);    //141
    append_gint32(&p, ndetects);   //145
    append_float(&p, hwi*1e10);        //149
    append_float(&p, hwd*1e10);        //153
    append_gint32(&p, density);   //157...160
    // The rest of the header are zeros.
    if (fwrite(buffer, 1, HEADER_SIZE, fh) != HEADER_SIZE)
        goto fail;

    // Data.
    p = buffer;
    for (i = 0; i < n; i++)
        append_float(&p, data[i]);

    if (fwrite(buffer, sizeof(gfloat), n, fh) != n)
        goto fail;

    ok = TRUE;

fail:
    g_free(buffer);

    return ok;
}


/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
