/*
 *  $Id: reorder.c 28622 2025-09-26 13:11:32Z yeti-dn $
 *  Copyright (C) 2025 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <gtk/gtk.h>
#include <libprocess/level.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "preview.h"

#define RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

enum {
    PARAM_METHOD,
};

typedef enum {
    REORDER_MEANDER_LTR,
    REORDER_MEANDER_RTL,
    REORDER_INTERLACE,
    REORDER_DEINTERLACE,
} ReorderMethod;

typedef struct {
    GwyParams *params;
    GwyDataField *field;
    GwyDataField *result;
    /* Cached input image properties. */
    gboolean square_image;
    gboolean power2_image;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
} ModuleGUI;

static gboolean         module_register     (void);
static GwyParamDef*     define_module_params(void);
static void             module_main         (GwyContainer *data,
                                             GwyRunType run);
static void             execute             (ModuleArgs *args);
static GwyDialogOutcome run_gui             (ModuleArgs *args);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Reorders image data in different ways."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2008",
};

GWY_MODULE_QUERY2(module_info, reorder)

static gboolean
module_register(void)
{
    gwy_process_func_register("reorder",
                              (GwyProcessFunc)&module_main,
                              N_("/_Distortion/_Reorder..."),
                              NULL,
                              RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Reorder image data"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum methods[] = {
        { "Meander (left to right)", REORDER_MEANDER_LTR, },
        { "Meander (right to left)", REORDER_MEANDER_RTL, },
        { "Interlace",               REORDER_INTERLACE,   },
        { "Deinterlace",             REORDER_DEINTERLACE, },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_process_func_current());
    gwy_param_def_add_gwyenum(paramdef, PARAM_METHOD, "method", _("Reordering"),
                              methods, G_N_ELEMENTS(methods), REORDER_MEANDER_LTR);
    return paramdef;
}

static void
module_main(GwyContainer *data, GwyRunType run)
{
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    ModuleArgs args;
    GQuark quark;
    gint id, newid, xres;

    g_return_if_fail(run & RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &args.field,
                                     GWY_APP_DATA_FIELD_KEY, &quark,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(args.field);
    args.result = gwy_data_field_duplicate(args.field);
    xres = gwy_data_field_get_xres(args.field);
    args.square_image = (gwy_data_field_get_yres(args.field)) == xres;
    args.power2_image = FALSE;
    if (args.square_image)
        args.power2_image = !(xres ^ (xres-1));

    args.params = gwy_params_new_from_settings(define_module_params());
    if (run == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (outcome != GWY_DIALOG_HAVE_RESULT)
        execute(&args);

    newid = gwy_app_data_browser_add_data_field(args.result, data, TRUE);
    gwy_app_set_data_field_title(data, newid, _("Reordered"));
    gwy_app_sync_data_items(data, data, id, newid, FALSE,
                            GWY_DATA_ITEM_GRADIENT,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            GWY_DATA_ITEM_RANGE,
                            0);
    gwy_app_channel_log_add_proc(data, id, newid);

end:
    g_clear_object(&args.result);
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args)
{
    ModuleGUI gui;
    GwyDialog *dialog;
    GwyParamTable *table;

    gui.args = args;
    gui.dialog = gwy_dialog_new(_("Reorder"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    table = gui.table = gwy_param_table_new(args->params);
    gwy_param_table_append_combo(table, PARAM_METHOD);
    gwy_dialog_add_content(dialog, gwy_param_table_widget(table), FALSE, FALSE, 0);
    gwy_dialog_add_param_table(dialog, table);

    return gwy_dialog_run(dialog);
}

static void
execute(ModuleArgs *args)
{
    GwyParams *params = args->params;
    ReorderMethod method = gwy_params_get_enum(params, PARAM_METHOD);
    GwyDataField *field = args->field, *result = gwy_data_field_new_alike(field, FALSE);
    gint xres = gwy_data_field_get_xres(field), yres = gwy_data_field_get_yres(field);
    const gdouble *src = gwy_data_field_get_data_const(field);
    double *dest = gwy_data_field_get_data(result);
    gint i, j;

    if (method == REORDER_MEANDER_LTR || method == REORDER_MEANDER_RTL) {
        for (i = 0; i < yres; i++) {
            const gdouble *srow = src + i*xres;
            double *drow = dest + i*xres;
            gboolean is_odd = i % 2, is_even = !is_odd;

            if ((method == REORDER_MEANDER_LTR && is_odd) || (method == REORDER_MEANDER_RTL && is_even)) {
                for (j = 0; j < xres; j++)
                    drow[xres-1 - j] = srow[j];
            }
            else
                gwy_assign(drow, srow, xres);
        }
    }
    else if (method == REORDER_INTERLACE) {
        gint yhalf = (yres + 1)/2;
        for (i = 0; i < yres; i++) {
            const gdouble *srow = src + i*xres;
            gboolean is_even = !(i % 2);
            gdouble *drow = dest + i/2*xres + (is_even ? 0 : yhalf*xres);
            gwy_assign(drow, srow, xres);
        }
    }
    else if (method == REORDER_DEINTERLACE) {
        gint yhalf = (yres + 1)/2;
        for (i = 0; i < yhalf; i++) {
            const gdouble *srow = src + i*xres;
            gdouble *drow = dest + 2*i*xres;
            gwy_assign(drow, srow, xres);
        }
        for (i = yhalf; i < yres; i++) {
            const gdouble *srow = src + i*xres;
            gdouble *drow = dest + (2*(i - yhalf) + 1)*xres;
            gwy_assign(drow, srow, xres);
        }
    }

    args->result = result;
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
