/*
 *  $Id: volumeops.c 26421 2024-06-28 11:19:22Z yeti-dn $
 *  Copyright (C) 2003 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libprocess/brick.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-volume.h>
#include <app/gwyapp.h>

#define VOLUMEOPS_RUN_MODES (GWY_RUN_IMMEDIATE)

static gboolean module_register(void);
static void     volume_imagize(GwyContainer *data,
                               GwyRunType run);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Extracts all XY slices from volume data"),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2025",
};

GWY_MODULE_QUERY2(module_info, volume_imagize)

static gboolean
module_register(void)
{
    gwy_volume_func_register("volume_imagize",
                             (GwyVolumeFunc)&volume_imagize,
                             N_("/_Basic Operations/_Extract All _XY Planes"),
                             NULL,
                             VOLUMEOPS_RUN_MODES,
                             GWY_MENU_FLAG_VOLUME,
                             N_("Extract all XY planes to images"));

    return TRUE;
}

static void
volume_imagize(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield = NULL;
    GwyBrick *brick;
    gint id, newid, k, zres;
    gchar *title;
    gchar newtitle[100];

    g_return_if_fail(run & VOLUMEOPS_RUN_MODES);

    gwy_app_data_browser_get_current(GWY_APP_BRICK, &brick,
                                     GWY_APP_BRICK_ID, &id, 0);

    title = gwy_app_get_brick_title(data, id);
    zres = gwy_brick_get_zres(brick);
    for (k = 0; k < zres; k++) {
        dfield = gwy_data_field_new(gwy_brick_get_xres(brick),
                                    gwy_brick_get_yres(brick),
                                    gwy_brick_get_xreal(brick),
                                    gwy_brick_get_yreal(brick),
                                    FALSE);
        gwy_brick_extract_xy_plane(brick, dfield, k);
        newid = gwy_app_data_browser_add_data_field(dfield, data, !k);
        g_object_unref(dfield);

        g_snprintf(newtitle, sizeof(newtitle), _("%s, slice %d"), title, k);
        gwy_app_set_data_field_title(data, newid, newtitle);

        gwy_app_channel_log_add(data, -1, newid, "volume::imagize", NULL);
    }
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
