/*
 *  $Id: volume_localdetect.c 29047 2025-12-27 13:44:22Z klapetek $
 *  Copyright (C) 2015-2023 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <fftw3.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwynlfit.h>
#include <libprocess/brick.h>
#include <libprocess/stats.h>
#include <libprocess/linestats.h>
#include <libprocess/gwyprocess.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/simplefft.h>
#include <libprocess/correlation.h>
#include <libgwydgets/gwydataview.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-volume.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "libgwyddion/gwyomp.h"

#define RUN_MODES (GWY_RUN_INTERACTIVE)

#define gwycreal(x) ((x)[0])
#define gwycimag(x) ((x)[1])

/* Do not require FFTW 3.3 just for a couple of trivial macros. */
#define gwy_fftw_new_real(n) (gdouble*)fftw_malloc((n)*sizeof(gdouble))
#define gwy_fftw_new_complex(n) (fftw_complex*)fftw_malloc((n)*sizeof(fftw_complex))

enum {
    PREVIEW_SIZE = 360,
    RESPONSE_HBW = 1000,
    HEADER_SIZE = 1024,
};

enum {
    PARAM_METHOD,
    PARAM_WIDTH,
    PARAM_HEIGHT,
    PARAM_THRESHOLD,
    PARAM_UPSCALE,
    PARAM_KEEPUP,
    PARAM_VOLDATA,
    PARAM_ZRES,
    PARAM_HBW,
    BUTTON_HBW,
//    PARAM_UPDATE
};

typedef enum {
    METHOD_PIXEL    = 0,
    METHOD_SUBPIXEL = 1,
    NMETHODS
} SearchMethod;

typedef struct {
    GwyParams *params;
    GwyBrick *brick;
    GwyBrick *volresult;
    GwyDataField *result;
    gdouble **px;
    gdouble **py;
    gdouble **pz;
    gint *np;
    GwyContainer *meta;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table_options;
    GwyContainer *data;
    gboolean computed;
} ModuleGUI;


static gboolean              module_register          (void);
static GwyParamDef*          define_module_params     (void);
static void                  localdetect              (GwyContainer *data,
                                                       GwyRunType runtype);
static void                  execute                  (ModuleArgs *args,
                                                       GtkWindow *wait_window,
                                                       gboolean resample,
                                                       gboolean volrun);
static GwyDialogOutcome      run_gui                  (ModuleArgs *args,
                                                       GwyContainer *data,
                                                       gint id);
static void                  param_changed            (ModuleGUI *gui,
                                                       gint id);
static void                  preview                  (gpointer user_data);
static gboolean              create_voldata           (ModuleArgs *args);
static gint                  fill_voldata             (ModuleArgs *args,
                                                       GwyBrick *out,
                                                       gdouble width);
static void                  gwy_brick_fft            (GwyBrick *in,
                                                       GwyBrick *out_re,
                                                       GwyBrick *out_im);
static void                  gwy_data_field_filter_gaussian_real(GwyDataField *data_field,
                                                                 gdouble sigma);
static void                  dialog_response          (ModuleGUI *gui,
                                                       gint response);
static gdouble               calculate_hbw            (ModuleGUI *gui);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Performs localization merge of all the levels"),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.1",
    "Petr Klapetek & David Nečas (Yeti)",
    "2023",
};

GWY_MODULE_QUERY2(module_info, volume_localdetect)

static gboolean
module_register(void)
{
    gwy_volume_func_register("volume_localdetect",
                             (GwyVolumeFunc)&localdetect,
                             N_("/SPM M_odes/_Localization Merge..."),
                             NULL,
                             RUN_MODES,
                             GWY_MENU_FLAG_VOLUME,
                             N_("Perform localization merge"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;
    static const GwyEnum methods[] = {
        { N_("_pixel"),      METHOD_PIXEL,     },
        { N_("_sub-pixel"),  METHOD_SUBPIXEL,  },
    };

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_volume_func_current());
    gwy_param_def_add_gwyenum(paramdef, PARAM_METHOD, "method", _("Local maxima search"),
                              methods, G_N_ELEMENTS(methods), METHOD_PIXEL);

    gwy_param_def_add_int(paramdef, PARAM_UPSCALE, "upscale", _("_Upsampling factor"), 1, 5, 1);
    gwy_param_def_add_int(paramdef, PARAM_ZRES, "zres", _("_Z resolution"), 10, 500, 100);
    gwy_param_def_add_boolean(paramdef, PARAM_KEEPUP, "keepup", _("Keep upsampled"), FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_VOLDATA, "voldata", _("Extract volume density"), FALSE);
    gwy_param_def_add_double(paramdef, PARAM_HBW, "hbw", _("_Half-bit wavelength"),
                             0, G_MAXDOUBLE, 0.2e-9);
     gwy_param_def_add_double(paramdef, PARAM_WIDTH, "peak_width", _("_Peak width"),
                             0, G_MAXDOUBLE, 0.2e-9);
    gwy_param_def_add_double(paramdef, PARAM_HEIGHT, "height", _("_Height threshold"),
                             0, G_MAXDOUBLE, 0.2e-9);
    gwy_param_def_add_double(paramdef, PARAM_THRESHOLD, "filter", _("Noise _filter width"),
                             0, G_MAXDOUBLE, 0.2e-9);
//    gwy_param_def_add_instant_updates(paramdef, PARAM_UPDATE, "update", NULL, TRUE);
    return paramdef;
}

static void
localdetect(GwyContainer *data, GwyRunType runtype)
{
    ModuleArgs args;
    GwyBrick *brick = NULL;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    gint oldid, newid, k;
    gboolean keepup, voldata;
    const guchar *gradient;

    g_return_if_fail(runtype & RUN_MODES);
    g_return_if_fail(g_type_from_name("GwyLayerPoint"));

    gwy_app_data_browser_get_current(GWY_APP_BRICK, &brick,
                                     GWY_APP_BRICK_ID, &oldid,
                                     0);
    g_return_if_fail(GWY_IS_BRICK(brick));
    args.result = NULL;
    args.volresult = NULL;
    args.meta = NULL;
    args.brick = brick;
    args.params = gwy_params_new_from_settings(define_module_params());
    args.px = args.py = args.pz = NULL;
    args.np = NULL;

    if (runtype == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, oldid);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }

    keepup = gwy_params_get_boolean(args.params, PARAM_KEEPUP);
    voldata = gwy_params_get_boolean(args.params, PARAM_VOLDATA);
    if (outcome != GWY_DIALOG_HAVE_RESULT || keepup || (voldata && !args.volresult)) {
        execute(&args, gwy_app_find_window_for_volume(data, oldid), !keepup, voldata);
    }

    if (args.result) {
        newid = gwy_app_data_browser_add_data_field(args.result, data, TRUE);

        if (gwy_container_gis_string(data, gwy_app_get_brick_palette_key_for_id(oldid), &gradient))
            gwy_container_set_const_string(data, gwy_app_get_data_palette_key_for_id(newid), gradient);

        gwy_app_set_data_field_title(data, newid, _("Localization result"));
        gwy_app_sync_data_items(data, data, oldid, newid, FALSE,
                                GWY_DATA_ITEM_GRADIENT,
                                0);
    }
    if (voldata && args.volresult) {
        newid = gwy_app_data_browser_add_brick(args.volresult, NULL, data, TRUE);

        /* XXX: The default spelling should be American. */
        gwy_app_set_brick_title(data, newid, _("Localisation density"));
        gwy_app_sync_volume_items(data, data, oldid, newid, FALSE,
                                  GWY_DATA_ITEM_GRADIENT,
                                  0);
        if (args.meta)
            gwy_container_set_object(data, gwy_app_get_brick_meta_key_for_id(newid), args.meta);
    }

end:
    g_object_unref(args.params);
    if (args.result)
        g_object_unref(args.result);
    if (args.volresult)
        g_object_unref(args.volresult);
    if (args.meta)
        g_object_unref(args.meta);

    if (args.px) {
        for (k = 0; k < gwy_brick_get_zres(brick); k++)
            g_free(args.px[k]);
        g_free(args.px);
    }
    if (args.py) {
        for (k = 0; k < gwy_brick_get_zres(brick); k++)
            g_free(args.py[k]);
        g_free(args.py);
    }
    if (args.pz) {
        for (k = 0; k < gwy_brick_get_zres(brick); k++)
            g_free(args.pz[k]);
        g_free(args.pz);
    }

}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyContainer *data, gint id)
{
    GtkWidget *hbox, *dataview;
    GwyParamTable *table;
    GwyDialog *dialog;
    ModuleGUI gui;
    GwyDialogOutcome outcome;
    GwyBrick *brick = args->brick;
    GwyDataField *result = gwy_data_field_new(gwy_brick_get_xres(brick), gwy_brick_get_yres(brick),
                                              gwy_brick_get_xreal(brick), gwy_brick_get_yreal(brick),
                                              TRUE);
    GwyDataField *dfield = gwy_data_field_new_alike(result, FALSE);
    GwyDataField *slice = gwy_data_field_new_alike(result, FALSE);
    const guchar *gradient;
    GwySIValueFormat *vf;
    gdouble min, max, grange;
    gint k, zres = gwy_brick_get_zres(brick);

    gwy_clear(&gui, 1);
    gui.args = args;
    gui.data = gwy_container_new();
    gui.computed = FALSE;
    args->result = result;

    gwy_data_field_set_si_unit_xy(result, gwy_brick_get_si_unit_x(brick));
    gwy_data_field_set_si_unit_z(result, gwy_brick_get_si_unit_w(brick));

    gwy_container_set_object(gui.data, gwy_app_get_data_key_for_id(0), result);
    if (gwy_container_gis_string(data, gwy_app_get_brick_palette_key_for_id(id), &gradient))
        gwy_container_set_const_string(gui.data, gwy_app_get_data_palette_key_for_id(0), gradient);

    gui.dialog = gwy_dialog_new(_("Localization Merge"));
    dialog = GWY_DIALOG(gui.dialog);
    gtk_dialog_add_button(GTK_DIALOG(dialog), gwy_sgettext("verb|_Update"), GWY_RESPONSE_UPDATE);
    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    dataview = gwy_create_preview(gui.data, 0, PREVIEW_SIZE, FALSE);
    hbox = gwy_create_dialog_preview_hbox(GTK_DIALOG(dialog), GWY_DATA_VIEW(dataview), FALSE);

    table = gui.table_options = gwy_param_table_new(args->params);

    gwy_param_table_append_combo(table, PARAM_METHOD);

    gwy_param_table_append_slider(table, PARAM_UPSCALE);
    gwy_param_table_append_checkbox(table, PARAM_KEEPUP);

    gwy_brick_extract_xy_plane(brick, dfield, 0);
    gwy_param_table_append_slider(table, PARAM_THRESHOLD);
    gwy_param_table_set_unitstr(table, PARAM_THRESHOLD, "px");
    vf = gwy_si_unit_get_format(gwy_data_field_get_si_unit_xy(dfield), GWY_SI_UNIT_FORMAT_VFMARKUP,
                                gwy_data_field_get_xreal(dfield)/10, NULL);
    vf->precision++;
    gwy_param_table_slider_set_factor(table, PARAM_THRESHOLD, 1.0/vf->magnitude);
    gwy_param_table_set_unitstr(table, PARAM_THRESHOLD, vf->units);
    gwy_param_table_slider_restrict_range(table, PARAM_THRESHOLD, 0, gwy_data_field_get_xreal(dfield)/10);


    gwy_param_table_append_slider(table, PARAM_HEIGHT);
    gwy_param_table_slider_set_mapping(table, PARAM_HEIGHT, GWY_SCALE_MAPPING_LINEAR);
   
    grange = -G_MAXDOUBLE;
    for (k = 0; k < zres; k++) {
        gwy_brick_extract_xy_plane(brick, slice, k);
        gwy_data_field_get_min_max(slice, &min, &max);
        if (grange < (max - min)) {
            grange = max - min;
        }
    }

    vf = gwy_si_unit_get_format(gwy_data_field_get_si_unit_z(dfield), GWY_SI_UNIT_FORMAT_VFMARKUP, grange, NULL);
    vf->precision++;
    gwy_param_table_slider_set_factor(table, PARAM_HEIGHT, 1.0/vf->magnitude);
    gwy_param_table_set_unitstr(table, PARAM_HEIGHT, vf->units);
    gwy_param_table_slider_restrict_range(table, PARAM_HEIGHT, 0, grange);

    gwy_param_table_append_slider(table, PARAM_WIDTH);
    vf = gwy_si_unit_get_format(gwy_data_field_get_si_unit_xy(dfield), GWY_SI_UNIT_FORMAT_VFMARKUP,
                                gwy_data_field_get_xreal(dfield)/10, NULL);
    vf->precision++;
    gwy_param_table_slider_set_factor(table, PARAM_WIDTH, 1.0/vf->magnitude);
    gwy_param_table_set_unitstr(table, PARAM_WIDTH, vf->units);
    gwy_param_table_slider_restrict_range(table, PARAM_WIDTH, 0, gwy_data_field_get_xreal(dfield)/10);

    gwy_param_table_append_checkbox(table, PARAM_VOLDATA);
    gwy_param_table_append_slider(table, PARAM_ZRES);

    gwy_param_table_append_slider(table, PARAM_HBW);
    vf = gwy_si_unit_get_format(gwy_data_field_get_si_unit_xy(dfield), GWY_SI_UNIT_FORMAT_VFMARKUP,
                                gwy_data_field_get_xreal(dfield)/10, NULL);
    vf->precision++;
    gwy_param_table_slider_set_factor(table, PARAM_HBW, 1.0/vf->magnitude);
    gwy_param_table_set_unitstr(table, PARAM_HBW, vf->units);
    gwy_param_table_slider_restrict_range(table, PARAM_HBW, 0, gwy_data_field_get_xreal(dfield));

    /* XXX: This should be "verb|_Estimate" */
 //   gwy_param_table_append_button(table, BUTTON_HBW, -1, RESPONSE_HBW, _("_Estimate"));

//    gwy_param_table_append_checkbox(table, PARAM_UPDATE);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), TRUE, TRUE, 0);

    g_signal_connect_swapped(gui.table_options, "param-changed", G_CALLBACK(param_changed), &gui);
    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_UPON_REQUEST, preview, &gui, NULL);
    g_signal_connect_swapped(dialog, "response", G_CALLBACK(dialog_response), &gui);


    param_changed(&gui, PARAM_VOLDATA);
    outcome = gwy_dialog_run(dialog);

    g_object_unref(gui.data);
    g_object_unref(dfield);

    return outcome;
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    GwyParams *params = gui->args->params;
    GwyParamTable *table = gui->table_options;

    if (id == PARAM_VOLDATA) {
        gwy_param_table_set_sensitive(table,
                                      PARAM_ZRES,
                                      gwy_params_get_boolean(params, PARAM_VOLDATA));
        gwy_param_table_set_sensitive(table,
                                      PARAM_HBW,
                                      gwy_params_get_boolean(params, PARAM_VOLDATA));
        gwy_param_table_set_sensitive(table,
                                      BUTTON_HBW,
                                      gwy_params_get_boolean(params, PARAM_VOLDATA) && gui->computed);
    }
    else {
        if (id != PARAM_ZRES && id != PARAM_HBW) {
           gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), GTK_RESPONSE_OK, FALSE);
           gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
           gui->computed = FALSE;
           gwy_param_table_set_sensitive(table,
                                         BUTTON_HBW,
                                         gwy_params_get_boolean(params, PARAM_VOLDATA) && gui->computed);
        }
    }
}

//!!!! width hieght in pixels
static void
add_gaussian(GwyDataField *dfield, gdouble x, gdouble y, gdouble height, gdouble width)
{
    gint i, j;
    gint ix = (gint)x;
    gint iy = (gint)y;
    gint xsize, ysize;
    gdouble xc, yc;
    gint xres = gwy_data_field_get_xres(dfield);
    gint yres = gwy_data_field_get_yres(dfield);
    gdouble *data = gwy_data_field_get_data(dfield);
    gdouble xww = gwy_data_field_rtoi(dfield, width);
    gdouble yww = gwy_data_field_rtoj(dfield, width);

    xsize = 10*xww;
    ysize = 10*yww;
    xww = 2*xww*xww;
    yww = 2*yww*yww;
 
    for (j = MAX(0, iy - ysize); j < MIN(yres, iy + ysize); j++) {
        for (i = MAX(0, ix - xsize); i < MIN(xres, ix + xsize); i++) {
            xc = x - i;
            yc = y - j;
            data[i + xres*j] += height * exp(-(xc*xc/xww + yc*yc/yww));
        }
    }
}

//!!!!!! width height in pixels
static void
add_3d_gaussian(GwyBrick *brick, gdouble x, gdouble y, gdouble z, gdouble width)
{
    gint i, j, k;
    gint xsize, ysize, zsize;
    gdouble xc, yc, zc;
    gint xres = gwy_brick_get_xres(brick);
    gint yres = gwy_brick_get_yres(brick);
    gint zres = gwy_brick_get_zres(brick);
    gdouble *data = gwy_brick_get_data(brick);
    gdouble xww = gwy_brick_rtoi(brick, width);
    gdouble yww = gwy_brick_rtoj(brick, width);
    gdouble zww = gwy_brick_rtok(brick, width);
    gdouble px = gwy_brick_rtoi(brick, x);
    gdouble py = gwy_brick_rtoj(brick, y);
    gdouble pz = gwy_brick_rtok(brick, z - gwy_brick_get_zoffset(brick));
    gint ix = (gint)px;
    gint iy = (gint)py;
    gint iz = (gint)pz;
 
    xsize = 10*xww;
    ysize = 10*yww;
    zsize = 10*zww;

    xww = 2*xww*xww;
    yww = 2*yww*yww;
    zww = 2*zww*zww;

    for (k = MAX(0, iz - zsize); k < MIN(zres, iz + zsize); k++) {
        for (j = MAX(0, iy - ysize); j < MIN(yres, iy + ysize); j++) {
            for (i = MAX(0, ix - xsize); i < MIN(xres, ix + xsize); i++) {
                xc = px - i;
                yc = py - j;
                zc = pz - k;
                data[i + xres*j + k*xres*yres] += exp(-(xc*xc/xww + yc*yc/yww + zc*zc/zww));
            }
        }
    }
}

static gint
fill_voldata(ModuleArgs *args, GwyBrick *out, gdouble width)
{
    gint zres = gwy_brick_get_zres(args->brick);
    gdouble zmin, zmax;
    gint i, k, np;
 
    if (!gwy_app_wait_set_message("Creating volume density..."))
        return FALSE;

    zmin = G_MAXDOUBLE;
    zmax = -G_MAXDOUBLE;
    for (k = 0; k < zres; k++) {
        for (i = 0; i < args->np[k]; i++) {
            if (args->pz[k][i] < zmin)
                zmin = args->pz[k][i];
            if (args->pz[k][i] > zmax)
                zmax = args->pz[k][i];
         }
    }

    //add 2*width to the z size to have space for gaussians
    zmin -= 2*width;
    zmax += 2*width;

    gwy_brick_set_zreal(out, zmax-zmin);
    gwy_brick_set_zoffset(out, zmin);
    gwy_brick_fill(out, 0);

    if (!gwy_app_wait_set_fraction(0))
        return FALSE;

    for (k = 0; k < zres; k++) {
        for (i = 0; i < args->np[k]; i++) {
            add_3d_gaussian(out, args->px[k][i], args->py[k][i], args->pz[k][i],
                            width);
        }
        np += args->np[k];

        if (!gwy_app_wait_set_fraction((gdouble)k/zres))
            return FALSE;
    }

    return np;
}

static gboolean
create_voldata(ModuleArgs *args)
{
    GwyBrick *brick = args->brick;
    GwyBrick *out = args->volresult;
    GwyParams *params = args->params;
    gint xres = gwy_brick_get_xres(brick);
    gint yres = gwy_brick_get_yres(brick);
    gint zres = gwy_brick_get_zres(brick);
    gint ezres = gwy_params_get_int(params, PARAM_ZRES);
    gdouble width = gwy_params_get_double(params, PARAM_HBW);
    gint upscale = gwy_params_get_int(params, PARAM_UPSCALE);
    gint np;
    GwySIValueFormat *vf;

    args->volresult = out = gwy_brick_new_alike(brick, FALSE);
    gwy_brick_set_si_unit_z(out, gwy_si_unit_new("m"));
    gwy_brick_resample(out, xres, yres, ezres, GWY_INTERPOLATION_NONE);
    np = fill_voldata(args, out, width);

    if (args->meta)
        g_object_unref(args->meta);

    args->meta = gwy_container_new();
    gwy_container_set_string_by_name(args->meta, "Number of particles", g_strdup_printf("%d", np));
    gwy_container_set_string_by_name(args->meta, "X particle pixel size",
                                     g_strdup_printf("%g", gwy_brick_rtoi(out, width)));
    gwy_container_set_string_by_name(args->meta, "Y particle pixel size",
                                     g_strdup_printf("%g", gwy_brick_rtoj(out, width)));

    vf = gwy_si_unit_get_format(gwy_brick_get_si_unit_z(out), GWY_SI_UNIT_FORMAT_PLAIN,
                                width,
                                NULL);
    gwy_container_set_string_by_name(args->meta, "X particle real size", 
                                     g_strdup_printf("%.*f %s", vf->precision,
                                                     width/vf->magnitude,
                                                     vf->units));
    gwy_container_set_string_by_name(args->meta, "Y particle real size", 
                                     g_strdup_printf("%.*f %s", vf->precision,
                                                     width/vf->magnitude,
                                                     vf->units));

    vf = gwy_si_unit_get_format(gwy_brick_get_si_unit_z(out), GWY_SI_UNIT_FORMAT_PLAIN,
                                gwy_brick_get_zreal(out)/zres,
                                vf);
    gwy_container_set_string_by_name(args->meta, "Z particle resolution", 
                                     g_strdup_printf("%.*f %s", vf->precision,
                                                     gwy_brick_get_zreal(out)/zres/vf->magnitude,
                                                     vf->units));

    gwy_container_set_string_by_name(args->meta, "Bicubic expansion scale", g_strdup_printf("%d", upscale));
    gwy_container_set_string_by_name(args->meta, "Detections", g_strdup_printf("%d", np));

    vf = gwy_si_unit_get_format(gwy_brick_get_si_unit_x(brick), GWY_SI_UNIT_FORMAT_PLAIN,
                                gwy_brick_get_xreal(brick)/xres,
                                vf);
    gwy_container_set_string_by_name(args->meta, "Half-bit WL of image stack", 
                                     g_strdup_printf("%.*f %s", vf->precision,
                                                     gwy_brick_get_xreal(brick)/xres/vf->magnitude,
                                                     vf->units));

    vf = gwy_si_unit_get_format(gwy_brick_get_si_unit_x(out), GWY_SI_UNIT_FORMAT_PLAIN,
                                gwy_brick_get_xreal(out)/gwy_brick_get_xres(out),
                                vf);
    gwy_container_set_string_by_name(args->meta, "Half-bit WL of density volume", 
                                     g_strdup_printf("%.*f %s", vf->precision,
                                                     gwy_brick_get_xreal(out)/gwy_brick_get_xres(out)/vf->magnitude,
                                                     vf->units));

    gwy_container_set_string_by_name(args->meta, "Density function", g_strdup_printf("%d", 0));

    gwy_si_unit_value_format_free(vf);
    return TRUE;
}

static void
dialog_response(ModuleGUI *gui, gint response)
{   
    if (response == RESPONSE_HBW) {
        gwy_param_table_set_double(gui->table_options, PARAM_HBW, calculate_hbw(gui));
    }
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    execute(gui->args, GTK_WINDOW(gui->dialog), TRUE, FALSE);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), GTK_RESPONSE_OK, TRUE);
    gwy_dialog_have_result(GWY_DIALOG(gui->dialog));
    gui->computed = TRUE;
    param_changed(gui, PARAM_VOLDATA);
}

static gint
count_2dmask_particles(ModuleArgs *args, GwyDataField *mask)
{
    GwyBrick *brick = args->brick;
    gint xres = gwy_brick_get_xres(brick);
    gint zres = gwy_brick_get_zres(brick);
    gint i, k, sum = 0;
    gdouble *dm = gwy_data_field_get_data(mask);
    gint px, py;

    for (k = 0; k < zres; k++) {
        for (i = 0; i < args->np[k]; i++) {
            px = gwy_brick_rtoi(brick, args->px[k][i]);
            py = gwy_brick_rtoj(brick, args->py[k][i]);
            if (dm[px + xres*py])
                sum++;
        }
    }
    return sum;
}

static gint
count_3dmask_particles(ModuleArgs *args, GwyBrick *mask)
{
    GwyBrick *brick = args->brick;
    gint xres = gwy_brick_get_xres(brick);
    gint yres = gwy_brick_get_yres(brick);
    gint zres = gwy_brick_get_zres(brick);
    gint i, k, sum = 0;
    gdouble *dm = gwy_brick_get_data(mask);
    gint px, py, pz;

    for (k = 0; k < zres; k++) {
        for (i = 0; i < args->np[k]; i++) {
            px = gwy_brick_rtoi(brick, args->px[k][i]);
            py = gwy_brick_rtoj(brick, args->py[k][i]);
            pz = gwy_brick_rtoj(brick, args->py[k][i]);
            if (dm[px + xres*py + xres*yres*pz])
                sum++;
        }
    }
    return sum;
}

static void
grow_3dmask(GwyBrick *mask)
{
    gint col, row, lev;
    gint xres = gwy_brick_get_xres(mask);
    gint yres = gwy_brick_get_yres(mask);
    gint zres = gwy_brick_get_zres(mask);
    gdouble *newzl = g_new0(gdouble, zres);
    gdouble *mdata = gwy_brick_get_data(mask);

    for (col=0; col < yres; col++) {
        for (row = 0; row < xres; row++) {
           for (lev = 0; lev < zres; lev++)
                newzl[lev] = 0;
           for (lev = 1; lev < (zres - 1); lev++) {
               if (mdata[xres*yres*lev + xres*row + col]) {
                   newzl[lev-1] = newzl[lev] = newzl[lev+1] = 1;
               }
           }
           for (lev = 1; lev < (zres - 1); lev++) {
               mdata[xres*yres*lev + xres*row + col] = newzl[lev];
           }
        }
    }
    g_free(newzl);
}
static void
shrink_3dmask(GwyBrick *mask)
{
    gint col, row, lev;
    gint xres = gwy_brick_get_xres(mask);
    gint yres = gwy_brick_get_yres(mask);
    gint zres = gwy_brick_get_zres(mask);
    gdouble *newzl = g_new(gdouble, zres);
    gdouble *mdata = gwy_brick_get_data(mask);

    for (col = 0; col < yres; col++) {
        for (row = 0; row < xres; row++) { 
           for (lev = 0; lev < zres; lev++)
                newzl[lev] = 1;
           for (lev = 1; lev < (zres - 1); lev++) {
               if (mdata[xres*yres*lev + xres*row + col] == 0) {
                   newzl[lev-1] = newzl[lev] = newzl[lev+1] = 0;
               }
           }
           for (lev = 1; lev < (zres - 1); lev++) {
               mdata[xres*yres*lev + xres*row + col] = newzl[lev];
           }
        }
    }
    g_free(newzl);
}


static gdouble
calculate_hbw(ModuleGUI *gui)
{
    gint i, j, k;
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;
    GwyBrick *brick = args->brick;
    GwyBrick *density, *dmask, *dmodmask, *half1, *half2;
    gint xres = gwy_brick_get_xres(brick);
    gint yres = gwy_brick_get_yres(brick);
    gint zres = gwy_brick_get_zres(brick);
    gint ezres = gwy_params_get_int(params, PARAM_ZRES);
    gdouble *gsresults, *gsdiff;
    gint ngs;
    GwyDataField *mask = gwy_data_field_new(xres, yres, xres, yres, FALSE);
    GwyDataField *modmask = gwy_data_field_new(xres, yres, xres, yres, FALSE);

    gwy_app_wait_start(GTK_WINDOW(gui->dialog), _("HBW estimate: initializing 3D mask..."));

    //create volume density with small gaussian
    density = gwy_brick_new_alike(brick, TRUE);
    gwy_brick_set_si_unit_z(density, gwy_si_unit_new("m"));
    gwy_brick_resample(density, xres, yres, ezres, GWY_INTERPOLATION_NONE);
    fill_voldata(args, density, gwy_brick_get_xreal(brick)/xres);

    dmask = gwy_brick_new_alike(density, TRUE);
    dmodmask = gwy_brick_new_alike(density, TRUE);
 
    //create 2d mask, init and grow/shrink
    gwy_brick_sum_xy_plane(density, mask);
    gwy_data_field_filter_mean(mask, 2); //should be user selectable?
    gwy_data_field_threshold(mask, gwy_data_field_otsu_threshold(mask), 0.0, 1.0);

    fprintf(stderr, "%d %d\n", xres, yres);
    for (j=0; j<yres; j++) {
        for (i = 0; i < xres; i++) 
           fprintf(stderr, "%g ", gwy_data_field_get_val(mask, i, j));
        fprintf(stderr, "\n");
    }

    ngs = xres/2;
    gsresults = g_new(gdouble, ngs);
    gsdiff = g_new(gdouble, ngs);
    gsresults[ngs/2] = count_2dmask_particles(args, mask);

    gwy_data_field_copy(mask, modmask, FALSE);
    for (i = ngs/2 + 1; i < ngs; i++) {
        gwy_data_field_grains_grow(modmask, 1, GWY_DISTANCE_TRANSFORM_EUCLIDEAN, FALSE);
        gsresults[i] = count_2dmask_particles(args, modmask);
    }

    gwy_data_field_copy(mask, modmask, FALSE);
    for (i = (ngs/2 - 1); i >= 0; i--) {
        gwy_data_field_grains_shrink(modmask, 1, GWY_DISTANCE_TRANSFORM_EUCLIDEAN, FALSE);
        gsresults[i] = count_2dmask_particles(args, modmask);
    }
    gsdiff[0] = 0;
    for (i = 1; i < ngs; i++)
        gsdiff[i] = gsresults[i] - gsresults[i-1];

    for (i = 0; i < ngs; i++)
        printf("%d %g %g\n", i, gsresults[i], gsdiff[i]);

    printf("going to 3D\n");
    //for test purposes use original otsu's mask
    //create 3d mask, init and grow/shrink
    gwy_brick_copy(density, dmask, FALSE);
    gwy_brick_threshold(dmask, (gwy_brick_get_max(density) - gwy_brick_get_min(density))/2, 0, 1);
    gsresults[ngs/2] = count_3dmask_particles(args, dmask);

    gwy_brick_copy(dmask, dmodmask, FALSE);
    for (i = ngs/2 + 1; i < ngs; i++) {
        grow_3dmask(dmodmask);
        gsresults[i] = count_3dmask_particles(args, dmodmask);
    }
    gwy_brick_copy(dmask, dmodmask, FALSE);
    for (i = (ngs/2 - 1); i >= 0; i--) {
        shrink_3dmask(dmodmask);
        gsresults[i] = count_3dmask_particles(args, dmodmask);
    }
    gsdiff[0] = 0;
    for (i = 1; i < ngs; i++)
        gsdiff[i] = gsresults[i] - gsresults[i-1];

    for (i = 0; i < ngs; i++)
        printf("%d %g %g\n", i, gsresults[i], gsdiff[i]);

    //as the effectiveness of grow/shrink strategy might be problematic on
    //general datasets, use the original 3d mask for further tests

    //create masked volume density, two halfs
    if (!gwy_app_wait_set_fraction(0))
        return FALSE;

    for (k = 0; k < zres; k++) {
        for (i = 0; i < (args->np[k]-1); i+=2) {
            add_3d_gaussian(half1, args->px[k][i], args->py[k][i], args->pz[k][i],
                            1.0);
            add_3d_gaussian(half2, args->px[k][i+1], args->py[k][i+1], args->pz[k][i+1],
                            1.0);
         }
         if (!gwy_app_wait_set_fraction((gdouble)k/zres))
            return FALSE;
    }

    //calculate HBW by FSC
    //fsc(half1, half2);

    gwy_app_wait_finish();

    g_free(gsresults);
    g_free(gsdiff);
    gwy_object_unref(density);
    gwy_object_unref(dmask);
    gwy_object_unref(dmodmask);
    gwy_object_unref(half1);
    gwy_object_unref(half2);

    return 2e-9;
}


static gint
filter_extrema_field(GwyDataField *dfield, GwyDataField *maxs, gdouble threshold)
{
    gint i, n = gwy_data_field_get_xres(dfield)*gwy_data_field_get_yres(dfield);
    gdouble *data = gwy_data_field_get_data(dfield);
    gdouble *mdata = gwy_data_field_get_data(maxs);
    gint sum = 0;

    for (i = 0; i < n; i++) {
        if (mdata[i] > 0 && data[i] > threshold) {
            mdata[i] = data[i];
            sum++;
        } else
            mdata[i] = 0;
    }
    return sum;
}


static gint
subpixel_gaussians(GwyDataField *topography, GwyDataField *peaks, gdouble width,
                   gdouble *px, gdouble *py, gdouble *pz, gint maxnp)
{
    gint col, row, ri, rj, rn, n = 0;
    gdouble xoffset, yoffset, *pdata, *tdata;
    GwyDataField *original = gwy_data_field_duplicate(peaks);
    gint xres = gwy_data_field_get_xres(original);
    gint yres = gwy_data_field_get_yres(original);
    gdouble zvals[9];

    gwy_data_field_clear(peaks);
    pdata = gwy_data_field_get_data(original);
    tdata = gwy_data_field_get_data(topography);

    for (row = 2; row < yres-2; row++) {
        for (col = 2; col < xres-2; col++) {
            if (pdata[col + xres*row] > 0 && n < maxnp) { //a local maximum in copied peaks channel
                //refine local maximum position
                rn = 0;
                for (rj = -1; rj <= 1; rj++) {
                    for (ri = -1; ri <= 1; ri++) {
                        zvals[rn++] = tdata[col + ri + xres*(row + rj)]; //use topography for neighborhood
                    }
                }
                gwy_math_refine_maximum_2d(zvals, &xoffset, &yoffset);
                xoffset += col;
                yoffset += row;

                add_gaussian(peaks, xoffset, yoffset, tdata[col + xres*row], width);

                px[n] = gwy_data_field_itor(topography, xoffset);
                py[n] = gwy_data_field_jtor(topography, yoffset);
                pz[n] = tdata[col + xres*row];
                n++;
             }
        }
    }
    g_object_unref(original);
    return n;
}

static gint
fill_particles(GwyDataField *maxs, gdouble *px, gdouble *py, gdouble *pz, gint maxnp)
{
    gint col, row, n = 0;
    gdouble *data = gwy_data_field_get_data(maxs);
    gint xres = gwy_data_field_get_xres(maxs);
    gint yres = gwy_data_field_get_yres(maxs);

    for (row = 0; row < yres; row++) {
        for (col = 0; col < xres; col++) { 
            if (data[col + xres*row] != 0 && n < maxnp) {
                px[n] = gwy_data_field_itor(maxs, col);
                py[n] = gwy_data_field_jtor(maxs, row);
                pz[n] = data[col + xres*row];
                n++;
            }
        } 
    }
    return n;
}


static void
execute(ModuleArgs *args, GtkWindow *wait_window, gboolean resample, gboolean volrun)
{
    GwyParams *params = args->params;
    gint k;
    GwyBrick *brick = args->brick;
    GwyDataField *result = args->result;
    GwyDataField *dfield, *maxs, *sum;
    gint upscale = gwy_params_get_int(params, PARAM_UPSCALE);
    gdouble width = gwy_params_get_double(params, PARAM_WIDTH);
    gdouble height = gwy_params_get_double(params, PARAM_HEIGHT);
    gdouble threshold = gwy_params_get_double(params, PARAM_THRESHOLD);
    gboolean voldata = gwy_params_get_boolean(params, PARAM_VOLDATA);

    gint method = gwy_params_get_enum(params, PARAM_METHOD);
    gboolean cancelled = FALSE;

    gint xres = gwy_brick_get_xres(brick);
    gint yres = gwy_brick_get_yres(brick);
    gint zres = gwy_brick_get_zres(brick);

    gwy_app_wait_start(wait_window, _("Running localization..."));

    dfield = gwy_data_field_new(xres, yres, gwy_brick_get_xreal(brick), gwy_brick_get_yreal(brick), FALSE);
    sum = gwy_data_field_new(upscale*xres, upscale*yres,
                             gwy_brick_get_xreal(brick), gwy_brick_get_yreal(brick),
                             TRUE);
    maxs = gwy_data_field_new_alike(sum, TRUE);

    if (args->px) {
        for (k = 0; k < zres; k++)
            g_free(args->px[k]);
        g_free(args->px);
    }
    if (args->py) {
        for (k = 0; k < zres; k++)
            g_free(args->py[k]);
        g_free(args->py);
    }
    if (args->pz) {
        for (k = 0; k < zres; k++)
            g_free(args->pz[k]);
        g_free(args->pz);
    }
    if (args->np)
        g_free(args->np);

    args->px = g_new(gdouble*, (guint)zres);
    args->py = g_new(gdouble*, (guint)zres);
    args->pz = g_new(gdouble*, (guint)zres);
    args->np = g_new0(gint, (guint)zres);

    for (k = 0; k < zres; k++) {
        gwy_brick_extract_xy_plane(brick, dfield, k);
        gwy_data_field_filter_gaussian_real(dfield, threshold);  //run pre-filtering
        gwy_data_field_add(dfield, -gwy_data_field_get_min(dfield)); //shift to minimum to enable height threshold:
                                                                     //FIXME should be not impacted by noise?

        if (upscale != 1)   //if requested, work with upsampled images
            gwy_data_field_resample(dfield, upscale*xres, upscale*yres, GWY_INTERPOLATION_SCHAUM);

        gwy_data_field_mark_extrema(dfield, maxs, TRUE);    //find maxima mask, now in maxs
        args->np[k] = filter_extrema_field(dfield, maxs, height); //add height information to maxs

        args->px[k] = g_new(gdouble, args->np[k]);
        args->py[k] = g_new(gdouble, args->np[k]);
        args->pz[k] = g_new(gdouble, args->np[k]);

        if (method == METHOD_PIXEL) {
            args->np[k] = fill_particles(maxs, args->px[k], args->py[k], args->pz[k], args->np[k]);
            gwy_data_field_filter_gaussian_real(maxs, upscale*width);  //broaden it only
        }
        else {
            args->np[k] = subpixel_gaussians(dfield, maxs, upscale*width,
                                             args->px[k], args->py[k], args->pz[k], args->np[k]); 
            //find sub-pixel peak positions and broaden it
        }
        gwy_data_field_sum_fields(sum, sum, maxs);     //add it to the result

        if (!gwy_app_wait_set_fraction((gdouble)k/zres)) {
            cancelled = TRUE;
            break;
        }
    }

    if (voldata && volrun) {
        if (!create_voldata(args))
            gwy_object_unref(args->volresult);
    }

    gwy_app_wait_finish();

    if (!cancelled) {
        //upsample result field or downsample calculation output
        if (!resample)
            gwy_data_field_resample(result, upscale*xres, upscale*yres, GWY_INTERPOLATION_NONE);
        else
            gwy_data_field_resample(sum, xres, yres, GWY_INTERPOLATION_SCHAUM);

        gwy_data_field_copy(sum, result, FALSE);
        gwy_data_field_data_changed(result);
        gwy_data_field_multiply(result, 1.0/zres); //divide by number of images to get an average
    }

    g_object_unref(dfield);
    g_object_unref(maxs);
    g_object_unref(sum);
}

static void
gwy_data_field_filter_gaussian_real(GwyDataField *data_field,
                                    gdouble sigma)
{
    GwyDataLine *kernel;
    gdouble w;
    gint res, i;
    gint xres = gwy_data_field_get_xres(data_field);
    gint yres = gwy_data_field_get_yres(data_field);
    gdouble xsigma = gwy_data_field_rtoi(data_field, sigma);
    gdouble ysigma = gwy_data_field_rtoj(data_field, sigma);

    g_return_if_fail(GWY_IS_DATA_FIELD(data_field));
    if (xsigma == 0.0 && ysigma == 0.0)
        return;

    res = (gint)ceil(5.0*xsigma);
    res = 2*res + 1;
    i = 3*data_field->xres;
    if (res > i) {
        res = i;
        if (res % 2 == 0)
            res--;
    }

    kernel = gwy_data_line_new(res, 1.0, FALSE);
    for (i = 0; i < res; i++) {
        w = i - (res - 1)/2.0;
        w /= xsigma;
        kernel->data[i] = exp(-w*w/2.0);
    }
    gwy_data_line_multiply(kernel, 1.0/gwy_data_line_get_sum(kernel));
    gwy_data_field_area_convolve_1d(data_field, kernel, GWY_ORIENTATION_HORIZONTAL, 0, 0, xres, yres);

    res = (gint)ceil(5.0*ysigma);
    res = 2*res + 1;
    i = 3*data_field->yres;
    if (res > i) {
        res = i;
        if (res % 2 == 0)
            res--;
    }

    kernel = gwy_data_line_new(res, 1.0, FALSE);
    for (i = 0; i < res; i++) {
        w = i - (res - 1)/2.0;
        w /= ysigma;
        kernel->data[i] = exp(-w*w/2.0);
    }
    gwy_data_line_multiply(kernel, 1.0/gwy_data_line_get_sum(kernel));
    gwy_data_field_area_convolve_1d(data_field, kernel, GWY_ORIENTATION_VERTICAL, 0, 0, xres, yres);
    g_object_unref(kernel);
}

static void
gwy_brick_fft(GwyBrick *in, GwyBrick *out_re, GwyBrick *out_im)
{
    gint xres = gwy_brick_get_xres(in);
    gint yres = gwy_brick_get_yres(in);
    gint zres = gwy_brick_get_zres(in);
    gint i;
    gint n = xres*yres*zres;
    gdouble *indata = gwy_brick_get_data(in);
    gdouble *outredata = gwy_brick_get_data(out_re);
    gdouble *outimdata = gwy_brick_get_data(out_im);

    fftw_complex *buf1 = gwy_fftw_new_complex(n);
    fftw_complex *buf2 = gwy_fftw_new_complex(n);

    fftw_plan plan = fftw_plan_dft_3d(xres, yres, zres, buf1, buf2, FFTW_FORWARD, FFTW_ESTIMATE | FFTW_DESTROY_INPUT);

    for (i = 0; i < n; i++) {
        gwycreal(buf1[i]) = indata[i];
        gwycimag(buf1[i]) = 0.0;
    }
    fftw_execute(plan);

    for (i = 0; i < n; i++) {
        outredata[i] = gwycreal(buf2[i]);
        outimdata[i] = gwycimag(buf2[i]);
    }
 
    fftw_destroy_plan(plan);

    fftw_free(buf1);
    fftw_free(buf2);
}


/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
