/*
 *  $Id: xyz_tstep.c 26355 2024-05-21 08:23:55Z yeti-dn $
 *  Copyright (C) 2016-2023 David Necas (Yeti).
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/surface.h>
#include <libprocess/peaks.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-xyz.h>
#include <app/gwyapp.h>


#define RUN_MODES (GWY_RUN_INTERACTIVE)

enum {
    PREVIEW_WIDTH = 500,
    PREVIEW_HEIGHT = 300,
};

/* The same as in mark_disconn. */
typedef enum {
    FEATURES_POSITIVE = 1 << 0,
    FEATURES_NEGATIVE = 1 << 2,
    FEATURES_BOTH     = (FEATURES_POSITIVE | FEATURES_NEGATIVE),
} FeatureType;

enum {
    PARAM_STAT,
    PARAM_THRESHOLD_UP,
    PARAM_THRESHOLD_DOWN,
    PARAM_TYPE,
};

typedef enum {
    XYZ_STAT_X = 0,
    XYZ_STAT_Y = 1,
    XYZ_STAT_Z = 2,
} XYZStatType;

typedef struct {
    GwyParams *params;
    GwySurface *surface;
    GwySurface *result;
    gdouble *split;
    gdouble *step;
    gint nsplit;
    gint tmax;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table_display;
    GwyParamTable *table_options;
    GwyParamTable *table_type;
    GwyContainer *data;
    GwyGraphModel *gmodel;
    GwySelection *graph_selection;
} ModuleGUI;


static gboolean         module_register         (void);
static GwyParamDef*     define_module_params    (void);
static void             execute                 (ModuleArgs *args);
static GwyDialogOutcome run_gui                 (ModuleArgs *args);
static void             param_changed           (ModuleGUI *gui,
                                                 gint id);
static void             update_graph_curve      (ModuleGUI *gui);
static void             update_graph_selection  (ModuleGUI *gui);
static void             xyztstep                (GwyContainer *data,
                                                 GwyRunType runtype);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Detect and remove steps in time series XYZ data."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.0",
    "Petr Klapetek",
    "2025",
};

GWY_MODULE_QUERY2(module_info, xyz_tstep)

static gboolean
module_register(void)
{
    gwy_xyz_func_register("xyz_tstep",
                          (GwyXYZFunc)&xyztstep,
                          N_("/Remove Steps from T_ime Series..."),
                          NULL,
                          RUN_MODES,
                          GWY_MENU_FLAG_XYZ,
                          N_("Remove steps from XYZ data"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum stats[] = {
        { N_("X value"), XYZ_STAT_X, },
        { N_("Y value"), XYZ_STAT_Y, },
        { N_("Z value"), XYZ_STAT_Z, },
    };
    static const GwyEnum feature_types[] = {
        { N_("Positive"), FEATURES_POSITIVE, },
        { N_("Negative"), FEATURES_NEGATIVE, },
        { N_("Both"),     FEATURES_BOTH,     },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_xyz_func_current());
    gwy_param_def_add_gwyenum(paramdef, PARAM_STAT, "graph", _("Graph"),
                              stats, G_N_ELEMENTS(stats), XYZ_STAT_Z);
    gwy_param_def_add_percentage(paramdef, PARAM_THRESHOLD_UP, "threshold_up", _("Positive threshold"), 0.9);
    gwy_param_def_add_percentage(paramdef, PARAM_THRESHOLD_DOWN, "threshold_down", _("Negative threshold"), 0.9);
   gwy_param_def_add_gwyenum(paramdef, PARAM_TYPE, "type", _("Step direction"),
                             feature_types, G_N_ELEMENTS(feature_types), FEATURES_POSITIVE);

    return paramdef;
}

static void
xyztstep(GwyContainer *data, GwyRunType runtype)
{
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    ModuleArgs args;
    gint id, newid;
    const guchar *gradient;

    g_return_if_fail(runtype & RUN_MODES);

    gwy_app_data_browser_get_current(GWY_APP_SURFACE, &args.surface,
                                     GWY_APP_SURFACE_ID, &id,
                                     0);
    g_return_if_fail(GWY_IS_SURFACE(args.surface));

    args.params = gwy_params_new_from_settings(define_module_params());
    args.tmax = gwy_surface_get_npoints(args.surface);
    args.split = NULL;
    args.step = NULL;
    args.nsplit = 0;

    if (runtype == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }

    if (outcome == GWY_DIALOG_PROCEED)
        execute(&args);

    newid = gwy_app_data_browser_add_surface(args.result, data, TRUE);
    gwy_app_set_surface_title(data, newid, _("Steps removed"));
    if (gwy_container_gis_string(data, gwy_app_get_surface_palette_key_for_id(id), &gradient))
        gwy_container_set_const_string(data, gwy_app_get_surface_palette_key_for_id(newid), gradient);

    gwy_app_xyz_log_add(data, -1, newid, "xyz::xyz_tstep", NULL);

    g_object_unref(args.result);

end:
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args)
{
    GwyDialog *dialog;
    ModuleGUI gui;
    GwyParamTable *table;
    GtkWidget *graph, *area, *hbox;
    GwyGraphCurveModel *gcmodel;

    gui.dialog = gwy_dialog_new(_("Remove Steps from Time Series"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    gui.args = args;
    gui.gmodel = gwy_graph_model_new();
    g_object_set(gui.gmodel,
                 "label-visible", FALSE,
                 NULL);

    gcmodel = gwy_graph_curve_model_new();
    g_object_set(gcmodel, "mode", GWY_GRAPH_CURVE_LINE, NULL);
    gwy_graph_model_add_curve(gui.gmodel, gcmodel);
    g_object_unref(gcmodel);

    graph = gwy_graph_new(gui.gmodel);
    gtk_widget_set_size_request(graph, PREVIEW_WIDTH, PREVIEW_HEIGHT);
    gwy_dialog_add_content(dialog, graph, FALSE, FALSE, 0);
    gwy_graph_enable_user_input(GWY_GRAPH(graph), FALSE);
    gwy_graph_set_status(GWY_GRAPH(graph), GWY_GRAPH_STATUS_XLINES);
    area = gwy_graph_get_area(GWY_GRAPH(graph));
    gwy_graph_area_set_selection_editable(GWY_GRAPH_AREA(area), FALSE);
    gui.graph_selection = gwy_graph_area_get_selection(GWY_GRAPH_AREA(area), GWY_GRAPH_STATUS_XLINES);

    hbox = gwy_hbox_new(3);

    gui.table_display = table = gwy_param_table_new(args->params);
    gwy_param_table_append_radio(table, PARAM_STAT);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    gui.table_type = table = gwy_param_table_new(args->params);
    gwy_param_table_append_radio(table, PARAM_TYPE);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    gui.table_options = table = gwy_param_table_new(args->params);
    gwy_param_table_append_slider(table, PARAM_THRESHOLD_UP);
    gwy_param_table_append_slider(table, PARAM_THRESHOLD_DOWN);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);
    gwy_dialog_add_content(dialog, hbox, FALSE, TRUE, 0);

    g_signal_connect_swapped(gui.table_type, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.table_display, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.table_options, "param-changed", G_CALLBACK(param_changed), &gui);

    return gwy_dialog_run(dialog);
}

static gdouble
get_step(const gdouble *xdata, const gdouble *ydata, gint ndata, gdouble pos)
{
    gint ipos, i, j, ss = 2;
    gdouble sumleft, sumright;

    ipos = 0;
    for (i = 0; i < (ndata-1); i++) {
        if (xdata[i] <= pos && xdata[i+1] > pos) {
            ipos = i;
            break;
        }
    }

    if (ipos < (ss + 1) || ipos >= (ndata - ss - 1))
        return 0;

    sumleft = sumright = 0;
    for (j = 0; j < ss; j++) {
        sumleft += ydata[ipos - j - 1];
    }
    for (j = 0; j < ss; j++) {
        sumright += ydata[ipos + j + 1];
    }

    return (sumright - sumleft)/ss;
}


static void
get_split_points(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    GwyPeaks *analyser = gwy_peaks_new();
    GwyGraphCurveModel *gcmodel = gwy_graph_model_get_curve(gui->gmodel, 0);
    const gdouble *xdata = gwy_graph_curve_model_get_xdata(gcmodel);
    const gdouble *ydata = gwy_graph_curve_model_get_ydata(gcmodel);
    double threshold_up = gwy_params_get_double(args->params, PARAM_THRESHOLD_UP);
    double threshold_down = gwy_params_get_double(args->params, PARAM_THRESHOLD_DOWN);
    FeatureType type = gwy_params_get_enum(args->params, PARAM_TYPE);
    gint ndata = gwy_graph_curve_model_get_ndata(gcmodel);
    gint i, n, nup, ndown, nfup, nfdown;
    GwyDataLine *vals, *der;
    gdouble *vdata, *ddata, *buf;
    gdouble *upheight, *upposition, maxupheight, upthreshold;
    gdouble *downheight, *downposition, maxdownheight, downthreshold;
    gint *index = NULL;

    vals = gwy_data_line_new(ndata, ndata, FALSE);
    der = gwy_data_line_new(ndata, ndata, FALSE);
    vdata = gwy_data_line_get_data(vals);
    ddata = gwy_data_line_get_data(der);

    for (i = 0; i < ndata; i++)
        vdata[i] = ydata[i];

    for (i = 0; i < ndata; i++)
        ddata[i] = gwy_data_line_get_der(vals, i);

    upheight = NULL;
    upposition = NULL;
    downheight = NULL;
    downposition = NULL;
    nup = ndown = nfup = nfdown = 0;
    if (type == FEATURES_POSITIVE || type == FEATURES_BOTH) {
        nup = gwy_peaks_analyze(analyser, xdata, ddata, ndata, G_MAXUINT);
        upheight = g_new(gdouble, nup);
        upposition = g_new(gdouble, nup);

        gwy_peaks_get_quantity(analyser, GWY_PEAK_HEIGHT, upheight);
        gwy_peaks_get_quantity(analyser, GWY_PEAK_ABSCISSA, upposition);
        maxupheight = -G_MAXDOUBLE;
        for (i = 0; i < nup; i++) {
            if (maxupheight < upheight[i])
                maxupheight = upheight[i];
        }

        upthreshold = threshold_up*maxupheight;
        nfup = 0;
        for (i = 0; i < nup; i++) {
            if (upheight[i] > upthreshold) {
                nfup++;
            }
        }
    }

    if (type == FEATURES_NEGATIVE || type == FEATURES_BOTH) {
        gwy_data_line_invert(der, FALSE, TRUE);
        ndown = gwy_peaks_analyze(analyser, xdata, ddata, ndata, G_MAXUINT);
        downheight = g_new(gdouble, ndown);
        downposition = g_new(gdouble, ndown);

        gwy_peaks_get_quantity(analyser, GWY_PEAK_HEIGHT, downheight);
        gwy_peaks_get_quantity(analyser, GWY_PEAK_ABSCISSA, downposition);
        maxdownheight = -G_MAXDOUBLE;
        for (i = 0; i < ndown; i++) {
            if (maxdownheight < downheight[i])
                maxdownheight = downheight[i];
        }

        downthreshold = threshold_down*maxdownheight;
        nfdown = 0;
        for (i = 0; i < ndown; i++) {
            if (downheight[i] > downthreshold) {
                nfdown++;
            }
        }
    }

    if (args->split)
        g_free(args->split);
    if (args->step)
        g_free(args->step);

    args->split = g_new(gdouble, nfup + nfdown);
    args->step = g_new(gdouble, nfup + nfdown);
    n = 0;
    for (i = 0; i < nup; i++) {
        if (upheight[i] > upthreshold) {
            args->split[n] = upposition[i];
            args->step[n] = get_step(xdata, ydata, ndata, upposition[i]);
            n++;
        }
    }
    for (i = 0; i < ndown; i++) {
        if (downheight[i] > downthreshold) {
            args->split[n] = downposition[i];
            args->step[n] = get_step(xdata, ydata, ndata, downposition[i]);
            n++;
        }
    }

    if (nup*ndown > 0) {
        index = g_new(gint, n);
        for (i = 0; i < n; i++)
            index[i] = i;
        buf = g_new(gdouble, n);
        gwy_math_sort_with_index(n, args->split, index);
        for (i = 0; i < n; i++) {
            buf[i] = args->step[index[i]];
        }
        for (i = 0; i < n; i++)
            args->step[i] = buf[i];
        g_free(index);
        g_free(buf);
    }

    args->nsplit = n;

    gwy_peaks_free(analyser);
    gwy_object_unref(der);
    gwy_object_unref(vals);
    if (upheight)
        g_free(upheight);
    if (upposition)
        g_free(upposition);
    if (downheight)
        g_free(downheight);
    if (downposition)
        g_free(downposition);
}


static void
param_changed(ModuleGUI *gui, gint id)
{
    GwyParamTable *table = gui->table_options;
    GwyParams *params = gui->args->params;

    if (id < 0 || id == PARAM_STAT)
        update_graph_curve(gui);

    if (id < 0 || id == PARAM_STAT || id == PARAM_THRESHOLD_UP
        || id == PARAM_THRESHOLD_DOWN || id == PARAM_TYPE) {
        get_split_points(gui);
        update_graph_selection(gui);
    }

    if (id < 0 || id == PARAM_TYPE) {
        FeatureType type = gwy_params_get_enum(params, PARAM_TYPE);
        if (type == FEATURES_POSITIVE || type == FEATURES_BOTH)
            gwy_param_table_set_sensitive(table, PARAM_THRESHOLD_UP, TRUE);
        else
            gwy_param_table_set_sensitive(table, PARAM_THRESHOLD_UP, FALSE);

        if (type == FEATURES_NEGATIVE || type == FEATURES_BOTH)
            gwy_param_table_set_sensitive(table, PARAM_THRESHOLD_DOWN, TRUE);
        else
            gwy_param_table_set_sensitive(table, PARAM_THRESHOLD_DOWN, FALSE);
     }
}

static void
update_graph_selection(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    gdouble *sel;
    gint nsel = args->nsplit;
    gint i;

    gwy_selection_clear(gui->graph_selection);
    if (nsel < 1)
        return;

    gwy_selection_set_max_objects(gui->graph_selection, nsel);

    sel = g_new(gdouble, nsel);
    for (i = 0; i < nsel; i++) {
        sel[i] = args->split[i];
    }

    gwy_selection_set_data(gui->graph_selection, nsel, sel);
    g_free(sel);
}


static void
update_graph_curve(ModuleGUI *gui)
{
    GwyGraphCurveModel *gcmodel = gwy_graph_model_get_curve(gui->gmodel, 0);
    GwyDataLine *line;
    GwySurface *surface = gui->args->surface;
    const GwyXYZ *xyz;
    gdouble *linedata;
    gint k, n, m, ndiv = 1;
    gint stat = gwy_params_get_enum(gui->args->params, PARAM_STAT);

    if (gui->args->tmax > 5000000)
        ndiv = 10000;
    else if (gui->args->tmax > 500000)
        ndiv = 1000;
    else if (gui->args->tmax > 50000)
        ndiv = 100;
    else if (gui->args->tmax > 5000)
        ndiv = 10;

    line = gwy_data_line_new(gui->args->tmax/ndiv, gui->args->tmax, FALSE);
    linedata = gwy_data_line_get_data(line);
    xyz = gwy_surface_get_data_const(surface);

    m = 0;
    for (n = 0; n < gui->args->tmax/ndiv; n++) {
        linedata[n] = 0;
        if (stat == XYZ_STAT_X)
            for (k = 0; k < ndiv; k++)
                linedata[n] += xyz[m++].x;
        else if (stat == XYZ_STAT_Y)
            for (k = 0; k < ndiv; k++)
                linedata[n] += xyz[m++].y;
        else
            for (k = 0; k < ndiv; k++)
                linedata[n] += xyz[m++].z;

         if (ndiv > 1)
             linedata[n] /= ndiv;
    }

    gwy_graph_curve_model_set_data_from_dataline(gcmodel, line, 0, 0);

    g_object_unref(line);
}

static void
execute(ModuleArgs *args)
{
    GwySurface *surface = args->surface;
    GwySurface *result;
    const GwyXYZ *xyz;
    GwyXYZ *xyz_result;
    guint k, n, ns;
    gdouble zshift;

    xyz = gwy_surface_get_data_const(surface);
    n = gwy_surface_get_npoints(surface);

    args->result = result = gwy_surface_new_sized(n);
    xyz_result = gwy_surface_get_data(result);
    gwy_surface_copy_units(surface, result);

    ns = 0;
    zshift = 0;
    for (k = 0; k < n; k++) {
        if (k <= args->split[ns] && (k+1) > args->split[ns]) {
            zshift -= args->step[ns];
            ns++;
        }
        xyz_result[k].x = xyz[k].x;
        xyz_result[k].y = xyz[k].y;
        xyz_result[k].z = xyz[k].z + zshift;
    }
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
