/*
 *  $Id: app.c 28779 2025-11-04 17:00:29Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <stdarg.h>
#include <math.h>
#include <glib/gi18n-lib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "libgwyddion/gwyddion.h"
#include "libgwyui/gwyui.h"
#include "libgwyapp/gwyapp.h"

#include "libgwyapp/gwyappinternal.h"
#include "libgwyapp/sanity.h"

static GtkWidget *gwy_app_main_window = NULL;

static GwyTool* current_tool = NULL;

static gboolean gwy_app_main_window_save_position   (void);
static void     gwy_app_main_window_restore_position(void);
static gint64   monitor_config_checksum             (GtkWidget *widget);

/********************************************************************************************************************
 *                                                                                                                  *
 *     Stubs FIXME GTK3                                                                                             *
 *                                                                                                                  *
 ********************************************************************************************************************/

void
gwy_app_gl_material_editor(void)
{
}

/********************************************************************************************************************
 *                                                                                                                  *
 *     Main, toolbox                                                                                                *
 *                                                                                                                  *
 ********************************************************************************************************************/

/**
 * gwy_app_quit:
 *
 * Quits the application.
 *
 * This function may present a confirmation dialog to the user and it may let the application to continue running.  If
 * it quits the application, it performs some shutdown actions and then quits the Gtk+ main loop with gtk_main_quit().
 *
 * Returns: Always %TRUE to be usable as an event handler.  However, if the application is actually terminated, this
 *          function does not return.
 **/
gboolean
gwy_app_quit(void)
{
    gwy_debug("");

    //gwy_data_browser_shut_down();
    gwy_app_main_window_save_position();
    g_clear_object(&current_tool);
    /* XXX: EXIT-CLEAN-UP */
    gtk_widget_destroy(gwy_app_main_window);
    g_object_unref(gwy_app_sensitivity_get_group());

    gtk_main_quit();
    return TRUE;
}

static gboolean
gwy_app_main_window_save_position(void)
{
    gwy_app_save_window_position(GTK_WINDOW(gwy_app_main_window), "/app/toolbox", TRUE, TRUE);
    return FALSE;
}

static void
gwy_app_main_window_restore_position(void)
{
    gwy_app_restore_window_position(GTK_WINDOW(gwy_app_main_window), "/app/toolbox", FALSE);
}

/**
 * gwy_app_add_main_accel_group:
 * @window: A window.
 *
 * Adds main (global) application accelerator group to a window.
 *
 * This includes accelerators for terminating Gwyddion, opening files, etc.
 **/
void
gwy_app_add_main_accel_group(GtkWindow *window)
{
    GtkWidget *main_window;
    GtkAccelGroup *accel_group;

    g_return_if_fail(GTK_IS_WINDOW(window));
    main_window = gwy_app_main_window_get();
    if (!main_window)
        return;

    g_return_if_fail(GTK_IS_WINDOW(main_window));

    accel_group = GTK_ACCEL_GROUP(g_object_get_data(G_OBJECT(main_window), "accel_group"));
    if (accel_group)
        gtk_window_add_accel_group(window, accel_group);
}

/**
 * gwy_app_main_window_get:
 *
 * Returns Gwyddion main application window (toolbox).
 *
 * Returns: The Gwyddion toolbox.
 **/
GtkWidget*
gwy_app_main_window_get(void)
{
    return gwy_app_main_window;
}

/**
 * gwy_app_main_window_set:
 * @window: A window, presumably the Gwyddion toolbox.
 *
 * Sets the main application window.
 *
 * This function is probably only useful in Gwyddion itself and should be ignored.
 *
 * It needs to be called exactly once at Gwyddion startup. It restores the toolbox position and makes
 * gwy_app_main_window_get() work.
 **/
void
gwy_app_main_window_set(GtkWidget *window)
{
    g_return_if_fail(GTK_IS_WINDOW(window));
    if (gwy_app_main_window && window != gwy_app_main_window) {
        g_critical("The main window was already set to %p.", gwy_app_main_window);
        return;
    }

    gwy_app_main_window = window;
    gwy_app_main_window_restore_position();
    g_signal_connect(window, "delete-event", G_CALLBACK(gwy_app_main_window_save_position), NULL);
    g_signal_connect(window, "show", G_CALLBACK(gwy_app_main_window_restore_position), NULL);
}

/********************************************************************************************************************
 *                                                                                                                  *
 *     Miscellaneous                                                                                                *
 *                                                                                                                  *
 ********************************************************************************************************************/

/**
 * gwy_app_switch_tool:
 * @toolname: Tool name, that is type name of the tool object in the GLib
 *            type system.  This can be for instance "GwyToolGrainMeasure".
 *
 * Switches the current tool to given tool.
 *
 * If the tool is already active it is shown when hidden and hidden when
 * visible.
 **/
void
gwy_app_switch_tool(const gchar *toolname)
{
    GwyContainer *settings;
    GwyTool *newtool;
    GwyDataView *data_view;
    GType type;
    gboolean do_restore = FALSE;

    gwy_debug("%s", toolname ? toolname : "NONE");
    type = g_type_from_name(toolname);
    g_return_if_fail(type);

    gwy_data_browser_get_current(GWY_APP_DATA_VIEW, &data_view, 0);
    if (current_tool && type == G_TYPE_FROM_INSTANCE(current_tool)) {
        if (!gwy_tool_is_visible(current_tool))
            gwy_tool_show(current_tool);
        else
            gwy_tool_hide(current_tool);
        return;
    }

    g_clear_object(&current_tool);
    newtool = (GwyTool*)g_object_new(type, NULL);
    current_tool = newtool;
    g_return_if_fail(GWY_IS_TOOL(newtool));

    settings = gwy_app_settings_get();
    gwy_container_gis_boolean_by_name(settings, "/app/restore-tool-position", &do_restore);
    if (do_restore)
        gwy_tool_restore_screen_position(newtool);

    if (data_view) {
        GwySpectra *spectra;

        gwy_tool_data_switched(current_tool, data_view);
        gwy_data_browser_get_current(GWY_APP_SPECTRA, &spectra, 0);
        gwy_tool_spectra_switched(current_tool, spectra);
        gwy_tool_show(current_tool);
    }
}

/**
 * gwy_app_current_tool_name:
 *
 * Obtains the name of currently active tool.
 *
 * See gwy_app_switch_tool() for the name description.
 *
 * In some rare circumstances, this function can return %NULL because no tool is active.  This includes program
 * startup and shutdown and during the construction of a new #GwyTool object while switching tools.  Also, %NULL is
 * typically returned outside Gwyddion in programs just using the libraries.
 *
 * Returns: The tool name.
 **/
const gchar*
gwy_app_current_tool_name(void)
{
    if (!current_tool)
        return NULL;

    return G_OBJECT_TYPE_NAME(current_tool);
}

/**
 * gwy_app_current_tool:
 *
 * Obtains the currently active tool object.
 *
 * In some rare circumstances, this function can return %NULL because no tool is active.  This includes program
 * startup and shutdown and during the construction of a new #GwyTool object while switching tools.  Also, %NULL is
 * typically returned outside Gwyddion in programs just using the libraries.
 *
 * Returns: The current tool object.
 **/
GwyTool*
gwy_app_current_tool(void)
{
    return current_tool;
}

/**
 * gwy_app_save_window_position:
 * @window: A window to save position of.
 * @prefix: Unique prefix in settings to store the information under.
 * @position: %TRUE to save position information.
 * @size: %TRUE to save size information.
 *
 * Saves position and/or size of a window to settings.
 *
 * Some sanity checks are included, therefore if window position and/or size is too suspicious, it is not saved.
 **/
void
gwy_app_save_window_position(GtkWindow *window,
                             const gchar *prefix,
                             gboolean position,
                             gboolean size)
{
    GwyContainer *settings;
    gint x, y, w, h, scw, sch;
    guint len;
    gchar *key;
    gint64 mconf;

    g_return_if_fail(GTK_IS_WINDOW(window));
    g_return_if_fail(prefix);
    if (!(position || size))
        return;

    len = strlen(prefix);
    key = g_new(gchar, len + sizeof("/position/height"));  /* The longest suffix */
    strcpy(key, prefix);

    settings = gwy_app_settings_get();
    scw = gwy_get_screen_width(GTK_WIDGET(window));
    sch = gwy_get_screen_height(GTK_WIDGET(window));
    /* FIXME: read the gtk_window_get_position() docs about how this is a broken approach. But it is better than
     * nothing. We should detect broken environments like Wayland which do not support programmatic positioning of
     * windows at all and avoid saving the poisition. On the other hand, gtk_window_move() should also be noop then
     * so we may not care. */
    if (position) {
        mconf = monitor_config_checksum(GTK_WIDGET(window));
        gtk_window_get_position(window, &x, &y);
        if (x >= 0 && y >= 0 && x+1 < scw && y+1 < sch) {
            strcpy(key + len, "/position/x");
            gwy_container_set_int32_by_name(settings, key, x);
            strcpy(key + len, "/position/y");
            gwy_container_set_int32_by_name(settings, key, y);
            strcpy(key + len, "/position/mconf");
            gwy_container_set_int64_by_name(settings, key, mconf);
        }
    }
    if (size) {
        gtk_window_get_size(window, &w, &h);
        if (w > 1 && h > 1) {
            strcpy(key + len, "/position/width");
            gwy_container_set_int32_by_name(settings, key, w);
            strcpy(key + len, "/position/height");
            gwy_container_set_int32_by_name(settings, key, h);
        }
    }
    g_free(key);
}

typedef struct {
    gulong hid;
    gint x;
    gint y;
} WindowPos;

static void
restore_pos(GtkWindow *window, WindowPos *wp)
{
    gwy_debug("moving window %p(%s) to (%d, %d)", window, gtk_window_get_title(window), wp->x, wp->y);
    GdkWindow *gdkwin = gtk_widget_get_window(GTK_WIDGET(window));
    gdk_window_move(gdkwin, wp->x, wp->y);
    g_signal_handler_disconnect(window, wp->hid);
    g_free(wp);
}

/**
 * gwy_app_restore_window_position:
 * @window: A window to restore position of.
 * @prefix: Unique prefix in settings to get the information from (the same as
 *          in gwy_app_save_window_position()).
 * @grow_only: %TRUE to only attempt set the window default size bigger than it
 *              requests, never smaller.
 *
 * Restores a window position and/or size from settings.
 *
 * Unlike gwy_app_save_window_position(), this function has no @position and @size arguments, it simply restores all
 * attributes that were saved.
 *
 * Note to restore position (not size) it should be called twice for each window to accommodate sloppy window
 * managers: once before the window is shown, second time immediately after showing the window.
 *
 * Some sanity checks are included, therefore if saved window position and/or size is too suspicious, it is not
 * restored.
 **/
void
gwy_app_restore_window_position(GtkWindow *window,
                                const gchar *prefix,
                                gboolean grow_only)
{
    GwyContainer *settings;
    GtkRequisition req;
    gint x, y, w, h, scw, sch;
    guint len;
    gchar *key;
    gint64 mconf, savedmconf = 0;

    g_return_if_fail(GTK_IS_WINDOW(window));
    g_return_if_fail(prefix);

    len = strlen(prefix);
    key = g_new(gchar, len + sizeof("/position/height"));  /* The longest suffix */
    strcpy(key, prefix);

    settings = gwy_app_settings_get();
    mconf = monitor_config_checksum(GTK_WIDGET(window));
    scw = gwy_get_screen_width(GTK_WIDGET(window));
    sch = gwy_get_screen_height(GTK_WIDGET(window));
    x = y = w = h = -1;
    strcpy(key + len, "/position/x");
    gwy_container_gis_int32_by_name(settings, key, &x);
    strcpy(key + len, "/position/y");
    gwy_container_gis_int32_by_name(settings, key, &y);
    strcpy(key + len, "/position/mconf");
    gwy_container_gis_int64_by_name(settings, key, &savedmconf);
    strcpy(key + len, "/position/width");
    gwy_container_gis_int32_by_name(settings, key, &w);
    strcpy(key + len, "/position/height");
    gwy_container_gis_int32_by_name(settings, key, &h);
    gwy_debug("mcof = %lx, savedmconf = %lx", mconf, savedmconf);
    if (mconf == savedmconf && x >= 0 && y >= 0 && x+1 < scw && y+1 < sch) {
        gtk_window_move(window, x, y);
        WindowPos *wp = g_new(WindowPos, 1);
        wp->x = x;
        wp->y = y;
        wp->hid = g_signal_connect_after(window, "map", G_CALLBACK(restore_pos), wp);
        /* FIXME: We lost wp if window is never mapped. */
    }
    if (w > 1 && h > 1) {
        if (grow_only) {
            gtk_widget_get_preferred_size(GTK_WIDGET(window), NULL, &req);
            w = MAX(w, req.width);
            h = MAX(h, req.height);
        }
        gtk_window_set_default_size(window, w, h);
    }

    g_free(key);
}

/* We probably do not have to this repeatedly but GDK has the monitor properties tabulated so it is just a few extra
 * function calls... */
static gint64
monitor_config_checksum(GtkWidget *widget)
{
    static GChecksum *checksummer = NULL;

    if (!widget)
        return 0;

    GdkDisplay *display = gtk_widget_get_display(widget);
    if (!display)
        display = gdk_display_get_default();
    if (!display)
        return 0;

    gint n = gdk_display_get_n_monitors(display);
    if (!n)
        return 0;

    GdkRectangle *mdims, mdim0;
    if (n == 1)
        mdims = &mdim0;
    else
        mdims = g_new(GdkRectangle, n);

    for (gint i = 0; i < n; i++) {
        GdkMonitor *monitor = gdk_display_get_monitor(display, i);
        gdk_monitor_get_geometry(monitor, mdims + i);
    }

    if (!checksummer)
        checksummer = g_checksum_new(G_CHECKSUM_MD5);
    else
        g_checksum_reset(checksummer);

    g_checksum_update(checksummer, (guchar*)mdims, n*sizeof(GdkRectangle));

    guchar digest[16];
    gsize len = G_N_ELEMENTS(digest);
    gwy_clear(digest, len);
    g_checksum_get_digest(checksummer, digest, &len);
    if (n != 1)
        g_free(mdims);

    guint64 r = digest[0] & 0x7f;
    for (gint i = 1; i < 8; i++) {
        r <<= 8ul;
        r |= digest[i];
    }

    return r;
}

gint
_gwy_app_get_n_recent_files(void)
{
    return 10;
}

/**
 * SECTION: app
 * @title: app
 * @short_description: Core application interface, window management
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
