/*
 *  $Id: selection-lattice.c 27855 2025-04-18 09:57:38Z yeti-dn $
 *  Copyright (C) 2024 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"

#include "libgwyddion/macros.h"
#include "libgwyddion/math.h"
#include "libgwyddion/selection-lattice.h"

enum {
    OBJECT_SIZE = 4
};

struct _GwySelectionLatticePrivate {
    gint dummy;
};

static gboolean crop_object(GwySelection *selection,
                            gint i,
                            gpointer user_data);
static void     crop       (GwySelection *selection,
                            gdouble xmin,
                            gdouble ymin,
                            gdouble xmax,
                            gdouble ymax);

static GObjectClass *parent_class = NULL;

G_DEFINE_TYPE(GwySelectionLattice, gwy_selection_lattice, GWY_TYPE_SELECTION)

static void
gwy_selection_lattice_class_init(GwySelectionLatticeClass *klass)
{
    GwySelectionClass *sel_class = GWY_SELECTION_CLASS(klass);

    parent_class = gwy_selection_lattice_parent_class;

    sel_class->object_size = OBJECT_SIZE;
    sel_class->crop = crop;
    /* Not implementing move() means gwy_selection_move() is a no-op, which is what we need. */
}

static void
gwy_selection_lattice_init(GwySelectionLattice *selection)
{
    G_GNUC_UNUSED GwySelectionLatticePrivate *priv;

    priv = selection->priv = gwy_selection_lattice_get_instance_private(selection);
}

/* This is kind of odd because lattice conceptually is not at any particular location in the plane. But we say the
 * vector ends should not be outside of the image rectangle if drawn from the centre. So they must be at most half the
 * its size. */
static gboolean
crop_object(GwySelection *selection, gint i, gpointer user_data)
{
    const gdouble *minmax = (const gdouble*)user_data;
    gdouble xy[OBJECT_SIZE];

    gwy_selection_get_object(selection, i, xy);
    return (2*fabs(xy[0]) <= minmax[2] - minmax[0]
            && 2*fabs(xy[1]) <= minmax[3] - minmax[1]
            && 2*fabs(xy[2]) <= minmax[2] - minmax[0]
            && 2*fabs(xy[3]) <= minmax[3] - minmax[1]);
}

static void
crop(GwySelection *selection, gdouble xmin, gdouble ymin, gdouble xmax, gdouble ymax)
{
    gdouble minmax[4] = { xmin, ymin, xmax, ymax };

    gwy_selection_filter(selection, crop_object, minmax);
}

/**
 * gwy_selection_lattice_new:
 *
 * Creates a new lattice selection.
 *
 * Returns: A new selection object.
 **/
GwySelection*
gwy_selection_lattice_new(void)
{
    return (GwySelection*)g_object_new(GWY_TYPE_SELECTION_LATTICE, NULL);
}

/************************** Documentation ****************************/

/**
 * SECTION:selection-lattice
 * @title: GwySelectionLattice
 * @short_description: Lattice selection
 *
 * #GwySelectionLattice represents lattices. Selection data consist of quadruples of coordinates (x₀, y₀, x₁, y₁),
 * where 0 and 1 denote the two lattice vectors.
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
