/*
 *  $Id: color-button.c 27952 2025-05-09 16:41:38Z yeti-dn $
 *  Copyright (C) 2024 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"

#include <gtk/gtk.h>

#include "libgwyddion/macros.h"
#include "libgwyddion/math.h"

#include "libgwyui/color-swatch.h"
#include "libgwyui/color-button.h"

struct _GwyColorButtonPrivate {
    GtkWidget *swatch;
};

static GtkWidgetClass *parent_class = NULL;

G_DEFINE_TYPE_WITH_CODE(GwyColorButton, gwy_color_button, GTK_TYPE_BUTTON,
                        G_ADD_PRIVATE(GwyColorButton))

static void
gwy_color_button_class_init(G_GNUC_UNUSED GwyColorButtonClass *klass)
{
    parent_class = gwy_color_button_parent_class;
}

static void
gwy_color_button_init(GwyColorButton *button)
{
    GwyColorButtonPrivate *priv;

    button->priv = priv = gwy_color_button_get_instance_private(button);
    priv->swatch = gwy_color_swatch_new();
    gtk_container_add(GTK_CONTAINER(button), priv->swatch);
    gtk_container_set_border_width(GTK_CONTAINER(button), 1);
    gtk_widget_set_hexpand(priv->swatch, TRUE);
    gtk_widget_set_vexpand(priv->swatch, TRUE);
}

/**
 * gwy_color_button_new:
 *
 * Creates a new colour button.
 *
 * The button contains a swatch representing the current selected colour.
 *
 * The button child must not be changed.
 *
 * Returns: A new colour button.
 **/
GtkWidget*
gwy_color_button_new(void)
{
    return g_object_new(GWY_TYPE_COLOR_BUTTON, NULL);
}

/**
 * gwy_color_button_set_color:
 * @button: A colour button.
 * @color: The new colour as #GwyRGBA.
 *
 * Sets the current color of a colour button.
 **/
void
gwy_color_button_set_color(GwyColorButton *button,
                           const GwyRGBA *color)
{
    g_return_if_fail(GWY_IS_COLOR_BUTTON(button));
    gwy_color_swatch_set_color(GWY_COLOR_SWATCH(button->priv->swatch), color);
}

/**
 * gwy_color_button_get_color:
 * @button: A colour button.
 * @color: Location to fill in with the current colour.
 *
 * Obtains the current colour of a colour button.
 **/
void
gwy_color_button_get_color(GwyColorButton *button,
                           GwyRGBA *color)
{
    g_return_if_fail(GWY_IS_COLOR_BUTTON(button));
    gwy_color_swatch_get_color(GWY_COLOR_SWATCH(button->priv->swatch), color);
}

/**
 * gwy_color_button_set_use_alpha:
 * @button: A colour button.
 * @use_alpha: %TRUE if colour swatch should visualise the alpha channel, %FALSE to ignore it.
 *
 * Sets whether or not the color button should use the alpha channel.
 **/
void
gwy_color_button_set_use_alpha(GwyColorButton *button,
                               gboolean use_alpha)
{
    g_return_if_fail(GWY_IS_COLOR_BUTTON(button));
    gwy_color_swatch_set_use_alpha(GWY_COLOR_SWATCH(button->priv->swatch), use_alpha);
}

/**
 * gwy_color_button_get_use_alpha:
 * @button: A colour button.
 *
 * Reports whether the colour button uses the alpha channel.
 **/
gboolean
gwy_color_button_get_use_alpha(GwyColorButton *button)
{
    g_return_val_if_fail(GWY_IS_COLOR_BUTTON(button), FALSE);
    return gwy_color_swatch_get_use_alpha(GWY_COLOR_SWATCH(button->priv->swatch));
}

/************************** Documentation ****************************/

/**
 * SECTION:color-button
 * @title: GwyColorButton
 * @short_description: A colour displaying button
 *
 * A button which displays a colour swatch.
 *
 * It does not have its own color selection dialog however, it is just an ordinary button displaying a colour.  You
 * need to connect some callback to its "clicked" signal.  Also it uses #GwyRGBA for colour representation.
 *
 * This widget is just a thin convenience wrapper to construct a button with a #GwyColorSwatch inside. It is kept
 * mainly for compatiblitity.
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
