/*
 *  $Id: graph-axis-dialog.c 28348 2025-08-11 16:29:06Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include "libgwyddion/macros.h"
#include "libgwyddion/math.h"
#include "libgwyui/utils.h"
#include "libgwyui/sci-text.h"
#include "libgwyui/graph-axis-dialog.h"

/* FIXME: Reverse dependence, change sanity.h to normal API and get rid of parts of it, as needed. */
#include "libgwyapp/sanity.h"

static gboolean deleted     (GtkWidget *widget,
                             GdkEventAny *event);
static void     response    (GtkDialog *dialog,
                             gint response_id);
static void     label_edited(GwyGraphAxisDialog *dialog,
                             GwySciText *sci_text);
static void     label_set   (GwyGraphAxisDialog *dialog,
                             GwyGraphAxis *axis);

G_DEFINE_TYPE(GwyGraphAxisDialog, _gwy_graph_axis_dialog, GTK_TYPE_DIALOG)

static void
_gwy_graph_axis_dialog_class_init(GwyGraphAxisDialogClass *klass)
{
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);
    GtkDialogClass *dialog_class = GTK_DIALOG_CLASS(klass);

    widget_class->delete_event = deleted;
    dialog_class->response = response;
}

static void
_gwy_graph_axis_dialog_init(GwyGraphAxisDialog *dialog)
{
    gtk_window_set_title(GTK_WINDOW(dialog), _("Axis Properties"));
    gwy_add_button_to_dialog(GTK_DIALOG(dialog), GWY_STOCK_CLOSE, GWY_ICON_GTK_CLOSE, GTK_RESPONSE_CLOSE);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_CLOSE);

    GtkGrid *grid = GTK_GRID(gtk_grid_new());
    gtk_grid_set_row_spacing(grid, 2);
    gtk_grid_set_column_spacing(grid, 6);
    gtk_container_set_border_width(GTK_CONTAINER(grid), 4);

    GtkWidget *vbox = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
    gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(grid), FALSE, TRUE, 0);

    gint row = 0;
    gtk_grid_attach(grid, gwy_label_new_header(_("Label Text")), 0, row++, 1, 1);

    dialog->sci_text = gwy_sci_text_new();
    gtk_grid_attach(grid, dialog->sci_text, 0, row++, 4, 1);

    gtk_widget_show_all(vbox);
}

static gboolean
deleted(GtkWidget *widget, G_GNUC_UNUSED GdkEventAny *event)
{
    gtk_widget_hide(widget);
    return TRUE;
}

static void
response(GtkDialog *dialog, gint response_id)
{
    if (response_id == GTK_RESPONSE_CLOSE)
        gtk_widget_hide(GTK_WIDGET(dialog));
}

/**
 * _gwy_graph_axis_dialog_new:
 * @axis: The axis to create dialog for,
 *
 * Creates a new axis dialog.
 *
 * Returns: A new axis dialog as a #GtkWidget.
 **/
GtkWidget*
_gwy_graph_axis_dialog_new(GwyGraphAxis *axis)
{
    GwyGraphAxisDialog *dialog = g_object_new(GWY_TYPE_GRAPH_AXIS_DIALOG, NULL);
    g_return_val_if_fail(GWY_IS_GRAPH_AXIS(axis), (GtkWidget*)dialog);

    dialog->axis = axis;
    gwy_sci_text_set_markup(GWY_SCI_TEXT(dialog->sci_text), gwy_graph_axis_get_label(axis));
    g_signal_connect_swapped(dialog->sci_text, "edited", G_CALLBACK(label_edited), dialog);
    g_signal_connect_swapped(axis, "notify::label", G_CALLBACK(label_set), dialog);

    return (GtkWidget*)dialog;
}

static void
label_edited(GwyGraphAxisDialog *dialog, GwySciText *sci_text)
{
    if (dialog->recursing)
        return;

    gchar *text = gwy_sci_text_get_markup(sci_text);
    dialog->recursing++;
    gwy_graph_axis_set_label(dialog->axis, text);
    dialog->recursing--;
    g_free(text);
}

static void
label_set(GwyGraphAxisDialog *dialog, GwyGraphAxis *axis)
{
    if (dialog->recursing)
        return;

    dialog->recursing++;
    gwy_sci_text_set_markup(GWY_SCI_TEXT(dialog->sci_text), gwy_graph_axis_get_label(axis));
    dialog->recursing--;
}

/*
 * SECTION: graph-axis-dialog
 * @title: GwyGraphAxisDialog
 * @short_description: Axis properties dialog
 *
 * #GwyGraphAxisDialog is used for setting the text properties of the axis. It is used namely with #GwyGraphAxis.
 **/

/* vim: set cin et columns=120 tw=118 ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
