/*
 *  $Id: widget-impl-utils.c 27952 2025-05-09 16:41:38Z yeti-dn $
 *  Copyright (C) 2024 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"

#include "libgwyddion/macros.h"
#include "libgwyddion/utils.h"

#include "libgwyui/widget-impl-utils.h"

GdkWindow*
gwy_create_widget_input_window(GtkWidget *widget, GdkEventMask event_mask)
{
    g_return_val_if_fail(GTK_IS_WIDGET(widget), NULL);

    GdkRectangle allocation;
    gtk_widget_get_allocation(widget, &allocation);
    GdkWindowAttr attributes = {
        .x = allocation.x,
        .y = allocation.y,
        .width = allocation.width,
        .height = allocation.height,
        .window_type = GDK_WINDOW_CHILD,
        .wclass = GDK_INPUT_ONLY,
        .event_mask = gtk_widget_get_events(widget) | event_mask,
    };

    gint attributes_mask = GDK_WA_X | GDK_WA_Y;
    GdkWindow *event_window = gdk_window_new(gtk_widget_get_parent_window(widget), &attributes, attributes_mask);
    gtk_widget_register_window(widget, event_window);

    /* Force widget to listen to relevant events.  Works around oxygen misdetection of ‘empty’ areas.  See KDE bug
     * #317292.
     * FIXME: Still necessary? */
    gtk_widget_add_events(widget, attributes.event_mask);

    return event_window;
}

void
gwy_destroy_widget_input_window(GtkWidget *widget,
                                GdkWindow **window)
{
    g_return_if_fail(GTK_IS_WIDGET(widget));
    g_return_if_fail(window);
    if (!*window)
        return;

    gtk_widget_unregister_window(widget, *window);
    gdk_window_destroy(*window);
    *window = NULL;
}

/* We might also add an out-argument for the state, but in general we should only ever draw with the current state. */
GtkStyleContext*
gwy_setup_widget_foreground_drawing(GtkWidget *widget,
                                    cairo_t *cr)
{
    GtkStyleContext *context = gtk_widget_get_style_context(widget);
    GtkStateFlags state = gtk_style_context_get_state(context);
    GdkRGBA color;
    gtk_style_context_get_color(context, state, &color);
    gdk_cairo_set_source_rgba(cr, &color);

    return context;
}

void
gwy_switch_window_cursor(GdkWindow *window,
                         const gchar **name,
                         const gchar *newname)
{
    if (!window)
        return;

    g_return_if_fail(GDK_IS_WINDOW(window));

    if (!newname) {
        if (*name) {
            *name = NULL;
            gdk_window_set_cursor(window, NULL);
        }
        return;
    }

    if (*name && gwy_strequal(newname, *name))
        return;

    GdkDisplay *display = gdk_window_get_display(window);
    if (!display) {
        g_warning("Trying to set cursor for a window with no display.");
        *name = NULL;
        gdk_window_set_cursor(window, NULL);
    }

    GdkCursor *cursor = gdk_cursor_new_from_name(display, newname);
    gdk_window_set_cursor(window, cursor);
    g_object_unref(cursor);
    *name = newname;
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
