/*
 *  $Id: witec-asc.c 28789 2025-11-04 17:14:03Z yeti-dn $
 *  Copyright (C) 2011-2025 David Necas (Yeti).
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
 * [FILE-MAGIC-FREEDESKTOP]
 * <mime-type type="application/x-witec-ascii-export">
 *   <comment>WITec ASCII data</comment>
 *   <magic priority="80">
 *     <match type="string" offset="0" value="//Exported ASCII-File"/>
 *   </magic>
 *   <glob pattern="*.asc"/>
 *   <glob pattern="*.ASC"/>
 * </mime-type>
 **/

/**
 * [FILE-MAGIC-FILEMAGIC]
 * # WITec ASCII export
 * 0 string //Exported\ ASCII-File WITec exported text data
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * WITec ASCII export
 * .dat
 * Read
 **/

#include "config.h"
#include <glib/gi18n-lib.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <gwy.h>

#include "err.h"

#define MAGIC "//Exported ASCII-File"
#define MAGIC_SIZE (sizeof(MAGIC)-1)
#define EXTENSION ".dat"

static gboolean module_register(void);
static gint     detect_file    (const GwyFileDetectInfo *fileinfo,
                                gboolean only_name);
static GwyFile* load_file      (const gchar *filename,
                                GwyRunModeFlags mode,
                                GError **error);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Imports WITec ASCII export files."),
    "Yeti <yeti@gwyddion.net>",
    "0.4",
    "David Nečas (Yeti)",
    "2011",
};

GWY_MODULE_QUERY2(module_info, witec_asc)

static gboolean
module_register(void)
{
    gwy_file_func_register("witec-asc",
                           N_("WITec ASCII files (.dat)"),
                           detect_file, load_file, NULL, NULL);

    return TRUE;
}

static gint
detect_file(const GwyFileDetectInfo *fileinfo, gboolean only_name)
{
    if (only_name)
        return g_str_has_suffix(fileinfo->name_lowercase, EXTENSION) ? 10 : 0;

    if (fileinfo->file_size > MAGIC_SIZE
        && memcmp(fileinfo->head, MAGIC, MAGIC_SIZE) == 0)
        return 100;

    return 0;
}

static gboolean
header_error(G_GNUC_UNUSED const GwyTextHeaderContext *context,
             GError *error,
             G_GNUC_UNUSED gpointer user_data)
{
    return error->code == GWY_TEXT_HEADER_ERROR_TERMINATOR;
}

static void
header_end(G_GNUC_UNUSED const GwyTextHeaderContext *context,
           gsize length,
           gpointer user_data)
{
    gchar **pp = (gchar**)user_data;

    *pp += length;
}

static GwyFile*
load_file(const gchar *filename,
          G_GNUC_UNUSED GwyRunModeFlags mode,
          GError **error)
{
    GwyFile *file = NULL;
    GwyField *dfield = NULL;
    GwyTextHeaderParser parser;
    GwyUnit *xyunit = NULL, *zunit = NULL;
    gint power10xy, power10z;
    gchar *line, *p, *value, *title, *buffer = NULL;
    GHashTable *hash = NULL;
    gsize size;
    GError *err = NULL;
    gdouble xreal, yreal, q;
    gint i, xres, yres;
    gdouble *data;

    if (!g_file_get_contents(filename, &buffer, &size, &err)) {
        err_GET_FILE_CONTENTS(error, &err);
        goto fail;
    }

    p = buffer;
    line = gwy_str_next_line(&p);
    if (!gwy_strequal(line, MAGIC)) {
        err_FILE_TYPE(error, "WITec ASCII export");
        goto fail;
    }

    line = gwy_str_next_line(&p);
    if (!line) {
        err_TRUNCATED_HEADER(error);
        goto fail;
    }
    g_strstrip(line);
    if (!gwy_strequal(line, "[Header]")) {
        g_set_error(error, GWY_MODULE_FILE_ERROR, GWY_MODULE_FILE_ERROR_DATA,
                    _("Expected header start marker ‘%s’ but found ‘%s’."), "[Header]", line);
        goto fail;
    }

    gwy_clear1(parser);
    parser.key_value_separator = "=";
    parser.terminator = "[Data]";
    parser.error = &header_error;
    parser.end = &header_end;
    if (!(hash = gwy_text_header_parse(p, &parser, &p, &err))) {
        g_propagate_error(error, err);
        goto fail;
    }
    if (!require_keys(hash, error,
                      "PointsPerLine", "LinesPerImage", "ScanUnit", "ScanWidth", "ScanHeight", "DataUnit", NULL))
        goto fail;

    xres = atoi(g_hash_table_lookup(hash, "PointsPerLine"));
    yres = atoi(g_hash_table_lookup(hash, "LinesPerImage"));
    if (err_DIMENSION(error, xres) || err_DIMENSION(error, yres))
        goto fail;

    xyunit = gwy_unit_new_parse(g_hash_table_lookup(hash, "ScanUnit"), &power10xy);
    zunit = gwy_unit_new_parse(g_hash_table_lookup(hash, "DataUnit"), &power10z);
    xreal = g_ascii_strtod(g_hash_table_lookup(hash, "ScanWidth"), NULL);
    yreal = g_ascii_strtod(g_hash_table_lookup(hash, "ScanHeight"), NULL);
    sanitise_real_size(&xreal, "x size");
    sanitise_real_size(&yreal, "y size");
    xreal *= gwy_exp10(power10xy);
    yreal *= gwy_exp10(power10xy);

    dfield = gwy_field_new(xres, yres, xreal, yreal, FALSE);

    gwy_unit_assign(gwy_field_get_unit_xy(dfield), xyunit);
    gwy_unit_assign(gwy_field_get_unit_z(dfield), zunit);
    q = gwy_exp10(power10z);

    data = gwy_field_get_data(dfield);
    value = p;
    for (i = 0; i < xres*yres; i++) {
        data[i] = q*g_ascii_strtod(value, &p);
        if (p == value && (!*p || g_ascii_isspace(*p))) {
            g_set_error(error, GWY_MODULE_FILE_ERROR, GWY_MODULE_FILE_ERROR_DATA,
                        _("End of file reached when reading sample #%d of %d"), i, xres*yres);
            goto fail;
        }
        if (p == value) {
            g_set_error(error, GWY_MODULE_FILE_ERROR, GWY_MODULE_FILE_ERROR_DATA,
                        _("Malformed data encountered when reading sample #%d of %d"), i, xres*yres);
            goto fail;
        }
        value = p;
    }

    file = gwy_file_new_in_construction();

    gwy_file_set_image(file, 0, dfield);

    title = g_hash_table_lookup(hash, "ImageName");
    if (title) {
        guint len = strlen(title);
        if (title[0] == '"' && title[len-1] == '"') {
            title[len-1] = '\0';
            title++;
        }
        gwy_file_set_title(file, GWY_FILE_IMAGE, 0, title, FALSE);
    }
    else
        gwy_image_title_fall_back(file, 0);

    gwy_check_nonsquare_image(file, 0);
    gwy_log_add_import(file, GWY_FILE_IMAGE, 0, NULL, filename);

fail:
    g_free(buffer);
    g_clear_object(&dfield);
    g_clear_object(&xyunit);
    g_clear_object(&zunit);
    if (hash)
        g_hash_table_destroy(hash);

    return file;
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
