/*
 *  $Id: presentationops.c 28860 2025-11-14 12:59:38Z yeti-dn $
 *  Copyright (C) 2003-2021 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <gwy.h>

#define RUN_MODES GWY_RUN_IMMEDIATE
#define ATTACH_RUN_MODES GWY_RUN_INTERACTIVE

enum {
    PARAM_SOURCE,
};

typedef struct {
    GwyParams *params;
    GwyField *target;
} ModuleArgs;

static gboolean         module_register      (void);
static GwyParamDef*     define_attach_params (void);
static void             presentation_remove  (GwyFile *data,
                                              GwyRunModeFlags mode);
static void             presentation_extract (GwyFile *data,
                                              GwyRunModeFlags mode);
static void             presentation_logscale(GwyFile *data,
                                              GwyRunModeFlags mode);
static void             presentation_attach  (GwyFile *data,
                                              GwyRunModeFlags mode);
static GwyDialogOutcome run_attach_gui       (ModuleArgs *args);
static gboolean         attach_source_filter (GwyFile *source,
                                              gint id,
                                              gpointer user_data);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Basic operations with presentation: extraction, removal."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, presentationops)

static GwyParamDef*
define_attach_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_process_func_current());
    gwy_param_def_add_image_id(paramdef, PARAM_SOURCE, "source", _("_Data to attach"));
    return paramdef;
}

static gboolean
module_register(void)
{
    gwy_process_func_register("presentation_remove",
                              &presentation_remove,
                              N_("/_Presentation/_Remove Presentation"),
                              NULL,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_SHOW | GWY_MENU_FLAG_IMAGE,
                              N_("Remove presentation from data"));
    gwy_process_func_register("presentation_extract",
                              &presentation_extract,
                              N_("/_Presentation/E_xtract Presentation"),
                              NULL,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_SHOW | GWY_MENU_FLAG_IMAGE,
                              N_("Extract presentation to a new image"));
    gwy_process_func_register("presentation_attach",
                              &presentation_attach,
                              N_("/_Presentation/_Attach Presentation..."),
                              NULL,
                              ATTACH_RUN_MODES,
                              GWY_MENU_FLAG_IMAGE,
                              N_("Attach another data field as presentation"));
    gwy_process_func_register("presentation_logscale",
                              &presentation_logscale,
                              N_("/_Presentation/_Logscale"),
                              NULL,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE,
                              N_("Creates a presentation with logarithmic color scale"));

    return TRUE;
}

static void
presentation_remove(GwyFile *data, GwyRunModeFlags mode)
{
    GQuark quark;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_SHOW_FIELD_KEY, &quark,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(quark);
    gwy_app_undo_qcheckpointv(GWY_CONTAINER(data), 1, &quark);
    gwy_container_remove(GWY_CONTAINER(data), quark);
    gwy_log_add(data, GWY_FILE_IMAGE, id, id);
}

static void
presentation_extract(GwyFile *data, GwyRunModeFlags mode)
{
    GwyField *dfield;
    GQuark quark;
    gint oldid, newid;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD_ID, &oldid,
                                 GWY_APP_SHOW_FIELD_KEY, &quark,
                                 GWY_APP_SHOW_FIELD, &dfield,
                                 0);
    g_return_if_fail(dfield && quark);

    dfield = gwy_field_copy(dfield);
    newid = gwy_file_add_image(data, dfield);
    gwy_file_set_visible(data, GWY_FILE_IMAGE, newid, TRUE);
    g_object_unref(dfield);
    gwy_file_sync_items(data, GWY_FILE_IMAGE, oldid,
                        data, GWY_FILE_IMAGE, newid,
                        GWY_FILE_ITEM_PALETTE, FALSE);
    gwy_file_set_title(data, GWY_FILE_IMAGE, newid, NULL, TRUE);
    gwy_log_add(data, GWY_FILE_IMAGE, oldid, newid);
}

static void
presentation_logscale(GwyFile *data, GwyRunModeFlags mode)
{
    GwyField *dfield, *sfield;
    GQuark squark;
    gdouble *d;
    gdouble min, max, m0;
    gint xres, yres, i, zeroes, id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD, &dfield,
                                 GWY_APP_SHOW_FIELD_KEY, &squark,
                                 GWY_APP_SHOW_FIELD, &sfield,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(dfield && squark);

    xres = gwy_field_get_xres(dfield);
    yres = gwy_field_get_yres(dfield);
    gwy_app_undo_qcheckpointv(GWY_CONTAINER(data), 1, &squark);
    if (!sfield) {
        sfield = gwy_field_copy(dfield);
        gwy_container_pass_object(GWY_CONTAINER(data), gwy_file_key_image_picture(id), sfield);
    }
    else {
        gwy_field_resize(sfield, xres, yres);
        gwy_field_copy_data(dfield, sfield);
    }

    d = gwy_field_get_data(sfield);
    zeroes = 0;
    max = 0;
    min = G_MAXDOUBLE;
    for (i = 0; i < xres*yres; i++) {
        d[i] = ABS(d[i]);
        if (G_UNLIKELY(d[i] > max))
            max = d[i];
        if (d[i] == 0.0)
            zeroes++;
        else if (G_UNLIKELY(d[i] < min))
            min = d[i];
    }
    if (min == max || zeroes == xres*yres)
        return;

    if (!zeroes) {
        for (i = 0; i < xres*yres; i++)
            d[i] = log(d[i]);
    }
    else {
        m0 = log(min) - log(max/min)/512.0;
        for (i = 0; i < xres*yres; i++)
            d[i] = d[i] ? log(d[i]) : m0;
    }

    gwy_field_data_changed(sfield);
    gwy_log_add(data, GWY_FILE_IMAGE, id, id);
}

static void
presentation_attach(GwyFile *data, GwyRunModeFlags mode)
{
    GwyDialogOutcome outcome;
    GwyField *dfield;
    ModuleArgs args;
    GQuark squark;
    gint id;

    g_return_if_fail(mode & ATTACH_RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD, &args.target,
                                 GWY_APP_FIELD_ID, &id,
                                 GWY_APP_SHOW_FIELD_KEY, &squark,
                                 0);

    args.params = gwy_params_new_from_settings(define_attach_params());
    outcome = run_attach_gui(&args);
    gwy_params_save_to_settings(args.params);
    if (outcome == GWY_DIALOG_CANCEL)
        goto end;

    dfield = gwy_field_copy(args.target);
    gwy_app_undo_qcheckpointv(GWY_CONTAINER(data), 1, &squark);
    gwy_container_pass_object(GWY_CONTAINER(data), gwy_file_key_image_picture(id), dfield);

end:
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_attach_gui(ModuleArgs *args)
{
    GwyDialog *dialog;
    GwyParamTable *table;

    dialog = GWY_DIALOG(gwy_dialog_new(_("Attach Presentation")));
    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    table = gwy_param_table_new(args->params);
    gwy_param_table_append_image_id(table, PARAM_SOURCE);
    gwy_param_table_data_id_set_filter(table, PARAM_SOURCE, attach_source_filter, args->target, NULL);

    gwy_dialog_add_content(dialog, gwy_param_table_widget(table), FALSE, FALSE, 0);
    gwy_dialog_add_param_table(dialog, table);

    return gwy_dialog_run(dialog);
}

static gboolean
attach_source_filter(GwyFile *data, gint id, gpointer user_data)
{
    GwyField *source, *target = (GwyField*)user_data;

    if (!(source = gwy_file_get_image(data, id)))
        return FALSE;
    return !gwy_field_is_incompatible(source, target,
                                      GWY_DATA_MISMATCH_RES | GWY_DATA_MISMATCH_REAL | GWY_DATA_MISMATCH_LATERAL);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
