/*
 *  $Id: nield.c 28981 2025-12-12 09:30:14Z yeti-dn $
 *  Copyright (C) 2025 David Nečas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "tests/testlibgwy.h"

enum {
    SOME_NUMBER = 3157,
};

static void
assert_nield_content(GwyNield *nield, const gint *ref_data, guint ref_n)
{
    g_assert_true(GWY_IS_NIELD(nield));
    guint xres = gwy_nield_get_xres(nield);
    guint yres = gwy_nield_get_yres(nield);
    g_assert_cmpuint(xres*yres, ==, ref_n);
    const gint *data = gwy_nield_get_data_const(nield);
    g_assert_nonnull(data);

    for (guint i = 0; i < ref_n; i++)
        g_assert_cmpint(data[i], ==, ref_data[i]);
}

void
nield_assert_equal(GObject *object, GObject *reference)
{
    g_assert_true(GWY_IS_NIELD(object));
    g_assert_true(GWY_IS_NIELD(reference));

    GwyNield *nield = GWY_NIELD(object), *nield_ref = GWY_NIELD(reference);
    g_assert_cmpint(gwy_nield_get_xres(nield), ==, gwy_nield_get_xres(nield_ref));
    g_assert_cmpint(gwy_nield_get_yres(nield), ==, gwy_nield_get_yres(nield_ref));
    assert_nield_content(nield, gwy_nield_get_data(nield_ref),
                         gwy_nield_get_xres(nield_ref)*gwy_nield_get_yres(nield_ref));
}

static GwyNield*
parse_nield(gint xres, gint yres, const gchar *s)
{
    GwyNield *nield = gwy_nield_new(xres, yres);
    gint *d = gwy_nield_get_data(nield);
    for (gint i = 0; i < xres*yres; i++) {
        if (g_ascii_isdigit(s[i]))
            d[i] = (gint)s[i] - (gint)'0';
        else if (g_ascii_islower(s[i]))
            d[i] = (gint)s[i] - (gint)'a' + 10;
        else if (s[i] == ' ')
            d[i] = 0;
        else if (s[i] == '-')
            d[i] = -1;
        else {
            g_assert_not_reached();
        }
    }
    g_assert(!s[xres*yres]);
    return nield;
}

void
test_nield_basic(void)
{
    const gint zeros[6] = { 0, 0, 0, 0, 0, 0 };
    const gint twos[6] = { 2, 2, 2, 2, 2, 2 };

    GwyNield *nield = gwy_nield_new(2, 3);
    assert_nield_content(nield, zeros, 6);

    gwy_nield_fill(nield, 2);
    assert_nield_content(nield, twos, 6);

    gwy_nield_clear(nield);
    assert_nield_content(nield, zeros, 6);

    g_assert_finalize_object(nield);
}

void
test_nield_data_changed(void)
{
    GwyNield *nield = gwy_nield_new(1, 1);
    guint item_changed = 0;
    g_signal_connect_swapped(nield, "data-changed", G_CALLBACK(record_signal), &item_changed);
    gwy_nield_data_changed(nield);
    g_assert_cmpuint(item_changed, ==, 1);
    gwy_nield_data_changed(nield);
    g_assert_cmpuint(item_changed, ==, 2);
    g_assert_finalize_object(nield);
}

static GwyNield*
create_nield_for_serialisation(void)
{
    GwyNield *nield = gwy_nield_new(13, 7);
    gint *data = gwy_nield_get_data(nield);
    for (guint i = 0; i < 13*7; i++)
        data[i] = (i*i) % 11;

    return nield;
}

void
test_nield_serialization(void)
{
    GwyNield *nield = create_nield_for_serialisation();
    serialize_object_and_back(G_OBJECT(nield), nield_assert_equal, FALSE, NULL);

    g_assert_finalize_object(nield);
}

void
test_nield_copy(void)
{
    GwyNield *nield = create_nield_for_serialisation();
    serializable_test_copy(GWY_SERIALIZABLE(nield), nield_assert_equal);

    g_assert_finalize_object(nield);
}

void
test_nield_assign(void)
{
    GwyNield *nield = create_nield_for_serialisation();
    serializable_test_assign(GWY_SERIALIZABLE(nield), NULL, nield_assert_equal);

    GwyNield *another = gwy_nield_new(4, 5);
    serializable_test_assign(GWY_SERIALIZABLE(nield), GWY_SERIALIZABLE(another), nield_assert_equal);
    g_assert_finalize_object(another);

    g_assert_finalize_object(nield);
}

void
test_nield_resize(void)
{
    guint n = g_test_slow() ? 10000 : 1000;

    GwyNield *nield = gwy_nield_new(1, 1);
    for (guint k = 0; k < n; k++) {
        gint width = g_test_rand_int_range(1, 12), height = g_test_rand_int_range(1, 12);
        gwy_nield_resize(nield, width, height);
        g_assert_cmpint(gwy_nield_get_xres(nield), ==, width);
        g_assert_cmpint(gwy_nield_get_yres(nield), ==, height);
        /* Try to crash if the memory is not allocated correctly. */
        gint *d = gwy_nield_get_data(nield);
        gwy_clear(d, width*height);
    }
    g_assert_finalize_object(nield);
}

GwyNield*
make_ij_nield(void)
{
    gint xres = g_test_rand_int_range(2, 20);
    gint yres = g_test_rand_int_range(2, 20);
    GwyNield *nield = gwy_nield_new(xres, yres);
    gint *d = gwy_nield_get_data(nield);

    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++)
            d[i*xres + j] = SOME_NUMBER*i + j;
    }

    return nield;
}

void
test_nield_area_extract(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_ij_nield();
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width), row = g_test_rand_int_range(0, yres-height);
        GwyNield *extracted = gwy_nield_area_extract(nield, col, row, width, height);
        const gint *d = gwy_nield_get_data_const(extracted);
        for (gint i = 0; i < height; i++) {
            for (gint j = 0; j < width; j++) {
                g_assert_cmpint(d[i*width + j], ==, SOME_NUMBER*(i + row) + (j + col));
            }
        }
        g_assert_finalize_object(extracted);
        g_assert_finalize_object(nield);
    }
}

void
test_nield_crop(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_ij_nield();
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width), row = g_test_rand_int_range(0, yres-height);
        gwy_nield_crop(nield, col, row, width, height);
        const gint *d = gwy_nield_get_data_const(nield);
        for (gint i = 0; i < height; i++) {
            for (gint j = 0; j < width; j++) {
                g_assert_cmpint(d[i*width + j], ==, SOME_NUMBER*(i + row) + (j + col));
            }
        }
        g_assert_finalize_object(nield);
    }
}

void
test_nield_area_copy(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_ij_nield();
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width+1), row = g_test_rand_int_range(0, yres-height+1);
        gint xres2 = g_test_rand_int_range(width, 25), yres2 = g_test_rand_int_range(height, 25);
        gint destcol = g_test_rand_int_range(0, xres2-width+1), destrow = g_test_rand_int_range(0, yres2-height+1);
        GwyNield *destnield = gwy_nield_new(xres2, yres2);
        gwy_nield_fill(destnield, G_PI);
        gwy_nield_area_copy(nield, destnield, col, row, width, height, destcol, destrow);
        const gint *d = gwy_nield_get_data_const(destnield);
        for (gint i = 0; i < yres2; i++) {
            for (gint j = 0; j < xres2; j++) {
                if (i >= destrow && i-destrow < height && j >= destcol && j-destcol < width)
                    g_assert_cmpint(d[i*xres2 + j], ==, SOME_NUMBER*(i + row - destrow) + (j + col - destcol));
                else
                    g_assert_cmpint(d[i*xres2 + j], ==, G_PI);
            }
        }
        g_assert_finalize_object(destnield);
        g_assert_finalize_object(nield);
    }
}

void
test_nield_flip_x(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_ij_nield();
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        gwy_nield_flip(nield, TRUE, FALSE);
        const gint *d = gwy_nield_get_data_const(nield);
        for (gint i = 0; i < yres; i++) {
            for (gint j = 0; j < xres; j++) {
                gint flipped_j = xres-1 - j;
                g_assert_cmpint(d[i*xres + j], ==, SOME_NUMBER*i + flipped_j);
            }
        }
        g_assert_finalize_object(nield);
    }
}

void
test_nield_flip_y(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_ij_nield();
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        gwy_nield_flip(nield, FALSE, TRUE);
        const gint *d = gwy_nield_get_data_const(nield);
        for (gint i = 0; i < yres; i++) {
            gint flipped_i = yres-1 - i;
            for (gint j = 0; j < xres; j++) {
                g_assert_cmpint(d[i*xres + j], ==, SOME_NUMBER*flipped_i + j);
            }
        }
        g_assert_finalize_object(nield);
    }
}

void
test_nield_flip_xy(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_ij_nield();
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        gwy_nield_flip(nield, TRUE, TRUE);
        const gint *d = gwy_nield_get_data_const(nield);
        for (gint i = 0; i < yres; i++) {
            gint flipped_i = yres-1 - i;
            for (gint j = 0; j < xres; j++) {
                gint flipped_j = xres-1 - j;
                g_assert_cmpint(d[i*xres + j], ==, SOME_NUMBER*flipped_i + flipped_j);
            }
        }
        g_assert_finalize_object(nield);
    }
}

void
test_nield_setval(void)
{
    gint xres = g_test_rand_int_range(2, 20);
    gint yres = g_test_rand_int_range(2, 20);
    GwyNield *nield = gwy_nield_new(xres, yres);
    gwy_nield_fill(nield, G_PI);

    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++)
            gwy_nield_set_val(nield, j, i, i*SOME_NUMBER + j);
    }
    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++)
            g_assert_cmpint(gwy_nield_get_val(nield, j, i), ==, i*SOME_NUMBER + j);
    }
    g_assert_finalize_object(nield);
}

void
test_nield_transpose_major(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_ij_nield();
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        GwyNield *transposed = gwy_nield_new(1, 1);
        gwy_nield_transpose(nield, transposed, FALSE);
        g_assert_cmpint(gwy_nield_get_xres(transposed), ==, yres);
        g_assert_cmpint(gwy_nield_get_yres(transposed), ==, xres);
        const gint *d = gwy_nield_get_data_const(transposed);
        for (gint i = 0; i < xres; i++) {
            for (gint j = 0; j < yres; j++)
                g_assert_cmpint(d[i*yres + j], ==, SOME_NUMBER*j + i);
        }
        GwyNield *transposed2 = gwy_nield_new(1, 1);
        gwy_nield_transpose(transposed, transposed2, FALSE);
        nield_assert_equal(G_OBJECT(nield), G_OBJECT(transposed2));
        g_assert_finalize_object(transposed2);
        g_assert_finalize_object(transposed);
        g_assert_finalize_object(nield);
    }
}

void
test_nield_transpose_minor(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_ij_nield();
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        GwyNield *transposed = gwy_nield_new(1, 1);
        gwy_nield_transpose(nield, transposed, TRUE);
        g_assert_cmpint(gwy_nield_get_xres(transposed), ==, yres);
        g_assert_cmpint(gwy_nield_get_yres(transposed), ==, xres);
        const gint *d = gwy_nield_get_data_const(transposed);
        for (gint i = 0; i < xres; i++) {
            for (gint j = 0; j < yres; j++)
                g_assert_cmpint(d[i*yres + j], ==, SOME_NUMBER*(yres-1 - j) + (xres-1 - i));
        }
        GwyNield *transposed2 = gwy_nield_new(1, 1);
        gwy_nield_transpose(transposed, transposed2, TRUE);
        nield_assert_equal(G_OBJECT(nield), G_OBJECT(transposed2));
        g_assert_finalize_object(transposed2);
        g_assert_finalize_object(transposed);
        g_assert_finalize_object(nield);
    }
}

void
test_nield_count_full(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        gint xres = g_test_rand_int_range(2, 20);
        gint yres = g_test_rand_int_range(2, 20);
        GwyNield *nield = gwy_nield_new(xres, yres);
        gint count = g_test_rand_int_range(0, 2*xres*yres/3);
        gint cnt = 0;
        while (cnt < count) {
            gint j = g_test_rand_int_range(0, xres);
            gint i = g_test_rand_int_range(0, yres);
            if (gwy_nield_get_val(nield, j, i) <= 0) {
                gwy_nield_set_val(nield, j, i, g_test_rand_int_range(1, 5));
                cnt++;
            }
        }
        g_assert_cmpint(gwy_nield_count(nield), ==, count);
        g_assert_finalize_object(nield);
    }
}

void
test_nield_count_area(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        gint xres = g_test_rand_int_range(3, 25);
        gint yres = g_test_rand_int_range(3, 25);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width), row = g_test_rand_int_range(0, yres-height);
        GwyNield *nield = gwy_nield_new(xres, yres);
        gwy_nield_fill(nield, 1);
        gwy_nield_area_clear(nield, col, row, width, height);
        gint count = g_test_rand_int_range(0, MAX(2*width*height/3, 1));
        gint cnt = 0;
        while (cnt < count) {
            gint j = g_test_rand_int_range(0, width);
            gint i = g_test_rand_int_range(0, height);
            if (gwy_nield_get_val(nield, col+j, row+i) <= 0) {
                gwy_nield_set_val(nield, col+j, row+i, g_test_rand_int_range(1, 5));
                cnt++;
            }
        }
        g_assert_cmpint(gwy_nield_area_count(nield, NULL, GWY_MASK_IGNORE, col, row, width, height), ==, count);
        g_assert_finalize_object(nield);
    }
}

void
test_nield_count_masked(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        gint xres = g_test_rand_int_range(3, 25);
        gint yres = g_test_rand_int_range(3, 25);
        gint width = g_test_rand_int_range(1, xres), height = g_test_rand_int_range(1, yres);
        gint col = g_test_rand_int_range(0, xres-width), row = g_test_rand_int_range(0, yres-height);
        GwyNield *nield = gwy_nield_new(xres, yres);
        GwyNield *mask = gwy_nield_new(xres, yres);
        gwy_nield_fill(nield, 1);
        gwy_nield_area_clear(nield, col, row, width, height);
        gwy_nield_fill(mask, 1);
        gint count = g_test_rand_int_range(0, MAX(2*width*height/3, 1));
        gint cnt = 0, expected = 0;
        while (cnt < count) {
            gint j = g_test_rand_int_range(0, width);
            gint i = g_test_rand_int_range(0, height);
            if (gwy_nield_get_val(nield, col+j, row+i) <= 0) {
                gwy_nield_set_val(nield, col+j, row+i, g_test_rand_int_range(1, 5));
                if (g_test_rand_int() % 2) {
                    gwy_nield_set_val(mask, col+j, row+i, -1);
                    expected++;
                }
                cnt++;
            }
        }
        g_assert_cmpint(gwy_nield_area_count(nield, mask, GWY_MASK_EXCLUDE, col, row, width, height), ==, expected);
        g_assert_finalize_object(nield);
    }
}

static void
assert_nield_compatibility(gint xres1, gint yres1,
                           gint xres2, gint yres2,
                           GwyDataMismatchFlags flags_to_test,
                           GwyDataMismatchFlags expected_result)
{
    GwyNield *nield1 = gwy_nield_new(xres1, yres1);
    GwyNield *nield2 = gwy_nield_new(xres2, yres2);

    g_assert_cmphex(gwy_nield_is_incompatible(nield1, nield2, flags_to_test), ==, expected_result);
    g_assert_finalize_object(nield1);
    g_assert_finalize_object(nield2);
}

void
test_nield_compatibility_res(void)
{
    assert_nield_compatibility(1, 2,
                               1, 2,
                               GWY_DATA_MISMATCH_RES, 0);
    assert_nield_compatibility(1, 2,
                               1, 1,
                               GWY_DATA_MISMATCH_RES, GWY_DATA_MISMATCH_RES);
    assert_nield_compatibility(1, 2,
                               2, 2,
                               GWY_DATA_MISMATCH_RES, GWY_DATA_MISMATCH_RES);
    assert_nield_compatibility(1, 2,
                               2, 1,
                               GWY_DATA_MISMATCH_RES, GWY_DATA_MISMATCH_RES);
}

G_GNUC_UNUSED
static void
dump_nield(GwyNield *nield, const gchar *name)
{
    g_printerr("=====[ %s ]=====\n", name);
    gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
    const gint *d = gwy_nield_get_data_const(nield);
    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++) {
            gint v = d[i*xres + j];
            if (v < 0)
                g_printerr("-");
            else if (v == 0)
                g_printerr(" ");
            else if (v < 10)
                g_printerr("%c", '0' + v);
            else
                g_printerr("%c", 'a' + (v - 10));
        }
        g_printerr("\n");
    }
}

void
test_nield_number_contiguous_simple(void)
{
    const gchar src[] =
        "  1                           22  2         "
        "  11  11111111  1   33   22  2222222222  2  "
        " 11   1  1  11111   3  3  22222222   22222  "
        "  1 111 11  11      3  3   22222222    2    "
        "111         11    333333    22  222         "
        "1 111111111111    3   3           2      4  "
        "1                333333  555 555      6     "
        "   7   88   555  3       5 5 5 5    66666666"
        " 777        5 5  3   555 5 5 5 5           6"
        "     9  555 5 5  3   5 5 5 5 5 5 555   66666"
        "        5 5 5 5  3   5 5 5 555 5 5 5       6"
        "aaaaaa  5 555 5  3   5 555     5 5 5 555    "
        "a    a  5     5      5         555 5 5 5   b"
        "a    a  5  5  55555555   c         5 5     b"
        "aaaaaa  5555             cccc  d   555  e   ";

    GwyNield *reference = parse_nield(44, 15, src);
    GwyNield *nield = gwy_nield_copy(reference);
    gwy_nield_flatten(nield);
    g_assert_cmpint(gwy_nield_max(nield), ==, 1);

    gwy_nield_number_contiguous(nield);
    nield_assert_equal(G_OBJECT(nield), G_OBJECT(reference));
    g_assert_cmpint(gwy_nield_max(nield), ==, 14);

    gwy_nield_flatten(nield);
    g_assert_cmpint(gwy_nield_max(nield), ==, 1);

    gwy_nield_split_noncontiguous(nield);
    nield_assert_equal(G_OBJECT(nield), G_OBJECT(reference));
    g_assert_cmpint(gwy_nield_max(nield), ==, 14);

    g_assert_finalize_object(nield);
    g_assert_finalize_object(reference);
}

void
test_nield_number_contiguous_spiral(void)
{
    const gchar src[] =
        "1 2222222222222222222222222 3"
        "  2                       2  "
        "2 2 22222222222222222222  2 4"
        "2 2 2                  2  2  "
        "2 2 2 222222222222222  2  2 5"
        "2 2 2 2             2  2  2  "
        "2 2 2 2 2222222222  2  2  2 6"
        "2 2 2 2 2        2  2  2  2  "
        "2 2 2 2 2 22222  2  2  2  2 7"
        "2 2 2 2 2     2  2  2  2  2  "
        "2 2 2 2 2222222  2  2  2  2 8"
        "2 2 2 2          2  2  2  2  "
        "2 2 2 222222222222  2  2  2 9"
        "2 2 2               2  2  2  "
        "2 2 22222222222222222  2  2 a"
        "2 2                    2  2  "
        "2 2222222222222222222222  2 b"
        "2                         2  "
        "222222222222222222222222222 c";

    GwyNield *reference = parse_nield(29, 19, src);
    GwyNield *nield = gwy_nield_copy(reference);
    gwy_nield_flatten(nield);
    g_assert_cmpint(gwy_nield_max(nield), ==, 1);

    gwy_nield_number_contiguous(nield);
    nield_assert_equal(G_OBJECT(nield), G_OBJECT(reference));
    g_assert_cmpint(gwy_nield_max(nield), ==, 12);

    gwy_nield_flatten(nield);
    g_assert_cmpint(gwy_nield_max(nield), ==, 1);

    gwy_nield_split_noncontiguous(nield);
    nield_assert_equal(G_OBJECT(nield), G_OBJECT(reference));
    g_assert_cmpint(gwy_nield_max(nield), ==, 12);

    g_assert_finalize_object(nield);
    g_assert_finalize_object(reference);
}

void
test_nield_number_contiguous_negative(void)
{
    const gchar src[] =
        " -- 11-222    "
        "    1    2    "
        "11111    2    "
        " -       2222 "
        " 3----------- "
        " 33333333333  "
        "   3     3 3  ";

    GwyNield *reference = parse_nield(14, 7, src);
    GwyNield *nield = gwy_nield_copy(reference);
    gint *d = gwy_nield_get_data(nield);
    for (gint i = 0; i < 14*7; i++) {
        if (d[i] > 0)
            d[i] = 1;
    }
    /* Implicit invalidate. */
    g_assert_cmpint(gwy_nield_max(nield), ==, 1);

    gwy_nield_number_contiguous(nield);
    nield_assert_equal(G_OBJECT(nield), G_OBJECT(reference));
    g_assert_cmpint(gwy_nield_max(nield), ==, 3);

    d = gwy_nield_get_data(nield);
    for (gint i = 0; i < 14*7; i++) {
        if (d[i] > 0)
            d[i] = 1;
    }
    /* Implicit invalidate. */
    g_assert_cmpint(gwy_nield_max(nield), ==, 1);

    gwy_nield_split_noncontiguous(nield);
    nield_assert_equal(G_OBJECT(nield), G_OBJECT(reference));
    g_assert_cmpint(gwy_nield_max(nield), ==, 3);

    g_assert_finalize_object(nield);
    g_assert_finalize_object(reference);
}

static GwyNield*
make_random_nield(gboolean also_negative, gint k_empty)
{
    gint xres = g_test_rand_int_range(2, 20);
    gint yres = g_test_rand_int_range(2, 20);
    GwyNield *nield = gwy_nield_new(xres, yres);
    gint *d = gwy_nield_get_data(nield);

    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++) {
            if (g_test_rand_int() % k_empty == 0) {
                if (also_negative && g_test_rand_int() % 10 == 0)
                    d[i*xres + j] = -1;
                else
                    d[i*xres + j] = 0;
            }
            else {
                gint k = 1;
                while (g_test_rand_int() % 2)
                    k++;
                d[i*xres + j] = k;
            }
        }
    }

    return nield;
}

GwyNield*
make_random_mask(gint xres, gint yres, gboolean also_negative, gint maxgno)
{
    GwyNield *nield = gwy_nield_new(xres, yres);
    gint *d = gwy_nield_get_data(nield);

    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++) {
            if (g_test_rand_int() % 2) {
                if (also_negative && g_test_rand_int() % 3 == 1)
                    d[i*xres + j] = -1;
                else
                    d[i*xres + j] = 0;
            }
            else
                d[i*xres + j] = g_test_rand_int_range(1, maxgno+1);
        }
    }

    return nield;
}

void
test_nield_isolate(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_random_nield(FALSE, 3);
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        gint m = gwy_nield_max(nield);
        GwyNield *isolated = gwy_nield_new_alike(nield);
        GwyNield *another = gwy_nield_new_alike(nield);

        /* Rebuild it grain by grain. */
        for (gint i = 1; i <= m; i++) {
            gwy_nield_copy_data(nield, isolated);
            gwy_nield_isolate(isolated, i);
            g_assert_cmpint(gwy_nield_max(isolated), >=, 0);
            g_assert_cmpint(gwy_nield_max(isolated), <=, 1);
            gint *d = gwy_nield_get_data(isolated);
            for (gint j = 0; j < xres*yres; j++) {
                if (d[j] > 0)
                    d[j] = i;
            }
            gwy_nield_union(another, isolated);
        }

        nield_assert_equal(G_OBJECT(another), G_OBJECT(nield));

        g_assert_finalize_object(another);
        g_assert_finalize_object(isolated);
        g_assert_finalize_object(nield);
    }
}

void
test_nield_clear_at(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield;

        do {
            nield = make_random_nield(TRUE, 3);
        }
        while (gwy_nield_max(nield) <= 0);

        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        gint i, j;
        do {
            j = g_test_rand_int_range(0, xres);
            i = g_test_rand_int_range(0, yres);
        } while (gwy_nield_get_val(nield, j, i) <= 0);

        GwyNield *another = gwy_nield_copy(nield);
        gwy_nield_split_noncontiguous(another);
        gwy_nield_clear_by_number(another, gwy_nield_get_val(another, j, i), FALSE);
        gwy_nield_clear_at(nield, j, i);
        gwy_nield_flatten(nield);
        gwy_nield_flatten(another);

        nield_assert_equal(G_OBJECT(another), G_OBJECT(nield));

        g_assert_finalize_object(another);
        g_assert_finalize_object(nield);
    }
}

void
test_nield_isolate_at(void)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_random_nield(FALSE, 3);
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        GwyNield *workspace = gwy_nield_copy(nield);
        GwyNield *isolated = gwy_nield_new_alike(nield);
        GwyNield *another = gwy_nield_new_alike(nield);

        /* Rebuild it grain piece by grain piece. */
        while (gwy_nield_max(workspace) > 0) {
            gint *w = gwy_nield_get_data(workspace);
            gint i, j;
            for (i = 0; i < yres; i++) {
                for (j = 0; j < xres; j++) {
                    if (w[i*xres + j] > 0)
                        break;
                }
                if (j < xres)
                    break;
            }
            gint v = w[i*xres + j];
            gwy_nield_copy_data(workspace, isolated);
            gwy_nield_clear_at(workspace, j, i);
            gwy_nield_isolate_at(isolated, j, i);
            g_assert_cmpint(gwy_nield_max(isolated), ==, 1);
            gint *d = gwy_nield_get_data(isolated);
            for (j = 0; j < xres*yres; j++) {
                if (d[j] > 0)
                    d[j] = v;
            }
            gwy_nield_union(another, isolated);
        }

        nield_assert_equal(G_OBJECT(another), G_OBJECT(nield));

        g_assert_finalize_object(another);
        g_assert_finalize_object(isolated);
        g_assert_finalize_object(workspace);
        g_assert_finalize_object(nield);
    }
}

typedef gdouble (*DistanceFunc)(gint j1, gint i1, gint j2, gint i2);

static gdouble
distance_4connect(gint j1, gint i1, gint j2, gint i2)
{
    return ABS(j1 - j2) + ABS(i1 - i2);
}

static gdouble
distance_8connect(gint j1, gint i1, gint j2, gint i2)
{
    return MAX(ABS(j1 - j2), ABS(i1 - i2));
}

static gdouble
distance_octagonal48(gint j1, gint i1, gint j2, gint i2)
{
    gint di = ABS(i1 - i2), dj = ABS(j1 - j2);
    gint m = MIN(di, dj);
    gint M = MAX(di, dj);
    if (M >= 2*m)
        return M;
    return 2*(di + dj + 1)/3;
}

static gdouble
distance_octagonal84(gint j1, gint i1, gint j2, gint i2)
{
    gint di = ABS(i1 - i2), dj = ABS(j1 - j2);
    gint m = MIN(di, dj);
    gint M = MAX(di, dj);
    if (M >= 2*m)
        return M;
    return (2*(di + dj) + 1)/3;
}

static gdouble
distance_octagonal(gint j1, gint i1, gint j2, gint i2)
{
    gint di = ABS(i1 - i2), dj = ABS(j1 - j2);
    gint m = MIN(di, dj);
    gint M = MAX(di, dj);
    if (M >= 2*m)
        return M;
    return ((2*(di + dj) + 1)/3 + 2*(di + dj + 1)/3)/2.0;
}

static gdouble
distance_euclidean(gint j1, gint i1, gint j2, gint i2)
{
    return hypot(j1 - j2, i1 - i2);
}

static void
test_one_distance(GwyDistanceTransformType dist_type, gboolean from_border,
                  DistanceFunc distance_func)
{
    guint n = g_test_slow() ? 1000 : 100;

    for (guint k = 0; k < n; k++) {
        GwyNield *nield = make_random_nield(FALSE, 5);
        gint xres = gwy_nield_get_xres(nield), yres = gwy_nield_get_yres(nield);
        GwyField *distances = gwy_field_new(1, 1, 1, 1, FALSE);

        dump_nield(nield, "nield");
        g_printerr("\n");

        gwy_nield_distance_transform(nield, distances, dist_type, from_border);
        g_assert_cmpint(gwy_field_get_xres(distances), ==, xres);
        g_assert_cmpint(gwy_field_get_yres(distances), ==, yres);

        const gdouble *dist = gwy_field_get_data(distances);
        const gint *grains = gwy_nield_get_data(nield);

        for (gint i1 = 0; i1 < yres; i1++) {
            for (gint j1 = 0; j1 < xres; j1++) {
                gint gno = grains[i1*xres + j1];
                gdouble d = dist[i1*xres + j1];
                if (gno <= 0) {
                    g_assert_cmpfloat(d, ==, 0.0);
                    continue;
                }
                gdouble d_bruteforce = G_MAXDOUBLE;
                for (gint i2 = 0; i2 < yres; i2++) {
                    for (gint j2 = 0; j2 < xres; j2++) {
                        if (grains[i2*xres + j2] <= 0)
                            d_bruteforce = fmin(d_bruteforce, distance_func(j1, i1, j2, i2));
                    }
                }
                /* Empty space around the image. */
                if (from_border) {
                    for (gint i2 = 0; i2 < yres; i2++) {
                        d_bruteforce = fmin(d_bruteforce, distance_func(j1, i1, -1, i2));
                        d_bruteforce = fmin(d_bruteforce, distance_func(j1, i1, xres, i2));
                    }
                    for (gint j2 = 0; j2 < xres; j2++) {
                        d_bruteforce = fmin(d_bruteforce, distance_func(j1, i1, j2, -1));
                        d_bruteforce = fmin(d_bruteforce, distance_func(j1, i1, j2, yres));
                    }
                }

                if (d_bruteforce == G_MAXDOUBLE)
                    g_assert_cmpfloat(d, >=, G_MAXDOUBLE);
                else
                    g_assert_cmpfloat_with_epsilon(d, d_bruteforce, 2*d_bruteforce*DBL_EPSILON);
            }
        }

        g_assert_finalize_object(distances);
        g_assert_finalize_object(nield);
    }
}

void
test_nield_distance_transform_conn4_noborder(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_CONN4, FALSE, distance_4connect);
}

void
test_nield_distance_transform_conn4_border(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_CONN4, TRUE, distance_4connect);
}

void
test_nield_distance_transform_conn8_noborder(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_CONN8, FALSE, distance_8connect);
}

void
test_nield_distance_transform_conn8_border(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_CONN8, TRUE, distance_8connect);
}

void
test_nield_distance_transform_octagonal48_noborder(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_OCTAGONAL48, FALSE, distance_octagonal48);
}

void
test_nield_distance_transform_octagonal48_border(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_OCTAGONAL48, TRUE, distance_octagonal48);
}

void
test_nield_distance_transform_octagonal84_noborder(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_OCTAGONAL84, FALSE, distance_octagonal84);
}

void
test_nield_distance_transform_octagonal84_border(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_OCTAGONAL84, TRUE, distance_octagonal84);
}

void
test_nield_distance_transform_octagonal_noborder(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_OCTAGONAL, FALSE, distance_octagonal);
}

void
test_nield_distance_transform_octagonal_border(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_OCTAGONAL, TRUE, distance_octagonal);
}

void
test_nield_distance_transform_euclidean_noborder(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_EUCLIDEAN, FALSE, distance_euclidean);
}

void
test_nield_distance_transform_euclidean_border(void)
{
    test_one_distance(GWY_DISTANCE_TRANSFORM_EUCLIDEAN, TRUE, distance_euclidean);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
