/*
 *  $Id: file.c 29477 2026-02-14 13:29:30Z yeti-dn $
 *  Copyright (C) 2025 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#define DEBUG 1
#include "config.h"
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <glib/gi18n-lib.h>

#include "libgwyddion/gwyddion.h"
#include "libgwyapp/settings.h"
#include "libgwyapp/file.h"
#include "libgwyapp/types.h"

#include "libgwyapp/gwyappinternal.h"
#include "libgwyapp/data-browser-internal.h"

G_STATIC_ASSERT(sizeof(GQuark) == sizeof(guint32));

#define TYPE_NAME "GwyFile"

#define gwy_strlen0(s) (s ? strlen(s) : 0)

enum {
    ITEM_LINKS,
    NUM_ITEMS
};

typedef GQuark (*GetKeyFunc)(gint id);

typedef struct {
    /* GSequence would be better for files with many data items. But it has a pretty large overhead. */
    GArray *ids;
} DataKindInfo;

struct _GwyFilePrivate {
    DataKindInfo data_kind_info[GWY_FILE_N_KINDS];

    GArray *links;
    const gchar **link_strings;
    GQuark setting_link;

    const gchar *format_name;
    gchar *filename_sys;

    glong id;
    gboolean is_managed;
};

typedef struct {
    const gchar *suffix;
    GwyFilePiece piece;
    GType type;
} FilePieceSpec;

typedef struct {
    GetKeyFunc get_key;
    const gchar *prefix;
    const guint *pieces;
    guint npieces;
    GType type;
} DataKindSpec;

static void             serializable_init     (GwySerializableInterface *iface);
static void             serializable_itemize  (GwySerializable *serializable,
                                               GwySerializableGroup *group);
static void             serializable_done     (GwySerializable *serializable);
static gboolean         serializable_construct(GwySerializable *serializable,
                                               GwySerializableGroup *group,
                                               GwyErrorList **error_list);
static GwySerializable* serializable_copy     (GwySerializable *serializable);
static void             serializable_assign   (GwySerializable *destination,
                                               GwySerializable *source);
static void             dispose               (GObject *object);
static void             finalize              (GObject *object);
static void             item_changed          (GwyDict *container,
                                               GQuark key);
static void             construction_finished (GwyDict *container);
static DataKindInfo*    ensure_data_kind_info (GwyFile *file,
                                               GwyDataKind data_kind);
static void             fill_piece_gtypes     (FilePieceSpec *pieces,
                                               guint nspec);
static void             copy_aux_info         (GwyFile *dest,
                                               GwyFile *src);
static GArray*          assign_garray         (GArray *dest,
                                               GArray *src);
static void             set_link              (GwyFile *file,
                                               GQuark key,
                                               GQuark linkquark,
                                               const gchar *linkstr);
static gint             find_link             (GwyFile *file,
                                               GQuark quark);

static GObjectClass *parent_class = NULL;
static GwySerializableInterface *serializable_parent_iface = NULL;

static GwySerializableItem serializable_items[NUM_ITEMS] = {
    { .name = "__links__", .ctype = GWY_SERIALIZABLE_STRING_ARRAY, },
};

static FilePieceSpec all_pieces[] = {
    { NULL,            GWY_FILE_PIECE_TITLE,         0, },
    { NULL,            GWY_FILE_PIECE_VISIBLE,       0, },
    { NULL,            GWY_FILE_PIECE_PALETTE,       0, },
    { NULL,            GWY_FILE_PIECE_LOG,           0, },
    { NULL,            GWY_FILE_PIECE_PICTURE,       0, },
    { NULL,            GWY_FILE_PIECE_META,          0, },
    { NULL,            GWY_FILE_PIECE_MASK,          0, },
    { NULL,            GWY_FILE_PIECE_MASK_COLOR,    0, },
    { NULL,            GWY_FILE_PIECE_COLOR_MAPPING, 0, },
    { "max",           GWY_FILE_PIECE_RANGE,         0, },
    { "min",           GWY_FILE_PIECE_RANGE,         0, },
    { NULL,            GWY_FILE_PIECE_SELECTION,     0, },
    { NULL,            GWY_FILE_PIECE_REAL_SQUARE,   0, },
    { "relative-size", GWY_FILE_PIECE_VIEW,          0, },
    { "scale",         GWY_FILE_PIECE_VIEW,          0, },
    { "width",         GWY_FILE_PIECE_VIEW,          0, },
    { "height",        GWY_FILE_PIECE_VIEW,          0, },
};

/* Which pieces from all_pieces[] are valid for given data kind. Use utils/make-file-pieces.py to expand the
 * definitions and generate the numerical id lists. */
#ifdef __GWY_DEFINE_FILE_PIECES__
(COMMON) = LOG META VISIBLE
(IMGLIKE) = (COMMON) TITLE PALETTE PICTURE COLOR_MAPPING RANGE/min,max SELECTION RANGE/min RANGE/max
(PIXELS) = (IMGLIKE) MASK MASK_COLOR REAL_SQUARE VIEW/relative-size,scale
image = (PIXELS)
volume = (PIXELS)
cmap = (PIXELS)
xyz = (IMGLIKE) VIEW/relative-size,width,height
graph = (COMMON) VIEW/relative-size,width,height
spectra = (COMMON)
#else
static const guint image_pieces[] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14 };
static const guint volume_pieces[] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14 };
static const guint cmap_pieces[] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14 };
static const guint xyz_pieces[] = { 0, 1, 2, 3, 4, 5, 8, 9, 10, 11, 13, 15, 16 };
static const guint graph_pieces[] = { 1, 3, 5, 13, 15, 16 };
static const guint spectra_pieces[] = { 1, 3, 5 };
#endif

/* NB: This array is indexed by GwyDataKind. */
static DataKindSpec data_kind_spec[GWY_FILE_N_KINDS] = {
    { gwy_file_key_image,   GWY_FILE_PREFIX_IMAGE,   image_pieces,   G_N_ELEMENTS(image_pieces),   0, },
    { gwy_file_key_graph,   GWY_FILE_PREFIX_GRAPH,   graph_pieces,   G_N_ELEMENTS(graph_pieces),   0, },
    { gwy_file_key_spectra, GWY_FILE_PREFIX_SPECTRA, spectra_pieces, G_N_ELEMENTS(spectra_pieces), 0, },
    { gwy_file_key_volume,  GWY_FILE_PREFIX_VOLUME,  volume_pieces,  G_N_ELEMENTS(volume_pieces),  0, },
    { gwy_file_key_xyz,     GWY_FILE_PREFIX_XYZ,     xyz_pieces,     G_N_ELEMENTS(xyz_pieces),     0, },
    { gwy_file_key_cmap,    GWY_FILE_PREFIX_CMAP,    cmap_pieces,    G_N_ELEMENTS(cmap_pieces),    0, },
};

static const GwyEnum piece_suffixes[] = {
    { "visible",       GWY_FILE_PIECE_VISIBLE,       },
    { "title",         GWY_FILE_PIECE_TITLE,         },
    { "palette",       GWY_FILE_PIECE_PALETTE,       },
    { "mask",          GWY_FILE_PIECE_MASK,          },
    { "mask-color",    GWY_FILE_PIECE_MASK_COLOR,    },
    { "color-mapping", GWY_FILE_PIECE_COLOR_MAPPING, },
    { "range",         GWY_FILE_PIECE_RANGE,         },
    { "show",          GWY_FILE_PIECE_PICTURE,       },
    { "meta",          GWY_FILE_PIECE_META,          },
    { "log",           GWY_FILE_PIECE_LOG,           },
    { "real-square",   GWY_FILE_PIECE_REAL_SQUARE,   },
    { "view",          GWY_FILE_PIECE_VIEW,          },
    { "selection",     GWY_FILE_PIECE_SELECTION,     },
};

G_DEFINE_TYPE_WITH_CODE(GwyFile, gwy_file, GWY_TYPE_DICT,
                        G_ADD_PRIVATE(GwyFile)
                        G_IMPLEMENT_INTERFACE(GWY_TYPE_SERIALIZABLE, serializable_init))

static void
serializable_init(GwySerializableInterface *iface)
{
    serializable_parent_iface = g_type_interface_peek_parent(iface);

    iface->itemize = serializable_itemize;
    iface->done = serializable_done;
    iface->construct = serializable_construct;
    iface->copy = serializable_copy;
    iface->assign = serializable_assign;
}

static void
gwy_file_class_init(GwyFileClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    GwyDictClass *container_class = GWY_DICT_CLASS(klass);
    //GType type = G_TYPE_FROM_CLASS(klass);

    parent_class = gwy_file_parent_class;

    gobject_class->dispose = dispose;
    gobject_class->finalize = finalize;

    container_class->item_changed = item_changed;
    container_class->construction_finished = construction_finished;

    data_kind_spec[GWY_FILE_IMAGE].type = GWY_TYPE_FIELD;
    data_kind_spec[GWY_FILE_GRAPH].type = GWY_TYPE_GRAPH_MODEL;
    data_kind_spec[GWY_FILE_SPECTRA].type = GWY_TYPE_SPECTRA;
    data_kind_spec[GWY_FILE_VOLUME].type = GWY_TYPE_BRICK;
    data_kind_spec[GWY_FILE_XYZ].type = GWY_TYPE_SURFACE;
    data_kind_spec[GWY_FILE_CMAP].type = GWY_TYPE_LAWN;
    fill_piece_gtypes(all_pieces, G_N_ELEMENTS(all_pieces));
}

static void
gwy_file_init(GwyFile *file)
{
    GwyFilePrivate *priv;

    file->priv = priv = gwy_file_get_instance_private(file);

    priv->id = GWY_FILE_ID_NONE;
}

static void
dispose(GObject *object)
{
    G_GNUC_UNUSED GwyFile *file = (GwyFile*)object;

    G_OBJECT_CLASS(parent_class)->dispose(object);
}

static void
finalize(GObject *object)
{
    GwyFile *file = (GwyFile*)object;
    GwyFilePrivate *priv = file->priv;

    for (guint data_kind = 0; data_kind < GWY_FILE_N_KINDS; data_kind++) {
        DataKindInfo *info = priv->data_kind_info + data_kind;
        GWY_FREE_ARRAY(info->ids);
    }
    GWY_FREE_ARRAY(priv->links);
    g_free(priv->filename_sys);

    /* Data browser holds a reference. So we should not get here when the file is managed. */
    g_assert(!priv->is_managed);

    G_OBJECT_CLASS(parent_class)->finalize(object);
}

/**
 * gwy_file_new: (constructor)
 *
 * Creates a new #GwyFile.
 *
 * Returns: (transfer full):
 *          A new data file container.
 **/
GwyFile*
gwy_file_new(void)
{
    return (GwyFile*)g_object_new(GWY_TYPE_FILE, NULL);
}

/**
 * gwy_file_new_in_construction: (constructor)
 *
 * Creates a new #GwyFile and marks is as being in construction.
 *
 * This is a convenience function calling gwy_dict_start_construction() on the newly created file.
 *
 * Returns: (transfer full):
 *          A new data file container.
 **/
GwyFile*
gwy_file_new_in_construction(void)
{
    GwyFile *file = (GwyFile*)g_object_new(GWY_TYPE_FILE, NULL);
    gwy_dict_start_construction(GWY_DICT(file));
    return file;
}

static gint
add_data_object(GwyFile *file, gpointer data_object, GwyDataKind data_kind)
{
    GwyDict *dict = GWY_DICT(file);
    if (gwy_dict_is_being_constructed(dict)) {
        g_critical("gwy_file_add_SOMETHING() cannot be used with files still in construction.");
        /* This triggers data enumeration. This will likely break in one way or another anyway, but try to not fail
         * immediately. */
        gwy_dict_finish_construction(dict);
    }

    DataKindInfo *info = ensure_data_kind_info(file, GWY_FILE_IMAGE);
    g_return_val_if_fail(info, -1);
    const DataKindSpec *spec = data_kind_spec + data_kind;
    g_return_val_if_fail(G_TYPE_CHECK_INSTANCE_TYPE(data_object, spec->type), -1);

    GArray *ids = info->ids;
    gint id = ids->len ? g_array_index(ids, gint, ids->len-1)+1 : 0;
    g_array_append_val(ids, id);
    /* Data browser responds to this by immediately updating its data lists and queuing an update of GUI, which
     * preferably happens after the caller finishes setting other things. */
    gwy_dict_set_object(dict, spec->get_key(id), data_object);
    return id;
}

/**
 * gwy_file_add_image:
 * @file: A data file container to add @field to.
 * @field: Data field representing an image to add.
 *
 * Adds a data field representing an image to a file data container.
 *
 * The file cannot be in construction (see gwy_dict_start_construction()). In particular, this function is not
 * suitable for the import of data from third party file formats, use gwy_file_pass_image().
 *
 * A new unused id is assigned to the image and returned. Generally, it is the lowest integer larger than any existing
 * image id (however, there are no specific guarantees).
 *
 * The image is not immediately shown in a new window when the GUI is running. Use gwy_file_set_visible() to show it,
 * preferrably after setting or synchronising visualisation settings like the colour mapping.
 *
 * Returns: The id of the newly added image in the file.
 **/
gint
gwy_file_add_image(GwyFile *file,
                   GwyField *field)
{
    g_return_val_if_fail(GWY_IS_FILE(file), -1);
    return add_data_object(file, field, GWY_FILE_IMAGE);
}

/**
 * gwy_file_add_graph:
 * @file: A data file container to add @graph to.
 * @graph: Graph model to add.
 *
 * Adds a graph model representing a graph to a file data container.
 *
 * The file cannot be in construction (see gwy_dict_start_construction()). In particular, this function is not
 * suitable for the import of data from third party file formats, use gwy_file_pass_graph().
 *
 * A new unused id is assigned to the graph and returned. Generally, it is the lowest integer larger than any existing
 * graph id (however, there are no specific guarantees).
 *
 * The graph is not immediately shown in a new window when the GUI is running. Use gwy_file_set_visible() to show it,
 * preferrably after setting or synchronising other settings.
 *
 * Returns: The id of the newly added graph in the file.
 **/
gint
gwy_file_add_graph(GwyFile *file,
                   GwyGraphModel *graph)
{
    g_return_val_if_fail(GWY_IS_FILE(file), -1);
    return add_data_object(file, graph, GWY_FILE_GRAPH);
}

/**
 * gwy_file_add_spectra:
 * @file: A data file container to add @spectra to.
 * @spectra: Single point spectra to add.
 *
 * Adds a spectra representing a single point spectra to a file data container.
 *
 * The file cannot be in construction (see gwy_dict_start_construction()). In particular, this function is not
 * suitable for the import of data from third party file formats, use gwy_file_pass_spectra().
 *
 * A new unused id is assigned to the spectra and returned. Generally, it is the lowest integer larger than any
 * existing spectra id (however, there are no specific guarantees).
 *
 * The spectra are not immediately shown in a new window when the GUI is running. Use gwy_file_set_visible() to show it,
 * preferrably after setting or synchronising other settings.
 *
 * Returns: The id of the newly added spectra in the file.
 **/
gint
gwy_file_add_spectra(GwyFile *file,
                     GwySpectra *spectra)
{
    g_return_val_if_fail(GWY_IS_FILE(file), -1);
    return add_data_object(file, spectra, GWY_FILE_SPECTRA);
}

/**
 * gwy_file_add_volume:
 * @file: A data file container to add @brick to.
 * @brick: Data brick to add.
 * @preview: (nullable): Data field to use as the preview.
 *
 * Adds a data brick representing an volume data to a file data container.
 *
 * The file cannot be in construction (see gwy_dict_start_construction()). In particular, this function is not
 * suitable for the import of data from third party file formats, use gwy_file_pass_volume().
 *
 * A new unused id is assigned to the volume data and returned. Generally, it is the lowest integer larger than any
 * existing volume data id (however, there are no specific guarantees).
 *
 * The volume data is not immediately shown in a new window when the GUI is running. Use gwy_file_set_visible() to
 * show it, preferrably after setting or synchronising visualisation settings like the colour mapping.
 *
 * Returns: The id of the newly added volume data in the file.
 **/
gint
gwy_file_add_volume(GwyFile *file,
                    GwyBrick *brick,
                    GwyField *preview)
{
    g_return_val_if_fail(GWY_IS_FILE(file), -1);
    gint id = add_data_object(file, brick, GWY_FILE_VOLUME);
    if (id < 0)
        return id;

    /* XXX: This seems unnecessary. But it is so only if the data browser would never create a preview itself (wasting
     * possibly a lot of time because volume data can be huge) before the caller can add one itself. The data browser
     * sort of wants previews immediately, so maybe just keep the 2.x behaviour here. */
    GQuark quark = gwy_file_key_volume_picture(id);
    GwyDict *container = GWY_DICT(file);
    if (preview) {
        if (GWY_IS_FIELD(preview))
            gwy_dict_set_object(container, quark, preview);
        preview = NULL;
    }
    if (!preview) {
        if (!gwy_dict_gis_object(container, quark, &preview) || !GWY_IS_FIELD(preview)) {
            preview = _gwy_app_create_brick_preview_field(brick);
            gwy_dict_pass_object(container, quark, preview);
        }
    }
    return id;
}

/**
 * gwy_file_add_xyz:
 * @file: A data file container to add @surface to.
 * @surface: XYZ data surface to add.
 *
 * Adds a data surface representing an XYZ data to a file data container.
 *
 * The file cannot be in construction (see gwy_dict_start_construction()). In particular, this function is not
 * suitable for the import of data from third party file formats, use gwy_file_pass_xyz().
 *
 * A new unused id is assigned to the XYZ data and returned. Generally, it is the lowest integer larger than any
 * existing XYZ data id (however, there are no specific guarantees).
 *
 * The XYZ data is not immediately shown in a new window when the GUI is running. Use gwy_file_set_visible() to
 * show it, preferrably after setting or synchronising visualisation settings like the colour mapping.
 *
 * Returns: The id of the newly added XYZ data in the file.
 **/
gint
gwy_file_add_xyz(GwyFile *file,
                 GwySurface *surface)
{
    g_return_val_if_fail(GWY_IS_FILE(file), -1);
    return add_data_object(file, surface, GWY_FILE_XYZ);
}

/**
 * gwy_file_add_cmap:
 * @file: A data file container to add @lawn to.
 * @lawn: Data lawn to add.
 * @preview: (nullable): Data field to use as the preview.
 *
 * Adds a data lawn representing an curve map data to a file data container.
 *
 * The file cannot be in construction (see gwy_dict_start_construction()). In particular, this function is not
 * suitable for the import of data from third party file formats, use gwy_file_pass_cmap().
 *
 * A new unused id is assigned to the curve map data and returned. Generally, it is the lowest integer larger than any
 * existing curve map data id (however, there are no specific guarantees).
 *
 * The curve map data is not immediately shown in a new window when the GUI is running. Use gwy_file_set_visible() to
 * show it, preferrably after setting or synchronising visualisation settings like the colour mapping.
 *
 * Returns: The id of the newly added curve map data in the file.
 **/
gint
gwy_file_add_cmap(GwyFile *file,
                  GwyLawn *lawn,
                  GwyField *preview)
{
    g_return_val_if_fail(GWY_IS_FILE(file), -1);
    gint id = add_data_object(file, lawn, GWY_FILE_CMAP);
    if (id < 0)
        return id;

    GQuark quark = gwy_file_key_cmap_picture(id);
    GwyDict *container = GWY_DICT(file);
    if (preview) {
        if (GWY_IS_FIELD(preview))
            gwy_dict_set_object(container, quark, preview);
        preview = NULL;
    }
    if (!preview) {
        if (!gwy_dict_gis_object(container, quark, &preview) || !GWY_IS_FIELD(preview)) {
            preview = _gwy_app_create_lawn_preview_field(lawn);
            gwy_dict_pass_object(container, quark, preview);
        }
    }
    return id;
}

/**
 * gwy_file_remove:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data in file.
 *
 * Removes a data object from a file data contained together with all auxiliary information.
 *
 * The function can be used with files in construction (see gwy_dict_new_in_construction()), although usually
 * there is no reason for it.
 **/
void
gwy_file_remove(GwyFile *file,
                GwyDataKind data_kind,
                gint id)
{
    g_return_if_fail(data_kind < (guint)GWY_FILE_N_KINDS);
    /* Remove the main data item first. This should, hopefully, cause everything to discard the data and stop caring
     * about it. In particular, everything should ignore the subsequent cleanup of auxiliary items. */
    GQuark quark = data_kind_spec[data_kind].get_key(id);
    g_return_if_fail(quark);
    gwy_dict_remove(GWY_DICT(file), quark);
    const gchar *prefix = g_quark_to_string(quark);
    gwy_dict_remove_by_prefix(GWY_DICT(file), prefix);
}

/**
 * gwy_file_remove_selections:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of image-like data in file.
 *
 * Removes all selections associated with image-like data from a file.
 *
 * This is the preferred selection handling after in-place changes in data geometry as they have generally
 * unpredictable effects on selections. The currently active tool generally needs a selection and will be forced to
 * re-create it.
 *
 * The function can be used with files in construction (see gwy_dict_new_in_construction()), although usually
 * there is no reason for it.
 **/
void
gwy_file_remove_selections(GwyFile *file,
                           GwyDataKind data_kind,
                           gint id)
{
    GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_SELECTION, .suffix = NULL };
    const gchar *prefix = gwy_file_form_string_key(&parsed);
    g_return_if_fail(prefix);
    gwy_dict_remove_by_prefix(GWY_DICT(file), prefix);
}

/**
 * gwy_file_remove_logs:
 * @file: A data file container.
 *
 * Removes all data processing logs from a file.
 **/
void
gwy_file_remove_logs(GwyFile *file)
{
    g_return_if_fail(GWY_IS_FILE(file));
    GwyFileKeyParsed parsed = { .piece = GWY_FILE_PIECE_LOG, .suffix = NULL };
    GwyFilePrivate *priv = file->priv;
    GwyDict *container = GWY_DICT(file);
    for (guint data_kind = 0; data_kind < GWY_FILE_N_KINDS; data_kind++) {
        GArray *ids = priv->data_kind_info->ids;
        guint n = ids->len;
        parsed.data_kind = data_kind;
        for (guint j = 0; j < n; j++) {
            parsed.id = g_array_index(ids, gint, j);
            gwy_dict_remove(container, gwy_file_form_key(&parsed));
        }
    }
}

/* XXX GTK3: Not nice for bindings; there is no way to tell gobject-introspection the array is -1-terminated. However,
 * this style of return value is used in the codebase and can be auto-transformed to the new function. */
/**
 * gwy_file_get_ids:
 * @file: A data file container.
 * @data_kind: Type of data object.
 *
 * Gets the list of ids of all objects of given kind in a file.
 *
 * When there are no data objects of given kind, a non-%NULL array is still returned. It contains the single value -1.
 *
 * The file must not be in construction.
 *
 * Returns: A newly allocated, −1-terminated list of all the ids.
 **/
gint*
gwy_file_get_ids(GwyFile *file,
                 GwyDataKind data_kind)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(data_kind < (guint)GWY_FILE_N_KINDS, NULL);
    GwyFilePrivate *priv = file->priv;
    GArray *ids = priv->data_kind_info[data_kind].ids;
    if (!ids) {
        gint *retval = g_new(gint, 1);
        retval[0] = -1;
        return retval;
    }
    gint *retval = g_new(gint, ids->len+1);
    gwy_assign(retval, &g_array_index(ids, gint, 0), ids->len);
    retval[ids->len] = -1;
    return retval;
}

/**
 * gwy_file_get_ndata:
 * @file: A data file container.
 * @data_kind: Type of data object.
 *
 * Gets the number of data objects of a given kind in a file.
 *
 * This is the preferred function for checking whether there are any data of given kind at all.
 *
 * The file must not be in construction.
 *
 * Returns: The number of data items.
 **/
guint
gwy_file_get_ndata(GwyFile *file,
                   GwyDataKind data_kind)
{
    g_return_val_if_fail(GWY_IS_FILE(file), 0);
    g_return_val_if_fail(data_kind < (guint)GWY_FILE_N_KINDS, 0);
    GwyFilePrivate *priv = file->priv;
    GArray *ids = priv->data_kind_info[data_kind].ids;
    return ids ? ids->len : 0;
}

/* XXX GTK3: This is a function nice for bindings, but incompatible with existing code which assumes the -1 terminated
 * new array.
 * XXX: If we change the internal representation (for instance to GSequence), it becomes quite difficult to return
 * a const array. So returning a newly allocated array is the only future-proof option. */
/**
 * gwy_file_get_idsv:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @n: (optional) (inout): Location to store the number of returned ids.
 *
 * Gets the list of ids of all objects of given kind in a file.
 *
 * The returned pointer remains valid only as long as @file exists and data objects of kind @data_kind are not added
 * or removed. The array is not −1-terminated so unless you obtain the number of data objects by other means you need
 * to get @n.
 *
 * The file must not be in construction.
 *
 * Returns: A newly allocated list of all the ids with @n elements.
 **/
gint*
gwy_file_get_idsv(GwyFile *file,
                  GwyDataKind data_kind,
                  guint *n)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(data_kind < (guint)GWY_FILE_N_KINDS, NULL);
    GwyFilePrivate *priv = file->priv;
    GArray *ids = priv->data_kind_info[data_kind].ids;
    if (!ids) {
        if (n)
            *n = 0;
        return NULL;
    }

    if (n)
        *n = ids->len;

    return g_memdup2(ids->data, ids->len*sizeof(gint));
}

/**
 * gwy_file_get_data:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of data in file.
 *
 * Gets the primary data object of specified type and id.
 *
 * The type of the returned object is whatever is appropriate for given data kind. This function is mainly useful
 * when you need to check generically a data object for some data management purpose. For data processing, usually
 * functions such as gwy_file_get_image() or gwy_file_get_volume() are preferred as they return a specific type.
 *
 * It is not an error to ask for an object which does not exist in the file. The function returns %NULL in such case.
 * However, @data_kind and @id must be valid in principle.
 *
 * Returns: (transfer none) (nullable):
 *          The main data object in the file corresponding to @data_kind and @id, or %NULL if it does not exist.
 **/
GObject*
gwy_file_get_data(GwyFile *file,
                  GwyDataKind data_kind,
                  gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail((guint)data_kind < GWY_FILE_N_KINDS, NULL);
    g_return_val_if_fail(id >= 0, NULL);

    GQuark quark = data_kind_spec[data_kind].get_key(id);
    GObject *object = NULL;
    gwy_dict_gis_object(GWY_DICT(file), quark, &object);
    return object;
}

/**
 * gwy_file_set_image:
 * @file: A data file container.
 * @id: Numerical id of an image in @file.
 * @field: (transfer none): Data field representing an image.
 *
 * Sets a data field as the image in a data file container with no ownership transfer.
 **/
void
gwy_file_set_image(GwyFile *file,
                   gint id,
                   GwyField *field)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_FIELD(field));
    g_return_if_fail(id >= 0);
    gwy_dict_set_object(GWY_DICT(file), gwy_file_key_image(id), field);
}

/**
 * gwy_file_pass_image:
 * @file: A data file container.
 * @id: Numerical id of an image in @file.
 * @field: (transfer full): Data field representing an image.
 *
 * Sets a data field as the image in a data file container passing the ownership to the file.
 **/
void
gwy_file_pass_image(GwyFile *file,
                    gint id,
                    GwyField *field)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_FIELD(field));
    g_return_if_fail(id >= 0);
    gwy_dict_pass_object(GWY_DICT(file), gwy_file_key_image(id), field);
}

/**
 * gwy_file_get_image:
 * @file: A data file container.
 * @id: Numerical id of an image in @file.
 *
 * Gets a data field representing an image in a data file container.
 *
 * It is not an error to ask for an image which does not exist in the file. The function returns %NULL in such case.
 * However, @id must be valid (non-negative).
 *
 * Returns: (transfer none) (nullable): Data field representing an image, or %NULL.
 **/
GwyField*
gwy_file_get_image(GwyFile *file,
                   gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    GwyField *field = NULL;
    gwy_dict_gis_object(GWY_DICT(file), gwy_file_key_image(id), &field);
    return field;
}

/**
 * gwy_file_set_image_mask:
 * @file: A data file container.
 * @id: Numerical id of an image mask in @file.
 * @mask: (transfer none) (nullable): Data field representing a mask.
 *
 * Sets a number field as the image mask in a data file container with no ownership transfer.
 *
 * If @mask is passed as %NULL, any existing mask object is removed.
 **/
void
gwy_file_set_image_mask(GwyFile *file,
                        gint id,
                        GwyNield *mask)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(!mask || GWY_IS_NIELD(mask));
    g_return_if_fail(id >= 0);
    if (mask)
        gwy_dict_set_object(GWY_DICT(file), gwy_file_key_image_mask(id), mask);
    else
        gwy_dict_remove(GWY_DICT(file), gwy_file_key_image_mask(id));
}

/**
 * gwy_file_pass_image_mask:
 * @file: A data file container.
 * @id: Numerical id of an image mask in @file.
 * @mask: (transfer full): Data field representing a mask.
 *
 * Sets a number field as the image mask in a data file container passing the ownership to the file.
 **/
void
gwy_file_pass_image_mask(GwyFile *file,
                         gint id,
                         GwyNield *mask)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_NIELD(mask));
    g_return_if_fail(id >= 0);
    gwy_dict_pass_object(GWY_DICT(file), gwy_file_key_image_mask(id), mask);
}

/**
 * gwy_file_get_image_mask:
 * @file: A data file container.
 * @id: Numerical id of an image mask in @file.
 *
 * Gets a number field representing the image mask in a data file container.
 *
 * It is not an error to ask for a mask which does not exist in the file. The function returns %NULL in such case.
 * However, @id must be valid (non-negative).
 *
 * Returns: (transfer none) (nullable): Data field representing the mask, or %NULL.
 **/
GwyNield*
gwy_file_get_image_mask(GwyFile *file,
                        gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    GwyNield *nield = NULL;
    gwy_dict_gis_object(GWY_DICT(file), gwy_file_key_image_mask(id), &nield);
    return nield;
}

/**
 * gwy_file_set_graph:
 * @file: A data file container.
 * @id: Numerical id of a graph in @file.
 * @gmodel: (transfer none): Graph model representing a graph.
 *
 * Sets a graph model as the graph in a data file container with no ownership transfer.
 **/
void
gwy_file_set_graph(GwyFile *file,
                   gint id,
                   GwyGraphModel *gmodel)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_GRAPH_MODEL(gmodel));
    g_return_if_fail(id >= 0);
    gwy_dict_set_object(GWY_DICT(file), gwy_file_key_graph(id), gmodel);
}

/**
 * gwy_file_pass_graph:
 * @file: A data file container.
 * @id: Numerical id of a graph in @file.
 * @gmodel: (transfer full): Graph model representing a graph.
 *
 * Sets a graph model as the graph in a data file container passing the ownership to the file.
 **/
void
gwy_file_pass_graph(GwyFile *file,
                    gint id,
                    GwyGraphModel *gmodel)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_GRAPH_MODEL(gmodel));
    g_return_if_fail(id >= 0);
    gwy_dict_pass_object(GWY_DICT(file), gwy_file_key_graph(id), gmodel);
}

/**
 * gwy_file_get_graph:
 * @file: A data file container.
 * @id: Numerical id of a graph in @file.
 *
 * Gets a graph model representing a graph in a data file container.
 *
 * It is not an error to ask for a graph which does not exist in the file. The function returns %NULL in such case.
 * However, @id must be valid (non-negative).
 *
 * Returns: (transfer none) (nullable): Graph model representing a graph, or %NULL.
 **/
GwyGraphModel*
gwy_file_get_graph(GwyFile *file,
                   gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    GwyGraphModel *gmodel = NULL;
    gwy_dict_gis_object(GWY_DICT(file), gwy_file_key_graph(id), &gmodel);
    return gmodel;
}

/**
 * gwy_file_set_volume:
 * @file: A data file container.
 * @id: Numerical id of volume data in @file.
 * @brick: (transfer none): Data brick representing volume data.
 *
 * Sets a data brick as the volume data in a data file container with no ownership transfer.
 **/
void
gwy_file_set_volume(GwyFile *file,
                    gint id,
                    GwyBrick *brick)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_BRICK(brick));
    g_return_if_fail(id >= 0);
    gwy_dict_set_object(GWY_DICT(file), gwy_file_key_volume(id), brick);
}

/**
 * gwy_file_pass_volume:
 * @file: A data file container.
 * @id: Numerical id of volume data in @file.
 * @brick: (transfer full): Data brick representing volume data.
 *
 * Sets a data brick as the volume data in a data file container passing the ownership to the file.
 **/
void
gwy_file_pass_volume(GwyFile *file,
                     gint id,
                     GwyBrick *brick)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_BRICK(brick));
    g_return_if_fail(id >= 0);
    gwy_dict_pass_object(GWY_DICT(file), gwy_file_key_volume(id), brick);
}

/**
 * gwy_file_get_volume:
 * @file: A data file container.
 * @id: Numerical id of volume data in @file.
 *
 * Gets a data brick representing volume data in a data file container.
 *
 * It is not an error to ask for volume data which does not exist in the file. The function returns %NULL in such case.
 * However, @id must be valid (non-negative).
 *
 * Returns: (transfer none) (nullable): Data brick representing volume data, or %NULL.
 **/
GwyBrick*
gwy_file_get_volume(GwyFile *file,
                    gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    GwyBrick *brick = NULL;
    gwy_dict_gis_object(GWY_DICT(file), gwy_file_key_volume(id), &brick);
    return brick;
}

/**
 * gwy_file_set_spectra:
 * @file: A data file container.
 * @id: Numerical id of single point spectra in @file.
 * @spectra: (transfer none): Spectra object representing single point spectra.
 *
 * Sets a spectra object as the single point spectra in a data file container with no ownership transfer.
 **/
void
gwy_file_set_spectra(GwyFile *file,
                     gint id,
                     GwySpectra *spectra)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_SPECTRA(spectra));
    g_return_if_fail(id >= 0);
    gwy_dict_set_object(GWY_DICT(file), gwy_file_key_spectra(id), spectra);
}

/**
 * gwy_file_pass_spectra:
 * @file: A data file container.
 * @id: Numerical id of single point spectra in @file.
 * @spectra: (transfer full): Spectra object representing single point spectra.
 *
 * Sets a spectra object as the single point spectra in a data file container passing the ownership to the file.
 **/
void
gwy_file_pass_spectra(GwyFile *file,
                      gint id,
                      GwySpectra *spectra)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_SPECTRA(spectra));
    g_return_if_fail(id >= 0);
    gwy_dict_pass_object(GWY_DICT(file), gwy_file_key_spectra(id), spectra);
}

/**
 * gwy_file_get_spectra:
 * @file: A data file container.
 * @id: Numerical id of single point spectra in @file.
 *
 * Gets a spectra object representing single point spectra in a data file container.
 *
 * It is not an error to ask for single point spectra which does not exist in the file. The function returns %NULL in
 * such case. However, @id must be valid (non-negative).
 *
 * Returns: (transfer none) (nullable): Spectra object representing single point spectra, or %NULL.
 **/
GwySpectra*
gwy_file_get_spectra(GwyFile *file,
                     gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    GwySpectra *spectra = NULL;
    gwy_dict_gis_object(GWY_DICT(file), gwy_file_key_spectra(id), &spectra);
    return spectra;
}

/**
 * gwy_file_set_xyz:
 * @file: A data file container.
 * @id: Numerical id of XYZ data in @file.
 * @surface: (transfer none): Surface object representing XYZ data.
 *
 * Sets a surface object as the XYZ data in a data file container with no ownership transfer.
 **/
void
gwy_file_set_xyz(GwyFile *file,
                 gint id,
                 GwySurface *surface)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_SURFACE(surface));
    g_return_if_fail(id >= 0);
    gwy_dict_set_object(GWY_DICT(file), gwy_file_key_xyz(id), surface);
}

/**
 * gwy_file_pass_xyz:
 * @file: A data file container.
 * @id: Numerical id of XYZ data in @file.
 * @surface: (transfer full): Surface object representing XYZ data.
 *
 * Sets a surface object as the XYZ data in a data file container passing the ownership to the file.
 **/
void
gwy_file_pass_xyz(GwyFile *file,
                  gint id,
                  GwySurface *surface)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_SURFACE(surface));
    g_return_if_fail(id >= 0);
    gwy_dict_pass_object(GWY_DICT(file), gwy_file_key_xyz(id), surface);
}

/**
 * gwy_file_get_xyz:
 * @file: A data file container.
 * @id: Numerical id of XYZ data in @file.
 *
 * Gets a surface object representing XYZ data in a data file container.
 *
 * It is not an error to ask for XYZ data which does not exist in the file. The function returns %NULL in such case.
 * However, @id must be valid (non-negative).
 *
 * Returns: (transfer none) (nullable): Surface object representing XYZ data, or %NULL.
 **/
GwySurface*
gwy_file_get_xyz(GwyFile *file,
                 gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    GwySurface *surface = NULL;
    gwy_dict_gis_object(GWY_DICT(file), gwy_file_key_xyz(id), &surface);
    return surface;
}

/**
 * gwy_file_set_cmap:
 * @file: A data file container.
 * @id: Numerical id of a curve map in @file.
 * @lawn: (transfer none): Data lawn representing a curve map.
 *
 * Sets a data lawn as the a curve map in a data file container with no ownership transfer.
 **/
void
gwy_file_set_cmap(GwyFile *file,
                  gint id,
                  GwyLawn *lawn)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_LAWN(lawn));
    g_return_if_fail(id >= 0);
    gwy_dict_set_object(GWY_DICT(file), gwy_file_key_cmap(id), lawn);
}

/**
 * gwy_file_pass_cmap:
 * @file: A data file container.
 * @id: Numerical id of a curve map in @file.
 * @lawn: (transfer full): Data lawn representing a curve map.
 *
 * Sets a data lawn as the a curve map in a data file container passing the ownership to the file.
 **/
void
gwy_file_pass_cmap(GwyFile *file,
                   gint id,
                   GwyLawn *lawn)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_LAWN(lawn));
    g_return_if_fail(id >= 0);
    gwy_dict_pass_object(GWY_DICT(file), gwy_file_key_cmap(id), lawn);
}

/**
 * gwy_file_get_cmap:
 * @file: A data file container.
 * @id: Numerical id of a curve map in @file.
 *
 * Gets a data lawn representing a curve map in a data file container.
 *
 * It is not an error to ask for a curve map which does not exist in the file. The function returns %NULL in such case.
 * However, @id must be valid (non-negative).
 *
 * Returns: (transfer none) (nullable): Data lawn representing a curve map, or %NULL.
 **/
GwyLawn*
gwy_file_get_cmap(GwyFile *file,
                  gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    GwyLawn *lawn = NULL;
    gwy_dict_gis_object(GWY_DICT(file), gwy_file_key_cmap(id), &lawn);
    return lawn;
}

static void
set_title(GwyFile *file, GwyDataKind data_kind, gint id,
          const gchar *const_name, gchar *allocated_name)
{
    /* TODO: We should clean up this mess. For now, abstract it. */
    GwyDict *container = GWY_DICT(file);
    GQuark quark = 0;

    GObject *object;
    if (!gwy_dict_gis_object(container, data_kind_spec[data_kind].get_key(id), &object)) {
        g_warning("Setting title of a non-existent data item.");
    }

    if (data_kind == GWY_FILE_IMAGE)
        quark = gwy_file_key_image_title(id);
    else if (data_kind == GWY_FILE_GRAPH)
        g_object_set(object, "title", const_name, NULL);
    else if (data_kind == GWY_FILE_SPECTRA)
        gwy_spectra_set_title(GWY_SPECTRA(object), const_name);
    else if (data_kind == GWY_FILE_VOLUME)
        quark = gwy_file_key_volume_title(id);
    else if (data_kind == GWY_FILE_XYZ)
        quark = gwy_file_key_xyz_title(id);
    else if (data_kind == GWY_FILE_CMAP)
        quark = gwy_file_key_cmap_title(id);
    else {
        g_assert_not_reached();
    }

    if (quark) {
        if (allocated_name) {
            gwy_dict_set_string(container, quark, allocated_name);
            allocated_name = NULL;
        }
        else
            gwy_dict_set_const_string(container, quark, const_name);
    }
    g_free(allocated_name);
}

/**
 * gwy_file_set_title:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of an existing data object in @file.
 * @name: (nullable): New title to set. It can be %NULL to use something like ‘Untitled’.
 * @numbered: %TRUE to append the numeric id to the title, %FALSE to set it as-is.
 *
 * Sets the title of a data object in a file data container.
 *
 * This is a convenience wrapper, abstracting setting the title for different data types.
 *
 * Depending on @numbered, the id is possibly appended to the name. When @name is %NULL, id is always appended,
 * regardless of @numbered.
 *
 * Passing @numbered as %TRUE is generally recommended for derived data computed in modules with generic names such as
 * ‘Drift-corrected’. Without numbering there would be lots of data with the same name, making it difficult to
 * distinguish them.
 *
 * For channels in imported files, which generally carry file-unique names, it is fine to not append numbers. See also
 * gwy_file_pass_title() which may be more useful on file import.
 *
 * The function can be used with files in construction (see gwy_dict_new_in_construction()), provided that the
 * tiltle is set after the primary data object.
 **/
void
gwy_file_set_title(GwyFile *file,
                   GwyDataKind data_kind,
                   gint id,
                   const gchar *name,
                   gboolean numbered)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(data_kind < (guint)GWY_FILE_N_KINDS);
    g_return_if_fail(id >= 0);

    if (!name) {
        name = _("Untitled");
        numbered = TRUE;
    }
    gchar *freeme = NULL;
    if (numbered)
        name = freeme = g_strdup_printf("%s %d", name, id);

    set_title(file, data_kind, id, name, freeme);
}

/**
 * gwy_file_pass_title:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of an existing data object in @file.
 * @name: (transfer full): Title to set. It may not be %NULL.
 *
 * Sets the title of a data object in a file data container, passing string ownership.
 *
 * This is a convenience wrapper, abstracting setting the title for different data types.
 *
 * Unlike in gwy_file_set_title(), @name is always used unmodified and may not be %NULL. Uts ownership is passed to
 * the file (or other object).
 *
 * The function can be used with files in construction (see gwy_dict_new_in_construction()), provided that the
 * tiltle is set after the primary data object.
 **/
void
gwy_file_pass_title(GwyFile *file,
                    GwyDataKind data_kind,
                    gint id,
                    gchar *name)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(data_kind < (guint)GWY_FILE_N_KINDS);
    g_return_if_fail(id >= 0);
    set_title(file, data_kind, id, name, name);
}

/**
 * gwy_file_get_title:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 *
 * Gets the title of a data object in a file data container.
 *
 * If no title is set this function can return %NULL. Use gwy_file_get_display_title() if you want to always get
 * some useful identifying string.
 *
 * Returns: (nullable):
 *          The title as a string owned by the corresponding data object.
 **/
const gchar*
gwy_file_get_title(GwyFile *file,
                   GwyDataKind data_kind,
                   gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(data_kind < (guint)GWY_FILE_N_KINDS, NULL);
    g_return_val_if_fail(id >= 0, NULL);

    GQuark quark = data_kind_spec[data_kind].get_key(id);
    GwyDict *container = GWY_DICT(file);
    GObject *object = NULL;
    gwy_dict_gis_object(container, quark, &object);

    /* TODO: We should clean up this mess. For now, abstract it. */
    const gchar *title = NULL;
    if (data_kind == GWY_FILE_IMAGE)
        gwy_dict_gis_string(container, gwy_file_key_image_title(id), &title);
    else if (data_kind == GWY_FILE_GRAPH) {
        if (object)
            title = gwy_graph_model_get_title(GWY_GRAPH_MODEL(object));
    }
    else if (data_kind == GWY_FILE_SPECTRA) {
        if (object)
            title = gwy_spectra_get_title(GWY_SPECTRA(object));
    }
    else if (data_kind == GWY_FILE_VOLUME)
        gwy_dict_gis_string(container, gwy_file_key_volume_title(id), &title);
    else if (data_kind == GWY_FILE_XYZ)
        gwy_dict_gis_string(container, gwy_file_key_xyz_title(id), &title);
    else if (data_kind == GWY_FILE_CMAP)
        gwy_dict_gis_string(container, gwy_file_key_cmap_title(id), &title);
    else {
        g_assert_not_reached();
    }

    return title;
}

/**
 * gwy_file_get_display_title:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 *
 * Gets a suitable title of a data object in a file data container.
 *
 * The function returns a non-%NULL string even if no data title has been set or is empty. It is something like
 * ‘Untitled 123’ in such case. It makes it useful for having at least some identifying label for all data objects,
 * regardless if they have any title set. However, if you really need to know whether a title has been set use
 * gwy_file_get_title().
 *
 * Returns: The title as a newly allocated string.
 **/
gchar*
gwy_file_get_display_title(GwyFile *file,
                           GwyDataKind data_kind,
                           gint id)
{
    const gchar *title = gwy_file_get_title(file, data_kind, id);

    if (title)
        return g_strdup(title);
    return g_strdup_printf("%s %d", _("Untitled"), id);
}

/**
 * gwy_file_find_by_title:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @titleglob: (nullable):
 *             Pattern, as used by #GPatternSpec, to match the data titles against.
 *
 * Gets the list of all data objects in a data file container whose titles match the specified pattern.
 *
 * If titleglob is %NULL the function is identical to gwy_file_get_ids().
 *
 * For data objects with no title the synthetic title produced by gwy_file_get_display_title() is used.
 *
 * Returns: A newly allocated, −1-terminated list of all the ids.
 **/
gint*
gwy_file_find_by_title(GwyFile *file,
                       GwyDataKind data_kind,
                       const gchar *titleglob)
{
    if (!titleglob || !*titleglob)
        return gwy_file_get_ids(file, data_kind);

    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    g_return_val_if_fail(data_kind < (guint)GWY_FILE_N_KINDS, NULL);

    GwyFilePrivate *priv = file->priv;
    GArray *ids = priv->data_kind_info[data_kind].ids;
    if (!ids) {
        gint *retval = g_new(gint, 1);
        retval[0] = -1;
        return retval;
    }

    GPatternSpec *pattern = g_pattern_spec_new(titleglob);
    GArray *filtered_ids = g_array_new(FALSE, FALSE, sizeof(gint));
    guint n = ids->len;
    for (guint i = 0; i < n; i++) {
        gint id = g_array_index(ids, gint, i);
        const gchar *title = gwy_file_get_title(file, data_kind, id);
        if (title) {
            if (g_pattern_match_string(pattern, title))
                g_array_append_val(filtered_ids, id);
        }
        else {
            /* This is the traditional behaviour – use the display title if no real title is set. */
            gchar *t = g_strdup_printf("%s %d", _("Untitled"), id);
            if (g_pattern_match_string(pattern, t))
                g_array_append_val(filtered_ids, id);
            g_free(t);
        }
    }
    g_pattern_spec_free(pattern);

    gint id = -1;
    g_array_append_val(filtered_ids, id);
    return (gint*)g_array_free(filtered_ids, FALSE);
}

/**
 * gwy_file_set_visible:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 * @visible: %TRUE to display the data in a window, %FALSE to hide it.
 *
 * Sets the visibility of a data item in a data file container.
 *
 * The function can be used to either just mark the visibility status of data objects in files or actually control the
 * visibility in files open in the GUI. Closing the last window displaying any data of a particular file can close the
 * file entirely.
 *
 * The function can be used with files in construction (see gwy_dict_new_in_construction()).
 **/
void
gwy_file_set_visible(GwyFile *file,
                     GwyDataKind data_kind,
                     gint id,
                     gboolean visible)
{
    g_return_if_fail(GWY_IS_FILE(file));
    GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_VISIBLE, .suffix = NULL };
    GQuark quark = gwy_file_form_key(&parsed);
    g_return_if_fail(quark);
    if (visible)
        gwy_dict_set_boolean(GWY_DICT(file), quark, TRUE);
    else
        gwy_dict_remove(GWY_DICT(file), quark);
    /* And data browser does the rest if the GUI is showing the file. */
}

/**
 * gwy_file_get_visible:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 *
 * Gets the visivility status of a data object in a data file container.
 *
 * The function returning %TRUE does not mean the data object is actually displayed in some window. It simply returns
 * the value of the flag and can be used with standalone files with no GUI running.
 *
 * Returns: %TRUE if the object is marked visible, %FALSE otherwise.
 **/
gboolean
gwy_file_get_visible(GwyFile *file,
                     GwyDataKind data_kind,
                     gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), FALSE);
    GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_VISIBLE, .suffix = NULL };
    GQuark quark = gwy_file_form_key(&parsed);
    g_return_val_if_fail(quark, FALSE);
    gboolean visible = FALSE;
    gwy_dict_gis_boolean(GWY_DICT(file), quark, &visible);
    return visible;
}

/**
 * gwy_file_pass_meta:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 * @meta: (transfer full): A data container holding data object's metadata. The reference is passed to @file.
 *
 * Sets the metadata for a data object in a data file container.
 *
 * This is a convenience wrapper for gwy_dict_pass_object() with slightly more checking. It can be used with
 * files in construction (see gwy_dict_new_in_construction()) and is most useful in file import modules.
 *
 * If @meta has still the in-construction flag set, the function calls gwy_dict_finish_construction() on it.
 * Preferably set the metadata after @meta has been filled with values.
 *
 * Analogous function gwy_file_set_meta() is missing intentionally. It creates too big temptation to use the same
 * object as metadata for multiple data, which is an error. Use #GwyFile functions to get the key and #GwyDict
 * functions to set the object if you really need to keep the metadata around after puttng it to the file. The useful
 * idioms are the following.
 *
 * With a function which creates metadata:
 * |[
 * gwy_file_pass_meta(file, GWY_FILE_IMAGE, id, create_metadata(...));
 * ]|
 *
 * With the same metadata for many data items:
 * |[
 * meta = create_metadata(...);
 * ...
 * gwy_file_pass_meta(file, GWY_FILE_IMAGE, id1, gwy_dict_copy(meta));
 * gwy_file_pass_meta(file, GWY_FILE_IMAGE, id2, gwy_dict_copy(meta));
 * gwy_file_pass_meta(file, GWY_FILE_IMAGE, id3, gwy_dict_copy(meta));
 * ...
 * g_object_unref(meta);
 * ]|
 *
 * With optional metadata:
 * |[
 * meta = gwy_dict_new();
 * ...
 * if (gwy_dict_get_n_items(meta))
 *     gwy_file_pass_meta(file, GWY_FILE_IMAGE, id, meta));
 * else
 *     g_object_unref(meta);
 * ]|
 **/
void
gwy_file_pass_meta(GwyFile *file,
                   GwyDataKind data_kind,
                   gint id,
                   GwyDict *meta)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_DICT(meta));
    GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_META, .suffix = NULL };
    GQuark quark = gwy_file_form_key(&parsed);
    g_return_if_fail(quark);
    if (gwy_dict_is_being_constructed(GWY_DICT(meta)))
        gwy_dict_finish_construction(GWY_DICT(meta));
    gwy_dict_pass_object(GWY_DICT(file), quark, meta);
}

/**
 * gwy_file_get_meta:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 *
 * Gets the metadata for a data object in a data file container.
 *
 * Returns: (transfer none) (nullable):
 *          A data container holding data object's metadata.
 **/
GwyDict*
gwy_file_get_meta(GwyFile *file,
                  GwyDataKind data_kind,
                  gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), FALSE);
    GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_META, .suffix = NULL };
    GQuark quark = gwy_file_form_key(&parsed);
    g_return_val_if_fail(quark, FALSE);
    GwyDict *meta = NULL;
    gwy_dict_gis_object(GWY_DICT(file), quark, &meta);
    return meta;
}

/**
 * gwy_file_set_palette:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 * @name: (nullable):
 *        Gradient name. It can be %NULL to unset the gradient (causing the current default gradient to be used).
 *
 * Sets the gradient name for data visualisation in a data file container.
 **/
void
gwy_file_set_palette(GwyFile *file,
                     GwyDataKind data_kind,
                     gint id,
                     const gchar *name)
{
    g_return_if_fail(GWY_IS_FILE(file));
    GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_PALETTE, .suffix = NULL };
    GQuark quark = gwy_file_form_key(&parsed);
    g_return_if_fail(quark);
    if (name)
        gwy_dict_set_const_string(GWY_DICT(file), quark, name);
    else
        gwy_dict_remove(GWY_DICT(file), quark);
}

/**
 * gwy_file_get_palette:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 *
 * Gets the gradient name for data visualisation in a data file container.
 *
 * Returns: (nullable):
 *          The gradient name as a string owned by the file. It can be %NULL if no gradient is set.
 **/
const gchar*
gwy_file_get_palette(GwyFile *file,
                     GwyDataKind data_kind,
                     gint id)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_PALETTE, .suffix = NULL };
    GQuark quark = gwy_file_form_key(&parsed);
    g_return_val_if_fail(quark, NULL);
    const gchar *gradient = NULL;
    gwy_dict_gis_string(GWY_DICT(file), quark, &gradient);
    return gradient;
}

/**
 * gwy_file_set_color_mapping:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 * @mapping: Mapping type to set.
 *
 * Sets the colour mapping type for data visualisation in a data file container.
 **/
void
gwy_file_set_color_mapping(GwyFile *file,
                           GwyDataKind data_kind,
                           gint id,
                           GwyColorMappingType mapping)
{
    g_return_if_fail(GWY_IS_FILE(file));
    GwyFileKeyParsed parsed = {
        .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_COLOR_MAPPING, .suffix = NULL
    };
    GQuark quark = gwy_file_form_key(&parsed);
    g_return_if_fail(quark);
    gwy_dict_set_enum(GWY_DICT(file), quark, mapping);
}

/**
 * gwy_file_get_color_mapping:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 *
 * Gets the colour mapping type for data visualisation in a data file container.
 *
 * If no mapping is set, the current default mapping type from user settings is returned.
 *
 * Returns: The colour mapping type.
 **/
GwyColorMappingType
gwy_file_get_color_mapping(GwyFile *file,
                           GwyDataKind data_kind,
                           gint id)
{
    GwyColorMappingType mapping = GWY_COLOR_MAPPING_FULL;
    g_return_val_if_fail(GWY_IS_FILE(file), mapping);
    GwyFileKeyParsed parsed = {
        .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_COLOR_MAPPING, .suffix = NULL
    };
    GQuark quark = gwy_file_form_key(&parsed);
    g_return_val_if_fail(quark, mapping);
    if (!gwy_dict_gis_enum(GWY_DICT(file), quark, &mapping))
        gwy_dict_gis_enum_by_name(gwy_app_settings_get(), "/app/default-range-type", &mapping);
    return mapping;
}

/**
 * gwy_file_set_fixed_range:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 * @min: Fixed range minimum.
 * @max: Fixed range maximum.
 *
 * Sets the fixed colour mapping range for data visualisation in a data file container.
 **/
void
gwy_file_set_fixed_range(GwyFile *file,
                         GwyDataKind data_kind,
                         gint id,
                         gdouble min,
                         gdouble max)
{
    g_return_if_fail(GWY_IS_FILE(file));
    GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_RANGE, .suffix = NULL };
    parsed.suffix = "min";
    gwy_dict_set_double(GWY_DICT(file), gwy_file_form_key(&parsed), min);
    parsed.suffix = "max";
    gwy_dict_set_double(GWY_DICT(file), gwy_file_form_key(&parsed), max);
}

/**
 * gwy_file_get_fixed_range:
 * @file: A data file container.
 * @data_kind: Type of data object.
 * @id: Numerical id of a data object in @file.
 * @min: (out) (nullable): Location to store fixed range minimum.
 * @max: (out) (nullable): Location to store fixed range maximum.
 *
 * Gets the fixed colour mapping range for data visualisation in a data file container.
 *
 * It is possible (although unusual) to request just one of the values, or even none.
 *
 * Returns: %TRUE if all requested values were retrieved, %FALSE otherwise.
 **/
gboolean
gwy_file_get_fixed_range(GwyFile *file,
                         GwyDataKind data_kind,
                         gint id,
                         gdouble *min,
                         gdouble *max)
{
    g_return_val_if_fail(GWY_IS_FILE(file), FALSE);
    GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_RANGE, .suffix = NULL };
    gboolean ok = TRUE;
    if (min) {
        parsed.suffix = "min";
        if (!gwy_dict_gis_double(GWY_DICT(file), gwy_file_form_key(&parsed), min))
            ok = FALSE;
    }
    if (max) {
        parsed.suffix = "max";
        if (!gwy_dict_gis_double(GWY_DICT(file), gwy_file_form_key(&parsed), max))
            ok = FALSE;
    }
    return ok;
}

static void
sync_one_item(GwyDict *src, GwyDict *dest,
              GwyDataKind src_data_kind, GwyDataKind dest_data_kind,
              gint from_id, gint to_id,
              GwyFilePiece piece, const gchar *suffix,
              gboolean delete_too)
{
    GwyFileKeyParsed srcparsed = { .id = from_id, .data_kind = src_data_kind, .piece = piece, .suffix = suffix };
    GwyFileKeyParsed destparsed = { .id = to_id, .data_kind = dest_data_kind, .piece = piece, .suffix = suffix };
    GQuark srckey = gwy_file_form_key(&srcparsed), destkey = gwy_file_form_key(&destparsed);

    gboolean src_exists = gwy_dict_contains(src, srckey);
    if (!src_exists) {
        if (delete_too)
            gwy_dict_remove(dest, destkey);
        return;
    }
    gwy_dict_copy_value(src, dest, srckey, destkey);
}

static void
sync_one_file_piece(GwyDict *source, GwyDict *dest,
                    GwyDataKind src_data_kind, GwyDataKind dest_data_kind,
                    gint from_id, gint to_id,
                    GwyFilePiece piece, gboolean delete_too)
{
    if (piece == GWY_FILE_PIECE_PALETTE
        || piece == GWY_FILE_PIECE_MASK
        || piece == GWY_FILE_PIECE_MASK_COLOR
        || piece == GWY_FILE_PIECE_COLOR_MAPPING
        || piece == GWY_FILE_PIECE_REAL_SQUARE
        || piece == GWY_FILE_PIECE_PICTURE
        || piece == GWY_FILE_PIECE_META) {
        sync_one_item(source, dest, src_data_kind, dest_data_kind, from_id, to_id, piece, NULL, delete_too);
    }
    else if (piece == GWY_FILE_PIECE_TITLE) {
        const gchar *title = gwy_file_get_title(GWY_FILE(source), src_data_kind, from_id);
        if (title)
            gwy_file_set_title(GWY_FILE(dest), dest_data_kind, to_id, title, FALSE);
        else if (delete_too) {
            g_warning("Implement me! We do not have a generic function to unset the underlying title.");
        }
    }
    else if (piece == GWY_FILE_PIECE_RANGE) {
        sync_one_item(source, dest, src_data_kind, dest_data_kind, from_id, to_id,
                      GWY_FILE_PIECE_COLOR_MAPPING, NULL, delete_too);
        sync_one_item(source, dest, src_data_kind, dest_data_kind, from_id, to_id,
                      GWY_FILE_PIECE_RANGE, "min", delete_too);
        sync_one_item(source, dest, src_data_kind, dest_data_kind, from_id, to_id,
                      GWY_FILE_PIECE_RANGE, "max", delete_too);
    }
    else if (piece == GWY_FILE_PIECE_VIEW) {
        const gchar *suffixes[] = { "relative-size", "scale", "width", "height" };
        for (guint i = 0; i < G_N_ELEMENTS(suffixes); i++) {
            sync_one_item(source, dest, src_data_kind, dest_data_kind, from_id, to_id,
                          GWY_FILE_PIECE_VIEW, suffixes[i], delete_too);
        }
    }
    else if (piece == GWY_FILE_PIECE_SELECTION) {
        GwyFileKeyParsed srcparsed = { .id = from_id, .data_kind = src_data_kind, .piece = piece, .suffix = NULL };
        GwyFileKeyParsed destparsed = { .id = to_id, .data_kind = dest_data_kind, .piece = piece, .suffix = NULL };
        const gchar *srcprefix = gwy_file_form_string_key(&srcparsed);
        const gchar *destprefix = gwy_file_form_string_key(&destparsed);
        gwy_dict_transfer(source, dest, srcprefix, destprefix, TRUE, delete_too);
    }
    else {
        g_critical("Unhandled data piece type %u.", piece);
    }
}

static void
sync_all_items(GwyFile *source, GwyDataKind data_kind, gint id,
               GwyFile *destination, gint newid,
               gboolean visualisation_done)
{
    guint syncitems = 0;
    const DataKindSpec *spec = data_kind_spec + data_kind;
    for (guint i = 0; i < spec->npieces; i++) {
        GwyFilePiece piece = all_pieces[spec->pieces[i]].piece;
        if (piece < 32 && (!visualisation_done || piece != GWY_FILE_PIECE_PICTURE))
            syncitems |= (1u << piece);
    }
    if (syncitems)
        gwy_file_sync_items(source, data_kind, id, destination, data_kind, newid, syncitems, FALSE);
}

/**
 * gwy_file_copy_data:
 * @source: Source data file container.
 * @data_kind: Type of data object.
 * @id: Source numerical id of data object.
 * @destination: Destination data file container.
 *
 * Completely duplicates data in a data file container including all auxiliary information.
 *
 * The source and destination files may be the same file. Visibility is not copied; the new data object is created as
 * not visible.
 *
 * See also gwy_data_browser_copy_data() for somewhat a more high-level function.
 *
 * Returns: The id of the newly added data in file @destination.
 **/
gint
gwy_file_copy_data(GwyFile *source,
                   GwyDataKind data_kind,
                   gint id,
                   GwyFile *destination)
{
    g_return_val_if_fail(GWY_IS_FILE(source), -1);
    g_return_val_if_fail(GWY_IS_FILE(destination), -1);
    g_return_val_if_fail(data_kind < (guint)GWY_FILE_N_KINDS, -1);

    GQuark srcquark = data_kind_spec[data_kind].get_key(id);
    g_return_val_if_fail(srcquark, -1);

    GwyDict *srcdata = GWY_DICT(source);
    GObject *object = gwy_dict_get_object(srcdata, srcquark);
    GObject *otherobject = NULL;
    gboolean visualisation_done = FALSE;
    g_return_val_if_fail(object, -1);

    gint newid = -1;
    if (data_kind == GWY_FILE_IMAGE)
        newid = gwy_file_add_image(destination, gwy_field_copy(GWY_FIELD(object)));
    else if (data_kind == GWY_FILE_GRAPH)
        newid = gwy_file_add_graph(destination, gwy_graph_model_copy(GWY_GRAPH_MODEL(object)));
    else if (data_kind == GWY_FILE_SPECTRA)
        newid = gwy_file_add_spectra(destination, gwy_spectra_copy(GWY_SPECTRA(object)));
    else if (data_kind == GWY_FILE_VOLUME || data_kind == GWY_FILE_CMAP) {
        GwyFileKeyParsed parsed = { .id = id, .data_kind = data_kind, .piece = GWY_FILE_PIECE_PICTURE, .suffix = NULL };
        if (gwy_dict_gis_object(srcdata, gwy_file_form_key(&parsed), &otherobject))
            otherobject = G_OBJECT(gwy_serializable_copy(GWY_SERIALIZABLE(otherobject)));
        if (data_kind == GWY_FILE_VOLUME)
            newid = gwy_file_add_volume(destination, gwy_brick_copy(GWY_BRICK(object)), (GwyField*)otherobject);
        else
            newid = gwy_file_add_cmap(destination, gwy_lawn_copy(GWY_LAWN(object)), (GwyField*)otherobject);
        visualisation_done = TRUE;
    }
    else if (data_kind == GWY_FILE_XYZ)
        newid = gwy_file_add_xyz(destination, gwy_surface_copy(GWY_SURFACE(object)));
    else {
        g_assert_not_reached();
    }

    sync_all_items(source, data_kind, id, destination, newid, visualisation_done);

    return newid;
}

/**
 * gwy_file_sync_items:
 * @source: Source data file container.
 * @src_data_kind: Type of source data object.
 * @from_id: Source numerical id of data object.
 * @destination: Destination data file container.
 * @dest_data_kind: Type of destination data object.
 * @to_id: Destination numerical id of data object.
 * @items: Flags indicating the items to synchronise.
 * @delete_too: %TRUE to delete items in target if source does not contain them, %FALSE to copy only.
 *
 * Synchronizes auxiliary data items between data file containers.
 *
 * The most common use is with @dest_data_kind = @src_data_kind. Nevertheless, some items make sense for multiple data
 * kinds and can be used with @dest_data_kind ≠ @src_data_kind. For example, the false colour gradient can be
 * synchronised between an image and volume data.
 **/
void
gwy_file_sync_items(GwyFile *source,
                    GwyDataKind src_data_kind,
                    gint from_id,
                    GwyFile *destination,
                    GwyDataKind dest_data_kind,
                    gint to_id,
                    GwyFileItemFlags items,
                    gboolean delete_too)
{
    g_return_if_fail(GWY_IS_FILE(source));
    g_return_if_fail(GWY_IS_FILE(destination));
    g_return_if_fail(from_id >= 0 && to_id >= 0);
    if (source == destination && from_id == to_id)
        return;

    GwyDict *src = GWY_DICT(source), *dest = GWY_DICT(destination);
    for (GwyFilePiece p = 0; items; p++) {
        if (items & 1)
            sync_one_file_piece(src, dest, src_data_kind, dest_data_kind, from_id, to_id, p, delete_too);
        items >>= 1;
    }
}

/**
 * gwy_file_merge:
 * @file: A data file container.
 * @otherfile: (transfer full):
 *             A file to merge into @file. It must not be @file itself and must not be managed by the data browser.
 *
 * Merges a data file container into another.
 *
 * The caller passes its reference on @otherfile to the function. The function releases the reference when it is
 * finished. The underlying assumption is that @otherfile is then destroyed. This enables an efficient merge as
 * objects in @otherfile can be simply added to @file and expensive copying is avoided. Keeping @otherfile around
 * after merging is not recommended.
 *
 * Use gwy_file_copy() on @otherfile if you want a merge with data duplication.
 **/
void
gwy_file_merge(GwyFile *file,
               GwyFile *otherfile)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(GWY_IS_FILE(otherfile));
    g_return_if_fail(otherfile != file);
    g_return_if_fail(!otherfile->priv->is_managed);

    g_warning("Implement me!");
    /* TODO. This is a long chunk of code, maybe do it all in a separate source file. */
}

static guint
append_uint(gchar *buf, guint u)
{
    static const gchar digits[11] = "0123456789";

    guint len = 0;
    do {
        buf[len++] = digits[u % 10];
        u /= 10;
    } while (u);

    for (guint i = 0; i < len/2; i++)
        GWY_SWAP(gchar, buf[i], buf[len-1 - i]);

    return len;
}

/**
 * gwy_file_form_key:
 * @parsed: Broken down key in a data file.
 *
 * Constructs a data file quark key from its broken down representation.
 *
 * Although the function does not accept complete nonsense, it can form keys corresponding to data type and piece
 * combinations which do not actually exists. In other words, if @parsed was obtained using gwy_file_parse_key(), this
 * function can be used to construct the string (quark) key back. However, a key constructed by this function is not
 * guaranteed to be parseable using gwy_file_parse_key().
 *
 * See gwy_file_form_string_key() for a string version.
 *
 * Returns: The corresponding key as a quark.
 **/
GQuark
gwy_file_form_key(const GwyFileKeyParsed *parsed)
{
    g_return_val_if_fail(parsed, 0);

    if (parsed->piece == GWY_FILE_PIECE_FILENAME) {
        g_assert(parsed->data_kind == GWY_FILE_NONE);
        g_assert(parsed->id < 0);
        g_assert(!parsed->suffix);
        return g_quark_from_static_string("/filename");
    }

    g_return_val_if_fail(parsed->data_kind >= 0 && parsed->data_kind < (guint)GWY_FILE_N_KINDS, 0);
    g_return_val_if_fail(parsed->id >= 0, 0);

    const DataKindSpec *spec = data_kind_spec + parsed->data_kind;
    const gchar *piece_suffix = gwy_enum_to_string(parsed->piece, piece_suffixes, G_N_ELEMENTS(piece_suffixes));
    guint plen = strlen(spec->prefix), pslen = gwy_strlen0(piece_suffix), slen = gwy_strlen0(parsed->suffix);
    if (slen && !pslen) {
        g_warning("Key suffix given for unknown piece %u.", parsed->piece);
    }
    gchar key[plen + pslen + slen + 14];
    memcpy(key, spec->prefix, plen);
    key[plen] = '/';
    guint n = append_uint(key + plen + 1, parsed->id) + 1;
    if (pslen) {
        key[plen + n] = '/';
        memcpy(key + plen + n + 1, piece_suffix, pslen);
        pslen++;
    }
    if (slen) {
        key[plen + n + pslen] = '/';
        memcpy(key + plen + n + pslen + 1, parsed->suffix, slen);
        slen++;
    }
    key[plen + n + pslen + slen] = '\0';

    return g_quark_from_string(key);
}

/**
 * gwy_file_form_string_key:
 * @parsed: Broken down key in a data file.
 *
 * Constructs a data file string key from its broken down representation.
 *
 * See gwy_file_form_key() for a quark version and remarks.
 *
 * Returns: The corresponding key as a static string.
 **/
const gchar*
gwy_file_form_string_key(const GwyFileKeyParsed *parsed)
{
    g_return_val_if_fail(parsed, 0);

    if (parsed->piece == GWY_FILE_PIECE_FILENAME) {
        g_assert(parsed->data_kind == GWY_FILE_NONE);
        g_assert(parsed->id < 0);
        g_assert(!parsed->suffix);
        return "/filename";
    }

    /* Laundering through the quark effectively interns the string. */
    return g_quark_to_string(gwy_file_form_key(parsed));
}

/**
 * gwy_file_parse_key:
 * @strkey: String data file container key.
 * @parsed: (out):
 *          Location to store the parsed information. The contents of @parsed is undefined if the function return
 *          %FALSE.
 *
 * Parses a data file container key.
 *
 * The parsing usually identifies the data kind, id and the specific setting or property for the data (if applicable).
 * If a specific combination is impossible, for instance mask colour with graphs, the function returns %FALSE even
 * though the key may seem syntactically valid.
 *
 * Upon successful parsing all fields are always set (possibly to values meaning ‘none’). There are some special
 * cases. The piece %GWY_FILE_PIECE_FILENAME is file-global and has neither data kind nor id.
 *
 * When @parsed.suffix is not set to %NULL will generally point inside @strkey. Since #GwyDict keys are backed by
 * #GQuark, the corresponding string key is a constant string which always exists. Hence, the pointer will be always
 * valid. However, if you use gwy_file_parse_key() with an arbitrary string, be aware that modifying or freeing it
 * invalidates the suffix.
 *
 * See also gwy_file_form_key() for constructing the key back.
 *
 * Returns: %TRUE if the key was sucessfully parsed; %FALSE if it was not.
 **/
gboolean
gwy_file_parse_key(const gchar *strkey,
                   GwyFileKeyParsed *parsed)
{
    g_return_val_if_fail(parsed, FALSE);
    *parsed = (GwyFileKeyParsed){ -1, GWY_FILE_NONE, GWY_FILE_PIECE_NONE, NULL };
    g_return_val_if_fail(strkey, FALSE);

    if (strkey[0] != GWY_DICT_PATHSEP) {
        gwy_debug("malformed key %s", strkey);
        return FALSE;
    }

    /* Identify the data kind prefix if it is data (a few things might not be). */
    const DataKindSpec *spec = NULL;
    guint pos = 0;
    for (guint data_kind = 0; data_kind < GWY_FILE_N_KINDS; data_kind++) {
        spec = data_kind_spec + data_kind;
        pos = strlen(spec->prefix);
        if (strncmp(strkey, spec->prefix, pos) == 0) {
            parsed->data_kind = data_kind;
            break;
        }
        spec = NULL;
    }

    if (!spec) {
        /* Handle special non-data stuff. */
        if (gwy_strequal(strkey, "/filename")) {
            parsed->piece = GWY_FILE_PIECE_FILENAME;
            return TRUE;
        }
        gwy_debug("unknown no-data key %s", strkey);
        return FALSE;
    }

    /* If it is a data item then it must be followed with /123456. */
    if (strkey[pos++] != GWY_DICT_PATHSEP) {
        gwy_debug("malformed key %s", strkey);
        return FALSE;
    }

    /* Parse the number. Do not use strtol directly. It allows weird stuff like spaces. */
    guint len = 0;
    while (g_ascii_isdigit(strkey[pos + len]))
        len++;

    if (!len || len > 9) {
        gwy_debug("bad numerical id in %s", strkey);
        return FALSE;
    }

    parsed->id = atoi(strkey + pos);
    pos += len;
    if (parsed->id < 0) {
        gwy_debug("negative numerical id in %s", strkey);
        return FALSE;
    }

    /* If nothing follows the number, it is the main data item key. */
    if (!strkey[pos])
        return TRUE;
    if (strkey[pos++] != GWY_DICT_PATHSEP) {
        gwy_debug("malformed key %s", strkey);
        return FALSE;
    }

    for (guint i = 0; i < G_N_ELEMENTS(piece_suffixes); i++) {
        len = strlen(piece_suffixes[i].name);
        /* Compare piece name. The string must either end there or have a separator (do not confuse mask with
         * mask-color). */
        if (strncmp(strkey + pos, piece_suffixes[i].name, len) == 0) {
            if (!strkey[pos + len] || strkey[pos + len] == GWY_DICT_PATHSEP) {
                parsed->piece = piece_suffixes[i].value;
                break;
            }
        }
    }
    pos += len;
    if (parsed->piece == GWY_FILE_PIECE_NONE) {
        gwy_debug("no corresponding piece for %s", strkey);
        return FALSE;
    }

    /* Find the suffix. */
    if (strkey[pos]) {
        if (strkey[pos++] != GWY_DICT_PATHSEP) {
            gwy_debug("malformed key %s", strkey);
            return FALSE;
        }
        parsed->suffix = strkey + pos;
    }

    /* Check if the data_kind/piece/suffix combination is valid. */
    const guint *piece_ids = spec->pieces;
    guint npieces = spec->npieces;
    for (guint i = 0; i < npieces; i++) {
        const FilePieceSpec *pspec = all_pieces + piece_ids[i];
        if (pspec->piece != parsed->piece)
            continue;

        /* Special-case selections. They must have a name but the name can be anything. */
        if (parsed->piece == GWY_FILE_PIECE_SELECTION)
            return parsed->suffix && strlen(parsed->suffix) && !strchr(parsed->suffix, '/');

        if (!pspec->suffix && !parsed->suffix)
            return TRUE;
        /* Currently all pieces are either identifiers of something or pure prefixes, not both. So if one is
         * %NULL the key is not valid. */
        if (!pspec->suffix || !parsed->suffix) {
            gwy_debug("suffix presence mismatch %s", strkey);
            return FALSE;
        }
        if (gwy_strequal(pspec->suffix, parsed->suffix))
            return TRUE;
    }
    /* Either we did not find the piece at all for this data kind, or it has an unknown suffix. */
    gwy_debug("fell completely through %s", strkey);
    return FALSE;
}

/**
 * gwy_file_set_link_quark:
 * @file: A data file container.
 * @key: Link source in @file, as a quark.
 * @value: Link target in @file, as a quark.
 *
 * Creates a link in a data file container, specified using quarks.
 *
 * The link value is always a string. Use this function instead of gwy_file_set_link() for convenience when
 * you already have quarks. See its description for remarks.
 **/
void
gwy_file_set_link_quark(GwyFile *file,
                        GQuark key,
                        GQuark value)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(key);
    g_return_if_fail(value);
    set_link(file, key, value, NULL);
}

/**
 * gwy_file_set_link:
 * @file: A data file container.
 * @key: Link source in @file, as a string.
 * @value: Link target in @file, as a string.
 *
 * Creates a link in a data file container, specified using string keys.
 *
 * Link is a string which holds the key of some other data in the file. Its value can be obtained using
 * gwy_dict_get_string() or gwy_dict_gis_string(). The difference between links and plain strings is that
 * when data are moved around or merged, they can be automatically updated to point to the new location.
 *
 * Modifying the value by a #GwyDict function, even to another string, makes it a non-link. Therefore, do not do
 * it. It makes little sense to have links and non-links at the same @key.
 **/
void
gwy_file_set_link(GwyFile *file,
                  const gchar *key,
                  const gchar *value)
{
    g_return_if_fail(GWY_IS_FILE(file));
    g_return_if_fail(key);
    g_return_if_fail(value);
    set_link(file, g_quark_from_string(key), 0, value);
}

/**
 * gwy_file_value_is_link:
 * @file: A data file container.
 * @key: Key sin @file, as a string.
 *
 * Checks if a value in a data file container is a link.
 *
 * It is not an error to call this function with keys which do not corresponds to strings at all. It will return
 * %FALSE.
 *
 * Returns: %TRUE if @key is a link, %FALSE if it is not.
 **/
gboolean
gwy_file_value_is_link(GwyFile *file,
                       const gchar *key)
{
    g_return_val_if_fail(GWY_IS_FILE(file), FALSE);
    g_return_val_if_fail(key, FALSE);
    return find_link(file, g_quark_from_string(key)) >= 0;
}

/**
 * gwy_file_get_format_name:
 * @file: A data file container.
 *
 * Gets the file format name for a data file container.
 *
 * The file format name is the name of file import function used to import the data.
 *
 * Returns: (nullable): The file format name, possibly %NULL.
 **/
const gchar*
gwy_file_get_format_name(GwyFile *file)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    return file->priv->format_name;
}

/**
 * gwy_file_get_filename_sys:
 * @file: A data file
 *
 * Gets the file name corresponding to a data file container.
 *
 * The file name is set on two ocasions: file load and successful file save. File export does not set it.
 *
 * Returns: File name of @data (in GLib encoding), or %NULL. The returned string is owned by @file and is valid only
 *          until it is destroyed or saved again.
 **/
const gchar*
gwy_file_get_filename_sys(GwyFile *file)
{
    g_return_val_if_fail(GWY_IS_FILE(file), NULL);
    return file->priv->filename_sys;
}

void
_gwy_file_set_info(GwyFile *file, const gchar *name, const gchar *filename_sys)
{
    g_return_if_fail(GWY_IS_FILE(file));
    GwyFilePrivate *priv = file->priv;
    /* The function names are normally static, but it is currently not guaranteed. */
    priv->format_name = g_intern_string(name);
    gwy_assign_string(&priv->filename_sys, filename_sys);
}

/**
 * gwy_file_get_id:
 * @file: (nullable) (transfer none): A data file container.
 *
 * Gets the numerical identifier of a file.
 *
 * Only files managed by the data browser have an id. %GWY_FILE_ID_NONE is returned for free-standing files. You can
 * check whether a file is managed by calling this function and comparing the returned id to %GWY_FILE_ID_NONE.
 *
 * The identifier is not reused and does not change during program's lifetime. See also gwy_data_browser_get_file().
 *
 * For convenience, %NULL can be also passed. The function then returns a negative value (%GWY_FILE_ID_NONE).
 *
 * Returns: The id (a nonnegative integer), possibly %GWY_FILE_ID_NONE.
 **/
gint
gwy_file_get_id(GwyFile *file)
{
    if (!file)
        return GWY_FILE_ID_NONE;
    g_return_val_if_fail(GWY_IS_FILE(file), GWY_FILE_ID_NONE);
    /* Return GWY_FILE_ID_NONE for unmanaged files, but preserve the id even when some repeatedly adds the file to
     * the data browser and removes it again. */
    GwyFilePrivate *priv = file->priv;
    return priv->is_managed ? priv->id : GWY_FILE_ID_NONE;
}

void
_gwy_file_set_id(GwyFile *file, gint id)
{
    g_return_if_fail(GWY_IS_FILE(file));
    GwyFilePrivate *priv = file->priv;
    g_assert(priv->id == GWY_FILE_ID_NONE);
    priv->id = id;
}

void
_gwy_file_set_managed(GwyFile *file, gboolean is_managed)
{
    g_return_if_fail(GWY_IS_FILE(file));
    GwyFilePrivate *priv = file->priv;
    priv->is_managed = is_managed;
}

static const gchar*
get_enum_nick(guint value, GType gtype)
{
    GEnumClass *klass = g_type_class_ref(gtype);
    const GEnumValue *enumval = g_enum_get_value(klass, value);
    const gchar *nick = "???";
    if (enumval)
        nick = enumval->value_nick;
    else {
        g_assert(enumval);
    }
    g_type_class_unref(klass);
    return nick;
}

const gchar*
_gwy_data_kind_name(GwyDataKind data_kind)
{
    return get_enum_nick(data_kind, GWY_TYPE_DATA_KIND);
}

const gchar*
_gwy_file_piece_name(GwyFilePiece piece)
{
    return get_enum_nick(piece, GWY_TYPE_FILE_PIECE);
}

static DataKindInfo*
ensure_data_kind_info(GwyFile *file, GwyDataKind data_kind)
{
    g_return_val_if_fail(data_kind < (guint)GWY_FILE_N_KINDS, NULL);

    GwyFilePrivate *priv = file->priv;
    DataKindInfo *info = priv->data_kind_info + data_kind;
    if (!info->ids) {
        info->ids = g_array_new(FALSE, FALSE, sizeof(gint));
    }
    return info;
}

static void
fill_piece_gtypes(FilePieceSpec *pieces, guint nspec)
{
    for (guint i = 0; i < nspec; i++) {
        if (pieces->piece == GWY_FILE_PIECE_TITLE || pieces->piece == GWY_FILE_PIECE_PALETTE)
            pieces->type = G_TYPE_STRING;
        else if (pieces->piece == GWY_FILE_PIECE_VISIBLE || pieces->piece == GWY_FILE_PIECE_REAL_SQUARE)
            pieces->type = G_TYPE_BOOLEAN;
        else if (pieces->piece == GWY_FILE_PIECE_PICTURE)
            pieces->type = GWY_TYPE_FIELD;
        else if (pieces->piece == GWY_FILE_PIECE_MASK)
            pieces->type = GWY_TYPE_NIELD;
        else if (pieces->piece == GWY_FILE_PIECE_LOG)
            pieces->type = GWY_TYPE_STRING_LIST;
        else if (pieces->piece == GWY_FILE_PIECE_META)
            pieces->type = GWY_TYPE_DICT;
        else if (pieces->piece == GWY_FILE_PIECE_COLOR_MAPPING)
            pieces->type = G_TYPE_INT;
        else if (pieces->piece == GWY_FILE_PIECE_MASK_COLOR)
            pieces->type = GWY_TYPE_RGBA;
        else if (pieces->piece == GWY_FILE_PIECE_RANGE)
            pieces->type = G_TYPE_DOUBLE;
        else if (pieces->piece == GWY_FILE_PIECE_VIEW) {
            if (gwy_strequal(pieces->suffix, "height") || gwy_strequal(pieces->suffix, "width"))
                pieces->type = G_TYPE_INT;
            else
                pieces->type = G_TYPE_DOUBLE;
        }
        else {
            g_assert_not_reached();
        }
    }
}

/* Preferrably, this is not happening when file import modules construct the imported file as they use
 * gwy_dict_start_construction(). But if they do not, we keep up. It is just slower. */
static void
item_changed(GwyDict *container, GQuark key)
{
    GType type_after_the_change = gwy_dict_value_type(container, key);
    GwyFile *file = GWY_FILE(container);
    GwyFilePrivate *priv = file->priv;

    if (priv->setting_link != key) {
        gint linkpos = find_link(file, key);
        if (linkpos >= 0)
            g_array_remove_index_fast(priv->links, linkpos);
    }

    const gchar *strkey = g_quark_to_string(key);
    GwyFileKeyParsed parsed;
    if (!gwy_file_parse_key(strkey, &parsed)) {
        g_warning("Cannot parse file key %s.", strkey);
        return;
    }

    if (parsed.data_kind == GWY_FILE_NONE)
        return;
    g_assert(parsed.id >= 0);

    /* We only care about the main data items here. */
    if (parsed.piece != GWY_FILE_PIECE_NONE)
        return;

    DataKindInfo *info = ensure_data_kind_info(file, parsed.data_kind);
    GArray *ids = info->ids;
    guint i, n = ids->len;
    for (i = 0; i < n; i++) {
        if (g_array_index(ids, gint, i) >= parsed.id)
            break;
    }
    if (i == n) {
        /* If we did not find the id in our list it had to be newly added, not removed (or changed). */
        g_assert(type_after_the_change);
        g_array_append_val(ids, parsed.id);
    }
    else if (g_array_index(ids, gint, i) != parsed.id) {
        /* If we did not find the id in our list it had to be newly added, not removed (or changed). */
        g_assert(type_after_the_change);
        g_array_insert_val(ids, i, parsed.id);
    }
    else if (!type_after_the_change) {
        /* The item was found but no longer exists. So it was deleted. */
        g_array_remove_index(ids, i);
    }
    /* Otherwise, the item was found and exists. It has only changed and we do not need to update the lists. */
}

static void
enumerate_data(GQuark quark, GValue *gvalue, gpointer user_data)
{
    const gchar *strkey = g_quark_to_string(quark);
    GwyFileKeyParsed parsed;
    if (!gwy_file_parse_key(strkey, &parsed)) {
        g_warning("Cannot parse file key %s.", strkey);
        return;
    }

    if (parsed.data_kind == GWY_FILE_NONE)
        return;
    g_assert(parsed.id >= 0);

    /* We only care about the main data items here. */
    if (parsed.piece != GWY_FILE_PIECE_NONE)
        return;

    GwyFile *file = (GwyFile*)user_data;
    DataKindInfo *info = ensure_data_kind_info(file, parsed.data_kind);
    g_array_append_val(info->ids, parsed.id);
}

static gint
compare_int(gconstpointer pa, gconstpointer pb)
{
    gint a = *(const gint*)pa;
    gint b = *(const gint*)pb;

    if (a < b)
        return -1;
    if (a > b)
        return 1;
    return 0;
}

static void
scan_file_contents(GwyFile *file, G_GNUC_UNUSED GwyErrorList **error_list)
{
    gwy_dict_foreach(GWY_DICT(file), NULL, enumerate_data, file);
    GwyFilePrivate *priv = file->priv;
    for (guint data_kind = 0; data_kind < GWY_FILE_N_KINDS; data_kind++) {
        DataKindInfo *info = priv->data_kind_info + data_kind;
        if (info->ids)
            g_array_sort(info->ids, compare_int);
    }

    /* TODO: Run some sanity check? Once FilePieceSpec are complete, they define the file structure and can replace
     * 60% of what validate currently does. Most of the rest is either quite low-level and should be rejected already
     * in deserialisation (non-UTF8 strings, non-ASCII item names). */
}

static void
construction_finished(GwyDict *container)
{
    scan_file_contents(GWY_FILE(container), NULL);
}

static void
serializable_itemize(GwySerializable *serializable, GwySerializableGroup *group)
{
    serializable_parent_iface->itemize(serializable, group);
    g_assert(!gwy_dict_contains_by_name(GWY_DICT(serializable), "__links__"));
    GwyFilePrivate *priv = GWY_FILE(serializable)->priv;
    if (priv->links && priv->links->len) {
        g_assert(!priv->link_strings);
        GQuark *quarks = &g_array_index(priv->links, GQuark, 0);
        gsize n = priv->links->len;
        priv->link_strings = g_new(const gchar*, n);
        for (gsize i = 0; i < n; i++)
            priv->link_strings[i] = g_quark_to_string(quarks[i]);
        gwy_serializable_group_append_string_array(group, serializable_items + ITEM_LINKS,
                                                   (gchar**)priv->link_strings, n);
    }
}

static void
serializable_done(GwySerializable *serializable)
{
    GwyFilePrivate *priv = GWY_FILE(serializable)->priv;
    GWY_FREE(priv->link_strings);
    if (serializable_parent_iface->done)
        serializable_parent_iface->done(serializable);
}

static gboolean
serializable_construct(GwySerializable *serializable, GwySerializableGroup *group, GwyErrorList **error_list)
{
    /* Handle our extra items first. The parent class would try to gobble up everything and complain. */
    GwySerializableItem its[NUM_ITEMS];
    gwy_assign(its, serializable_items, NUM_ITEMS);
    gwy_deserialize_filter_items(its, NUM_ITEMS, group, TYPE_NAME, error_list);

    gboolean ok = FALSE;
    if (!serializable_parent_iface->construct(serializable, group, error_list))
        goto finish;

    GwyFile *file = GWY_FILE(serializable);
    if (its[ITEM_LINKS].array_size) {
        GwyFilePrivate *priv = GWY_FILE(serializable)->priv;
        gsize n = its[ITEM_LINKS].array_size;
        priv->links = g_array_sized_new(FALSE, FALSE, sizeof(GQuark), n);
        for (gsize i = 0; i < n; i++) {
            GQuark quark = g_quark_from_string(its[ITEM_LINKS].value.v_string_array[i]);
            g_array_append_val(priv->links, quark);
        }
    }

    scan_file_contents(file, error_list);
    ok = TRUE;

finish:
    if (its[ITEM_LINKS].array_size) {
        gsize n = its[ITEM_LINKS].array_size;
        for (gsize i = 0; i < n; i++)
            g_free(its[ITEM_LINKS].value.v_string_array[i]);
        g_free(its[ITEM_LINKS].value.v_string_array);
    }

    return ok;
}

static GwySerializable*
serializable_copy(GwySerializable *serializable)
{
    /* GwyDict is made for this kind of subclassing and its copy() creates an object of the correct type. */
    GwySerializable *copy = serializable_parent_iface->copy(serializable);
    copy_aux_info(GWY_FILE(copy), GWY_FILE(serializable));
    return copy;
}

static void
serializable_assign(GwySerializable *destination, GwySerializable *source)
{
    serializable_parent_iface->assign(destination, source);
    copy_aux_info(GWY_FILE(destination), GWY_FILE(source));
}

static void
copy_aux_info(GwyFile *dest, GwyFile *src)
{
    GwyFilePrivate *dpriv = dest->priv, *spriv = src->priv;
    for (guint i = 0; i < GWY_FILE_N_KINDS; i++)
        dpriv->data_kind_info[i].ids = assign_garray(dpriv->data_kind_info[i].ids, spriv->data_kind_info[i].ids);
    dpriv->links = assign_garray(dpriv->links, spriv->links);
}

static GArray*
assign_garray(GArray *dest, GArray *src)
{
    if (dest && src && src->len) {
        g_assert(g_array_get_element_size(dest) == g_array_get_element_size(src));
        g_array_set_size(dest, 0);
        g_array_append_vals(dest, src->data, src->len);
    }
    else if (dest) {
        g_array_free(dest, TRUE);
        return NULL;
    }
    else if (src && src->len) {
        dest = g_array_sized_new(FALSE, FALSE, g_array_get_element_size(src), src->len);
        g_array_append_vals(dest, src->data, src->len);
    }

    return dest;
}

static void
set_link(GwyFile *file, GQuark key, GQuark linkquark, const gchar *linkstr)
{
    GwyDict *container = GWY_DICT(file);
    GwyFilePrivate *priv = file->priv;
    gint linkpos = -1;
    if (gwy_dict_value_type(container, key) == G_TYPE_STRING)
        linkpos = find_link(file, key);

    if (!linkstr)
        linkstr = g_quark_to_string(linkquark);

    /* First mark the key as holding a link, then actually set the string. This way anything that responds to
     * "item-changed" signal has the correct information whether the changed item is a link. */
    if (linkpos == -1) {
        if (!priv->links)
            priv->links = g_array_new(FALSE, FALSE, sizeof(GQuark));
        g_array_append_val(priv->links, key);
    }

    g_assert(!priv->setting_link);
    priv->setting_link = key;
    gwy_dict_set_const_string(container, key, linkstr);
    priv->setting_link = 0;
}

static gint
find_link(GwyFile *file, GQuark quark)
{
    GwyFilePrivate *priv = file->priv;
    if (!priv->links)
        return -1;

    guint n = priv->links->len;
    GQuark *quarks = &g_array_index(priv->links, GQuark, 0);
    for (guint i = 0; i < n; i++) {
        if (quarks[i] == quark)
            return i;
    }
    return -1;
}

/**
 * gwy_file_copy:
 * @file: A data file to duplicate.
 *
 * Create a new data file as a copy of an existing one.
 *
 * This function is a convenience gwy_serializable_copy() wrapper.
 *
 * Returns: (transfer full):
 *          A copy of the data file.
 **/
GwyFile*
gwy_file_copy(GwyFile *file)
{
    /* Try to return a valid object even on utter failure. Returning NULL probably would crash something soon. */
    if (!GWY_IS_FILE(file)) {
        g_assert(GWY_IS_FILE(file));
        return g_object_new(GWY_TYPE_FILE, NULL);
    }
    return GWY_FILE(gwy_serializable_copy(GWY_SERIALIZABLE(file)));
}

/**
 * SECTION: file
 * @title: GwyFile
 * @short_description: Dictionary representing a data file
 * @see_also: #GwyDict
 *
 * #GwyFile is a subclass of #GwyDict, assuming a specific organisation of items inside to represent scanning
 * probe microscopy data, more specifically a Gwyddion GWY file.
 *
 * The class implements #GwySerializable copy() and assign() methods. A convenience wrapper gwy_file_copy() is
 * provided only for the first one. Assigning files by value is less efficient than simply destroying the existing
 * #GwyFile and creating a new data file. It also generates lots of signals about changed items. So, generally, do not
 * do it.
 **/

/**
 * GwyFile:
 *
 * The #GwyFile struct contains private data only and should be accessed using the functions below.
 **/

/**
 * GWY_FILE_PREFIX_IMAGE:
 *
 * Prefix of image string keys in Gwyddion data files.
 **/

/**
 * GWY_FILE_PREFIX_GRAPH:
 *
 * Prefix of graph string keys in Gwyddion data files.
 **/

/**
 * GWY_FILE_PREFIX_SPECTRA:
 *
 * Prefix of single point spectra string keys in Gwyddion data files.
 **/

/**
 * GWY_FILE_PREFIX_VOLUME:
 *
 * Prefix of volume data string keys in Gwyddion data files.
 **/

/**
 * GWY_FILE_PREFIX_XYZ:
 *
 * Prefix of XYZ data string keys in Gwyddion data files.
 **/

/**
 * GWY_FILE_PREFIX_CMAP:
 *
 * Prefix of curve map string keys in Gwyddion data files.
 **/

/**
 * GwyFileKeyParsed:
 * @id: Numerical id of a data in file, -1 for no data item.
 * @data_kind: Type of the data object, %GWY_FILE_NONE for no data item.
 * @piece: Particular piece related or belonging to the data, %GWY_FILE_PIECE_NONE for the main data object.
 * @suffix: Remaning part of the key (pointer to @strkey), or %NULL.
 *
 * Broken down representation of a piece of information in a data file container.
 **/

/**
 * GwyDataKind:
 * @GWY_FILE_NONE: No data. Often meaning an error, sometimes also special data keys not related to any specific
 *                 data.
 * @GWY_FILE_IMAGE: An image, primarily represented as #GwyField.
 * @GWY_FILE_GRAPH: A graph, primarily represented as #GwyGraphModel.
 * @GWY_FILE_SPECTRA: Single point spectra, primarily represented as #GwySpectra.
 * @GWY_FILE_VOLUME: Volume data, primarily represented as #GwyBrick.
 * @GWY_FILE_XYZ: XYZ data, primarily represented as #GwySurface.
 * @GWY_FILE_CMAP: A curve map, primarily represented as #GwyLawn.
 *
 * Basic kind of data which can be found in Gwyddion data files.
 **/

/**
 * GwyFilePiece:
 * @GWY_FILE_PIECE_PALETTE: False colour mapping gradient.
 * @GWY_FILE_PIECE_MASK: Image mask.
 * @GWY_FILE_PIECE_MASK_COLOR: Mask colour.
 * @GWY_FILE_PIECE_RANGE: False colour mapping range.
 * @GWY_FILE_PIECE_COLOR_MAPPING: False colour mapping type.
 * @GWY_FILE_PIECE_REAL_SQUARE: Pixel/real aspect ratio.
 * @GWY_FILE_PIECE_SELECTION: Image selections.
 * @GWY_FILE_PIECE_META: Data item metadata.
 * @GWY_FILE_PIECE_PICTURE: Object representing the visualisation or preview of the data.
 * @GWY_FILE_PIECE_VIEW: Window size or scaling.
 * @GWY_FILE_PIECE_TITLE: Data item title.
 * @GWY_FILE_PIECE_LOG: Log of operations on the data item.
 * @GWY_FILE_PIECE_VISIBLE: Data item visibility.
 * @GWY_FILE_PIECE_FILENAME: Name of the file.
 * @GWY_FILE_PIECE_NONE: No data piece. It can indicate an invalid data piece, but also the main data object.
 *
 * Type of auxiliary data in a Gwyddion data file.
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
