/*
 *  $Id: module-utils-file--inline.c 29345 2026-01-25 09:59:36Z yeti-dn $
 *  Copyright (C) 2026 David Necas (Yeti)
 *  E-mail: yeti@gwyddion.net
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <stdlib.h>

#include "libgwyapp/module-utils-file.h"

/* In order to provide also linkable versions of the functions, kill the redirection macros and define the symbols.
 *
 * This is a separate source file to ensure everything else is compiled with the macros. */
#undef gwy_get_gboolean8
#undef gwy_get_gint16_le
#undef gwy_get_gint16_be
#undef gwy_get_guint16_le
#undef gwy_get_guint16_be
#undef gwy_get_gint32_le
#undef gwy_get_gint32_be
#undef gwy_get_guint32_le
#undef gwy_get_guint32_be
#undef gwy_get_gint64_le
#undef gwy_get_gint64_be
#undef gwy_get_guint64_le
#undef gwy_get_guint64_be
#undef gwy_get_gfloat_le
#undef gwy_get_gfloat_be
#undef gwy_get_gdouble_le
#undef gwy_get_gdouble_be
#undef gwy_get_pascal_real_le
#undef gwy_get_pascal_real_be

/**
 * gwy_get_gboolean8:
 * @ppv: Pointer to a pointer to boolean (stored as a signle byte) in a memory buffer.
 *
 * Reads a boolean value stored as a signle byte from a binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The #gboolean value read from the buffer.
 **/
gboolean
gwy_get_gboolean8(const guchar **ppv)
{
    return _gwy_get_gboolean8_implementation(ppv);
}

/**
 * gwy_get_gint16_le:
 * @ppv: Pointer to a pointer to a little-endian signed 16bit integer in a memory buffer.
 *
 * Reads a signed 16bit integer value from a little-endian binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The #gint16 value read from the buffer.
 **/
gint16
gwy_get_gint16_le(const guchar **ppv)
{
    return _gwy_get_gint16_le_implementation(ppv);
}

/**
 * gwy_get_gint16_be:
 * @ppv: Pointer to a pointer to a big-endian signed 16bit integer in a memory buffer.
 *
 * Reads a signed 16bit integer value from a big-endian binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The #gint16 value read from the buffer.
 **/
gint16
gwy_get_gint16_be(const guchar **ppv)
{
    return _gwy_get_gint16_be_implementation(ppv);
}

/**
 * gwy_get_guint16_le:
 * @ppv: Pointer to a pointer to a little-endian unsigned 16bit integer in a memory buffer.
 *
 * Reads an unsigned 16bit integer value from a little-endian binary data buffer, moving the buffer pointer to point
 * just after the value.
 *
 * Returns: The #guint16 value read from the buffer.
 **/
guint16
gwy_get_guint16_le(const guchar **ppv)
{
    return _gwy_get_guint16_le_implementation(ppv);
}

/**
 * gwy_get_guint16_be:
 * @ppv: Pointer to a pointer to a big-endian unsigned 16bit integer in a memory buffer.
 *
 * Reads an unsigned 16bit integer value from a big-endian binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The #guint16 value read from the buffer.
 **/
guint16
gwy_get_guint16_be(const guchar **ppv)
{
    return _gwy_get_guint16_be_implementation(ppv);
}

/**
 * gwy_get_gint32_le:
 * @ppv: Pointer to a pointer to a little-endian signed 32bit integer in a memory buffer.
 *
 * Reads a signed 32bit integer value from a little-endian binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The #gint32 value read from the buffer.
 **/
gint32
gwy_get_gint32_le(const guchar **ppv)
{
    return _gwy_get_gint32_le_implementation(ppv);
}

/**
 * gwy_get_gint32_be:
 * @ppv: Pointer to a pointer to a big-endian signed 32bit integer in a memory buffer.
 *
 * Reads a signed 32bit integer value from a big-endian binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The #gint32 value read from the buffer.
 **/
gint32
gwy_get_gint32_be(const guchar **ppv)
{
    return _gwy_get_gint32_be_implementation(ppv);
}

/**
 * gwy_get_guint32_le:
 * @ppv: Pointer to a pointer to a little-endian unsigned 32bit integer in a memory buffer.
 *
 * Reads an unsigned 32bit integer value from a little-endian binary data buffer, moving the buffer pointer to point
 * just after the value.
 *
 * Returns: The #guint32 value read from the buffer.
 **/
guint32
gwy_get_guint32_le(const guchar **ppv)
{
    return _gwy_get_guint32_le_implementation(ppv);
}

/**
 * gwy_get_guint32_be:
 * @ppv: Pointer to a pointer to a big-endian unsigned 32bit integer in a memory buffer.
 *
 * Reads an unsigned 32bit integer value from a big-endian binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The #guint32 value read from the buffer.
 **/
guint32
gwy_get_guint32_be(const guchar **ppv)
{
    return _gwy_get_guint32_be_implementation(ppv);
}

/**
 * gwy_get_gint64_le:
 * @ppv: Pointer to a pointer to a little-endian signed 64bit integer in a memory buffer.
 *
 * Reads a signed 64bit integer value from a little-endian binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The #gint64 value read from the buffer.
 **/
gint64
gwy_get_gint64_le(const guchar **ppv)
{
    return _gwy_get_gint64_le_implementation(ppv);
}

/**
 * gwy_get_gint64_be:
 * @ppv: Pointer to a pointer to a big-endian signed 64bit integer in a memory buffer.
 *
 * Reads a signed 64bit integer value from a big-endian binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The #gint64 value read from the buffer.
 **/
gint64
gwy_get_gint64_be(const guchar **ppv)
{
    return _gwy_get_gint64_be_implementation(ppv);
}

/**
 * gwy_get_guint64_le:
 * @ppv: Pointer to a pointer to a little-endian unsigned 64bit integer in a memory buffer.
 *
 * Reads an unsigned 64bit integer value from a little-endian binary data buffer, moving the buffer pointer to point
 * just after the value.
 *
 * Returns: The #guint64 value read from the buffer.
 **/
guint64
gwy_get_guint64_le(const guchar **ppv)
{
    return _gwy_get_guint64_le_implementation(ppv);
}

/**
 * gwy_get_guint64_be:
 * @ppv: Pointer to a pointer to a big-endian unsigned 64bit integer in a memory buffer.
 *
 * Reads an unsigned 64bit integer value from a big-endian binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The #guint64 value read from the buffer.
 **/
guint64
gwy_get_guint64_be(const guchar **ppv)
{
    return _gwy_get_guint64_be_implementation(ppv);
}

/**
 * gwy_get_gfloat_le:
 * @ppv: Pointer to a pointer to a little-endian single-precision IEEE float in a memory buffer.
 *
 * Reads a single-precision IEEE float value from a little-endian binary data buffer, moving the buffer pointer to
 * point just after the value.
 *
 * Returns: The #gfloat value read from the buffer.
 **/
gfloat
gwy_get_gfloat_le(const guchar **ppv)
{
    return _gwy_get_gfloat_le_implementation(ppv);
}

/**
 * gwy_get_gfloat_be:
 * @ppv: Pointer to a pointer to a big-endian single-precision IEEE float in a memory buffer.
 *
 * Reads a single-precision IEEE float value from a big-endian binary data buffer, moving the buffer pointer to point
 * just after the value.
 *
 * Returns: The #gfloat value read from the buffer.
 **/
gfloat
gwy_get_gfloat_be(const guchar **ppv)
{
    return _gwy_get_gfloat_be_implementation(ppv);
}

/**
 * gwy_get_gdouble_le:
 * @ppv: Pointer to a pointer to a little-endian double-precision IEEE float in a memory buffer.
 *
 * Reads a double-precision IEEE float value from a little-endian binary data buffer, moving the buffer pointer to
 * point just after the value.
 *
 * Returns: The #gdouble value read from the buffer.
 **/
gdouble
gwy_get_gdouble_le(const guchar **ppv)
{
    return _gwy_get_gdouble_le_implementation(ppv);
}

/**
 * gwy_get_gdouble_be:
 * @ppv: Pointer to a pointer to a big-endian double-precision IEEE float in a memory buffer.
 *
 * Reads a double-precision IEEE float value from a big-endian binary data buffer, moving the buffer pointer to point
 * just after the value.
 *
 * Returns: The #gdouble value read from the buffer.
 **/
gdouble
gwy_get_gdouble_be(const guchar **ppv)
{
    return _gwy_get_gdouble_be_implementation(ppv);
}

/**
 * gwy_get_pascal_real_le:
 * @ppv: Pointer to a pointer to a little-endian six-byte Pascal Real in a memory buffer.
 *
 * Reads a six-byte Pascale Real value from a little-endian binary data buffer, moving the buffer pointer to point
 * just after the value.
 *
 * Returns: The floating point value read from the buffer as a #gdouble.
 **/
gdouble
gwy_get_pascal_real_le(const guchar **ppv)
{
    return _gwy_get_pascal_real_le_implementation(ppv);
}

/**
 * gwy_get_pascal_real_be:
 * @ppv: Pointer to a pointer to a big-endian six-byte Pascal Real in a memory buffer.
 *
 * Reads a six-byte Pascale Real value from a big-endian binary data buffer, moving the buffer pointer to point just
 * after the value.
 *
 * Returns: The floating point value read from the buffer as a #gdouble.
 **/
gdouble
gwy_get_pascal_real_be(const guchar **ppv)
{
    return _gwy_get_pascal_real_be_implementation(ppv);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
