#pragma once
/*
 *  $Id: dict.h 29478 2026-02-14 13:56:53Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef __GWY_CONTAINER_H__
#define __GWY_CONTAINER_H__

#include <glib-object.h>

G_BEGIN_DECLS

typedef void (*GwyDictForeachFunc)(GQuark key,
                                   GValue *value,
                                   gpointer user_data);

#define GWY_TYPE_DICT            (gwy_dict_get_type())
#define GWY_DICT(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj), GWY_TYPE_DICT, GwyDict))
#define GWY_DICT_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST((klass), GWY_TYPE_DICT, GwyDictClass))
#define GWY_IS_DICT(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj), GWY_TYPE_DICT))
#define GWY_IS_DICT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass), GWY_TYPE_DICT))
#define GWY_DICT_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj), GWY_TYPE_DICT, GwyDictClass))

#define GWY_DICT_PATHSEP      '/'
#define GWY_DICT_PATHSEP_STR  "/"

typedef struct _GwyDict             GwyDict;
typedef struct _GwyDictClass        GwyDictClass;
typedef struct _GwyDictPrivate      GwyDictPrivate;

struct _GwyDict {
    GObject parent_instance;
    GwyDictPrivate *priv;
};

struct _GwyDictClass {
    GObjectClass parent_class;

    void (*item_changed)(GwyDict *dict, GQuark key);
    void (*construction_finished)(GwyDict *dict);
    /*< private >*/
    void (*reserved1)(void);
    void (*reserved2)(void);
    void (*reserved3)(void);
    void (*reserved4)(void);
    void (*reserved5)(void);
};

GType         gwy_dict_get_type                (void)                        G_GNUC_CONST;
GwyDict*      gwy_dict_new                     (void);
GwyDict*      gwy_dict_new_in_construction     (void);
guint         gwy_dict_get_n_items             (GwyDict *dict);
GType         gwy_dict_value_type              (GwyDict *dict,
                                                GQuark key);
gboolean      gwy_dict_contains                (GwyDict *dict,
                                                GQuark key);
GValue        gwy_dict_get_value               (GwyDict *dict,
                                                GQuark key);
gboolean      gwy_dict_gis_value               (GwyDict *dict,
                                                GQuark key,
                                                GValue *value);
void          gwy_dict_set_value               (GwyDict *dict,
                                                GQuark key,
                                                GValue *value);
gboolean      gwy_dict_remove                  (GwyDict *dict,
                                                GQuark key);
guint         gwy_dict_remove_by_prefix        (GwyDict *dict,
                                                const gchar *prefix);
GwyDict*      gwy_dict_duplicate_by_prefix     (GwyDict *dict,
                                                ...);
GwyDict*      gwy_dict_duplicate_by_prefixv    (GwyDict *dict,
                                                guint n,
                                                const gchar **prefixes);
void          gwy_dict_copy_value              (GwyDict *source,
                                                GwyDict *dest,
                                                GQuark srckey,
                                                GQuark destkey);
gint          gwy_dict_transfer                (GwyDict *source,
                                                GwyDict *dest,
                                                const gchar *source_prefix,
                                                const gchar *dest_prefix,
                                                gboolean deep,
                                                gboolean force);
gboolean      gwy_dict_rename                  (GwyDict *dict,
                                                GQuark key,
                                                GQuark newkey,
                                                gboolean force);
guint         gwy_dict_foreach                 (GwyDict *dict,
                                                const gchar *prefix,
                                                GwyDictForeachFunc function,
                                                gpointer user_data);
GQuark*       gwy_dict_keys                    (GwyDict *dict);
const gchar** gwy_dict_keys_by_name            (GwyDict *dict);
GQuark*       gwy_dict_keys_with_prefix        (GwyDict *dict,
                                                const gchar *prefix,
                                                guint *n);
const gchar** gwy_dict_keys_with_prefix_by_name(GwyDict *dict,
                                                const gchar *prefix,
                                                guint *n);
void          gwy_dict_set_boolean             (GwyDict *dict,
                                                GQuark key,
                                                gboolean value);
gboolean      gwy_dict_get_boolean             (GwyDict *dict,
                                                GQuark key);
gboolean      gwy_dict_gis_boolean             (GwyDict *dict,
                                                GQuark key,
                                                gboolean *value);
void          gwy_dict_set_uchar               (GwyDict *dict,
                                                GQuark key,
                                                guchar value);
guchar        gwy_dict_get_uchar               (GwyDict *dict,
                                                GQuark key);
gboolean      gwy_dict_gis_uchar               (GwyDict *dict,
                                                GQuark key,
                                                guchar *value);
void          gwy_dict_set_int32               (GwyDict *dict,
                                                GQuark key,
                                                gint32 value);
gint32        gwy_dict_get_int32               (GwyDict *dict,
                                                GQuark key);
gboolean      gwy_dict_gis_int32               (GwyDict *dict,
                                                GQuark key,
                                                gint32 *value);
void          gwy_dict_set_enum                (GwyDict *dict,
                                                GQuark key,
                                                guint value);
guint         gwy_dict_get_enum                (GwyDict *dict,
                                                GQuark key);
gboolean      gwy_dict_gis_enum                (GwyDict *dict,
                                                GQuark key,
                                                guint *value);
void          gwy_dict_set_int64               (GwyDict *dict,
                                                GQuark key,
                                                gint64 value);
gint64        gwy_dict_get_int64               (GwyDict *dict,
                                                GQuark key);
gboolean      gwy_dict_gis_int64               (GwyDict *dict,
                                                GQuark key,
                                                gint64 *value);
void          gwy_dict_set_double              (GwyDict *dict,
                                                GQuark key,
                                                gdouble value);
gdouble       gwy_dict_get_double              (GwyDict *dict,
                                                GQuark key);
gboolean      gwy_dict_gis_double              (GwyDict *dict,
                                                GQuark key,
                                                gdouble *value);
void          gwy_dict_set_string              (GwyDict *dict,
                                                GQuark key,
                                                gchar *value);
void          gwy_dict_set_const_string        (GwyDict *dict,
                                                GQuark key,
                                                const gchar *value);
const gchar*  gwy_dict_get_string              (GwyDict *dict,
                                                GQuark key);
gboolean      gwy_dict_gis_string              (GwyDict *dict,
                                                GQuark key,
                                                const gchar **value);
void          gwy_dict_set_object              (GwyDict *dict,
                                                GQuark key,
                                                gpointer value);
void          gwy_dict_pass_object             (GwyDict *dict,
                                                GQuark key,
                                                gpointer value);
gpointer      gwy_dict_get_object              (GwyDict *dict,
                                                GQuark key);
gboolean      gwy_dict_gis_object              (GwyDict *dict,
                                                GQuark key,
                                                gpointer value);
void          gwy_dict_set_boxed               (GwyDict *dict,
                                                GType type,
                                                GQuark key,
                                                gpointer value);
void          gwy_dict_pass_boxed              (GwyDict *dict,
                                                GType type,
                                                GQuark key,
                                                gpointer value);
gpointer      gwy_dict_get_boxed               (GwyDict *dict,
                                                GType type,
                                                GQuark key);
gboolean      gwy_dict_gis_boxed               (GwyDict *dict,
                                                GType type,
                                                GQuark key,
                                                gpointer value);
void          gwy_dict_start_construction      (GwyDict *dict);
void          gwy_dict_finish_construction     (GwyDict *dict);
gboolean      gwy_dict_is_being_constructed    (GwyDict *dict);
GwyDict*      gwy_dict_copy                    (GwyDict *dict);
void          gwy_dict_assign                  (GwyDict *destination,
                                                GwyDict *source);
GPtrArray*    gwy_dict_serialize_to_text       (GwyDict *dict);
GwyDict*      gwy_dict_deserialize_from_text   (const gchar *text);

#define gwy_dict_value_type_by_name(dict,name)               gwy_dict_value_type(dict,g_quark_try_string(name))
#define gwy_dict_contains_by_name(dict,name)                 gwy_dict_contains(dict,g_quark_try_string(name))
#define gwy_dict_set_value_by_name(dict,name,value)          gwy_dict_set_value(dict,g_quark_from_string(name),value)
#define gwy_dict_get_value_by_name(dict,name)                gwy_dict_get_value(dict,g_quark_from_string(name))
#define gwy_dict_gis_value_by_name(dict,name,value)          gwy_dict_gis_value(dict,g_quark_try_string(name),value)
#define gwy_dict_copy_value_by_name(dict1,dict2,name1,name2) gwy_dict_copy_value(dict1,dict2,g_quark_try_string(name1),g_quark_try_string(name2))
#define gwy_dict_remove_by_name(dict,name)                   gwy_dict_remove(dict,g_quark_try_string(name))
#define gwy_dict_rename_by_name(dict,name,nn,f)              gwy_dict_rename(dict,g_quark_try_string(name),g_quark_from_string(nn),f)
#define gwy_dict_set_boolean_by_name(dict,name,value)        gwy_dict_set_boolean(dict,g_quark_from_string(name),value)
#define gwy_dict_get_boolean_by_name(dict,name)              gwy_dict_get_boolean(dict,g_quark_from_string(name))
#define gwy_dict_gis_boolean_by_name(dict,name,value)        gwy_dict_gis_boolean(dict,g_quark_try_string(name),value)
#define gwy_dict_set_uchar_by_name(dict,name,value)          gwy_dict_set_uchar(dict,g_quark_from_string(name),value)
#define gwy_dict_get_uchar_by_name(dict,name)                gwy_dict_get_uchar(dict,g_quark_from_string(name))
#define gwy_dict_gis_uchar_by_name(dict,name,value)          gwy_dict_gis_uchar(dict,g_quark_try_string(name),value)
#define gwy_dict_set_int32_by_name(dict,name,value)          gwy_dict_set_int32(dict,g_quark_from_string(name),value)
#define gwy_dict_get_int32_by_name(dict,name)                gwy_dict_get_int32(dict,g_quark_from_string(name))
#define gwy_dict_gis_int32_by_name(dict,name,value)          gwy_dict_gis_int32(dict,g_quark_try_string(name),value)
#define gwy_dict_set_enum_by_name(dict,name,value)           gwy_dict_set_enum(dict,g_quark_from_string(name),value)
#define gwy_dict_get_enum_by_name(dict,name)                 gwy_dict_get_enum(dict,g_quark_from_string(name))
#define gwy_dict_gis_enum_by_name(dict,name,value)           gwy_dict_gis_enum(dict,g_quark_try_string(name),value)
#define gwy_dict_set_int64_by_name(dict,name,value)          gwy_dict_set_int64(dict,g_quark_from_string(name),value)
#define gwy_dict_get_int64_by_name(dict,name)                gwy_dict_get_int64(dict,g_quark_from_string(name))
#define gwy_dict_gis_int64_by_name(dict,name,value)          gwy_dict_gis_int64(dict,g_quark_try_string(name),value)
#define gwy_dict_set_double_by_name(dict,name,value)         gwy_dict_set_double(dict,g_quark_from_string(name),value)
#define gwy_dict_get_double_by_name(dict,name)               gwy_dict_get_double(dict,g_quark_from_string(name))
#define gwy_dict_gis_double_by_name(dict,name,value)         gwy_dict_gis_double(dict,g_quark_try_string(name),value)
#define gwy_dict_set_string_by_name(dict,name,value)         gwy_dict_set_string(dict,g_quark_from_string(name),value)
#define gwy_dict_set_const_string_by_name(dict,name,value)   gwy_dict_set_const_string(dict,g_quark_from_string(name),value)
#define gwy_dict_get_string_by_name(dict,name)               gwy_dict_get_string(dict,g_quark_from_string(name))
#define gwy_dict_gis_string_by_name(dict,name,value)         gwy_dict_gis_string(dict,g_quark_try_string(name),value)
#define gwy_dict_set_object_by_name(dict,name,value)         gwy_dict_set_object(dict,g_quark_from_string(name),value)
#define gwy_dict_pass_object_by_name(dict,name,value)        gwy_dict_pass_object(dict,g_quark_from_string(name),value)
#define gwy_dict_get_object_by_name(dict,name)               gwy_dict_get_object(dict,g_quark_from_string(name))
#define gwy_dict_gis_object_by_name(dict,name,value)         gwy_dict_gis_object(dict,g_quark_try_string(name),value)
#define gwy_dict_set_boxed_by_name(dict,type,name,value)     gwy_dict_set_boxed(dict,type,g_quark_from_string(name),value)
#define gwy_dict_pass_boxed_by_name(dict,type,name,value)    gwy_dict_pass_boxed(dict,type,g_quark_from_string(name),value)
#define gwy_dict_get_boxed_by_name(dict,type,name)           gwy_dict_get_boxed(dict,type,g_quark_from_string(name))
#define gwy_dict_gis_boxed_by_name(dict,type,name,value)     gwy_dict_gis_boxed(dict,type,g_quark_try_string(name),value)

G_END_DECLS

#endif

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
