/*
 *  $Id: cmap_fit.c 25639 2023-09-07 12:23:42Z yeti-dn $
 *  Copyright (C) 2021-2026 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gwy.h>
#include "libgwyddion/omp.h"

#define RUN_MODES (GWY_RUN_INTERACTIVE)

enum {
    PREVIEW_SIZE      = 360,
    RESPONSE_ESTIMATE = GWY_RESPONSE_USER,
    RESPONSE_FIT,
};

enum {
    PARAM_RANGE_FROM,
    PARAM_RANGE_TO,
    PARAM_ABSCISSA,
    PARAM_ORDINATE,
    PARAM_ENABLE_ABSCISSA,
    PARAM_SEGMENT,
    PARAM_ENABLE_SEGMENT,
    PARAM_XPOS,
    PARAM_YPOS,
    PARAM_FUNCTION,
    PARAM_ESTIMATE,
    PARAM_REPORT_ERR,
    PARAM_REPORT_CHI,

    WIDGET_FIT_PARAMETERS,
    INFO_CHISQ,
    INFO_FORMULA,
};

typedef struct {
    GwyParams *params;
    GwyNLFitPreset *fitfunc;
    GwyLawn *lawn;
    GwyField *field;
    gint nsegments;
    GPtrArray *result_fields;
    GwyNield *mask;
    gdouble *fit_param;
    gboolean *fixed;
    gdouble xmin;
    gdouble xmax;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
    GwyParamTable *table_fit;
    GwyParamTable *table_optimize;
    GtkWidget *fit_params;
    GwySelection *selection;
    GwySelection *graph_selection;
    GwyGraphModel *gmodel;
    GtkWidget *fitok;
} ModuleGUI;

static gboolean         module_register             (void);
static GwyParamDef*     define_module_params        (void);
static void             module_main                 (GwyFile *data,
                                                     GwyRunModeFlags mode);
static gboolean         execute                     (ModuleArgs *args,
                                                     GtkWindow *window);
static GwyDialogOutcome run_gui                     (ModuleArgs *args,
                                                     GwyFile *data,
                                                     gint id);
static void             dialog_response             (ModuleGUI *gui,
                                                     gint response);
static void             param_changed               (ModuleGUI *gui,
                                                     gint id);
static void             param_fit_changed           (ModuleGUI *gui,
                                                     gint id);
static void             preview                     (gpointer user_data);
static void             update_sensitivity          (ModuleGUI *gui);
static void             set_selection               (ModuleGUI *gui);
static GtkWidget*       create_fit_table            (gpointer user_data);
static void             point_selection_changed     (ModuleGUI *gui,
                                                     gint id,
                                                     GwySelection *selection);
static void             graph_selected              (GwySelection* selection,
                                                     gint i,
                                                     ModuleGUI *gui);
static void             extract_one_curve           (GwyLawn *lawn,
                                                     GwyGraphCurveModel *gcmodel,
                                                     gint col,
                                                     gint row,
                                                     gint segment,
                                                     GwyParams *params);
static void             estimate_one_curve          (GwyGraphCurveModel *gcmodel,
                                                     GwyParams *params,
                                                     GwyNLFitPreset *fitfunc,
                                                     gdouble *fitparams);
static gdouble          fit_one_curve               (GwyGraphCurveModel *gcmodel,
                                                     GwyParams *params,
                                                     GwyNLFitPreset *fitfunc,
                                                     gdouble *fitparams,
                                                     gboolean *fix,
                                                     gdouble *error,
                                                     gboolean *fitok);
static gdouble          do_fit                      (const gdouble *xdata,
                                                     const gdouble *ydata,
                                                     gint ndata,
                                                     GwyNLFitPreset *fitfunc,
                                                     gdouble from,
                                                     gdouble to,
                                                     gdouble *retparam,
                                                     gboolean *fix,
                                                     gdouble *error,
                                                     gboolean *fitok);
static void             do_estimate                 (const gdouble *xdata,
                                                     const gdouble *ydata,
                                                     gint ndata,
                                                     GwyNLFitPreset *fitfunc,
                                                     gdouble from,
                                                     gdouble to,
                                                     gdouble *retparam);
static void             fit_param_table_resize      (ModuleGUI *gui);
static void             fit_param_table_update_units(ModuleGUI *gui);
static void             update_graph_model_props    (GwyGraphModel *gmodel,
                                                     ModuleArgs *args);
static void             add_mask_and_copy_gradient  (GwyFile *data,
                                                     GwyNield *mask,
                                                     gint id,
                                                     gint newid);
static void             param_value_activate        (ModuleGUI *gui,
                                                     guint i);
static void             sanitise_params             (ModuleArgs *args);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Fit curves in volume data."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.2",
    "David Nečas (Yeti) & Petr Klapetek",
    "2024",
};

GWY_MODULE_QUERY2(module_info, cmap_fit)

static gboolean
module_register(void)
{
    gwy_curve_map_func_register("cmap_fit",
                                &module_main,
                                N_("/Fit _Curves..."),
                                NULL,
                                RUN_MODES,
                                GWY_MENU_FLAG_CMAP,
                                N_("Fit curves in volume data"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_curve_map_func_current());

    gwy_param_def_add_resource(paramdef, PARAM_FUNCTION, "function", _("_Function"),
                               gwy_nlfit_presets(), "Gaussian");
    gwy_param_def_add_lawn_curve(paramdef, PARAM_ABSCISSA, "abscissa", _("Abscissa"));
    gwy_param_def_add_lawn_curve(paramdef, PARAM_ORDINATE, "ordinate", _("Ordinate"));
    gwy_param_def_add_boolean(paramdef, PARAM_ENABLE_ABSCISSA, "enable_abscissa", NULL, FALSE);
    gwy_param_def_add_int(paramdef, PARAM_XPOS, "xpos", NULL, -1, G_MAXINT, -1);
    gwy_param_def_add_int(paramdef, PARAM_YPOS, "ypos", NULL, -1, G_MAXINT, -1);
    gwy_param_def_add_double(paramdef, PARAM_RANGE_FROM, "from", _("_From"), 0.0, 1.0, 0.0);
    gwy_param_def_add_double(paramdef, PARAM_RANGE_TO, "to", _("_To"), 0.0, 1.0, 1.0);
    gwy_param_def_add_lawn_segment(paramdef, PARAM_SEGMENT, "segment", NULL);
    gwy_param_def_add_boolean(paramdef, PARAM_ENABLE_SEGMENT, "enable_segment", NULL, FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_ESTIMATE, "estimate", _("Run _estimate at each point"), FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_REPORT_ERR, "reporterr", _("Output fit errors"), FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_REPORT_CHI, "reportchi", _("Output residual sum"), FALSE);

    return paramdef;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    ModuleArgs args;
    GwyLawn *lawn = NULL;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    gint id, i;

    g_return_if_fail(mode & RUN_MODES);

    gwy_clear1(args);
    gwy_data_browser_get_current(GWY_APP_LAWN, &lawn,
                                 GWY_APP_LAWN_ID, &id,
                                 0);
    g_return_if_fail(GWY_IS_LAWN(lawn));

    args.lawn = lawn;
    args.nsegments = gwy_lawn_get_n_segments(lawn);
    args.params = gwy_params_new_from_settings(define_module_params());
    sanitise_params(&args);

    args.result_fields = g_ptr_array_new_full(0, g_object_unref);
    args.field = gwy_field_new(gwy_lawn_get_xres(lawn), gwy_lawn_get_yres(lawn),
                               gwy_lawn_get_xreal(lawn), gwy_lawn_get_yreal(lawn), TRUE);
    gwy_field_set_xoffset(args.field, gwy_lawn_get_xoffset(lawn));
    gwy_field_set_yoffset(args.field, gwy_lawn_get_yoffset(lawn));
    gwy_unit_assign(gwy_field_get_unit_xy(args.field), gwy_lawn_get_unit_xy(lawn));

    if (mode == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, id);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (outcome != GWY_DIALOG_HAVE_RESULT) {
        if (!execute(&args, gwy_data_browser_get_window_for_data(data, GWY_FILE_CMAP, id)))
            goto end;
    }

    guint nparams = gwy_nlfit_preset_get_nparams(args.fitfunc);
    gboolean report_err = gwy_params_get_boolean(args.params, PARAM_REPORT_ERR);
    gboolean report_chi = gwy_params_get_boolean(args.params, PARAM_REPORT_CHI);

    for (i = 0; i < nparams; i++) {
        GwyField *field = g_ptr_array_index(args.result_fields, i);
        gint newid = gwy_file_add_image(data, field);
        gwy_file_set_visible(data, GWY_FILE_IMAGE, newid, TRUE);
        const gchar *parname = gwy_nlfit_preset_get_param_name(args.fitfunc, i);
        gchar *textparname = NULL;

        pango_parse_markup(parname, -1, 0, NULL, &textparname, NULL, NULL);
        gwy_file_set_title(data, GWY_FILE_IMAGE, newid, textparname, TRUE);
        g_free(textparname);
        add_mask_and_copy_gradient(data, args.mask, id, newid);
    }
    if (report_err) {
        for (i = 0; i < nparams; i++) {
            GwyField *field = g_ptr_array_index(args.result_fields, nparams + i);
            gint newid = gwy_file_add_image(data, field);
            gwy_file_set_visible(data, GWY_FILE_IMAGE, newid, TRUE);
            const gchar *parname = gwy_nlfit_preset_get_param_name(args.fitfunc, i);
            gchar *textparname = NULL;

            pango_parse_markup(parname, -1, 0, NULL, &textparname, NULL, NULL);
            gwy_file_set_title(data, GWY_FILE_IMAGE, newid, g_strdup_printf("error %s", textparname), TRUE);
            g_free(textparname);
            add_mask_and_copy_gradient(data, args.mask, id, newid);
        }
    }
    if (report_chi) {
        GwyField *field = g_ptr_array_index(args.result_fields, nparams + nparams*report_err);
        gint newid = gwy_file_add_image(data, field);
        gwy_file_set_visible(data, GWY_FILE_IMAGE, newid, TRUE);
        gwy_file_set_title(data, GWY_FILE_IMAGE, newid, g_strdup("Residual sum"), TRUE);
        add_mask_and_copy_gradient(data, args.mask, id, newid);
    }

end:
    g_ptr_array_free(args.result_fields, TRUE);
    g_object_unref(args.field);
    g_clear_object(&args.mask);
    g_object_unref(args.params);
    g_free(args.fit_param);
    g_free(args.fixed);
}

static void
add_mask_and_copy_gradient(GwyFile *data, GwyNield *mask, gint id, gint newid)
{
    if (gwy_nield_max(mask) > 0) {
        gwy_nield_number_contiguous(mask);
        gwy_file_pass_image_mask(data, newid, gwy_nield_copy(mask));
    }

    gwy_file_sync_items(data, GWY_FILE_CMAP, id,
                        data, GWY_FILE_IMAGE, newid,
                        GWY_FILE_ITEM_PALETTE, FALSE);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyFile *data, gint id)
{
    GtkWidget *hbox, *graph, *dataview, *area;
    GwyParamTable *table;
    GwyDialog *dialog;
    ModuleGUI gui;
    GwyField *field;
    GwyDialogOutcome outcome;
    GwyGraphCurveModel *gcmodel;

    gwy_clear1(gui);
    gui.args = args;

    gui.dialog = gwy_dialog_new(_("Fit Curves"));
    dialog = GWY_DIALOG(gui.dialog);
    gtk_dialog_add_button(GTK_DIALOG(dialog), C_("verb", "_Estimate single"), RESPONSE_ESTIMATE);
    gtk_dialog_add_button(GTK_DIALOG(dialog), C_("verb", "_Fit single"), RESPONSE_FIT);

    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    field = gwy_dict_get_object(GWY_DICT(data), gwy_file_key_cmap_picture(id));
    dataview = gwy_create_preview(field, NULL, PREVIEW_SIZE);
    gwy_setup_data_view(GWY_DATA_VIEW(dataview), data, GWY_FILE_CMAP, id,
                        GWY_FILE_ITEM_PALETTE | GWY_FILE_ITEM_REAL_SQUARE);
    gui.selection = gwy_create_preview_vector_layer(GWY_DATA_VIEW(dataview), GWY_TYPE_LAYER_POINT, 1, TRUE);
    set_selection(&gui);
    hbox = gwy_create_dialog_preview_hbox(GTK_DIALOG(dialog), GWY_DATA_VIEW(dataview), FALSE);

    gui.gmodel = gwy_graph_model_new();
    gcmodel = gwy_graph_curve_model_new();
    g_object_set(gcmodel,
                 "mode", GWY_GRAPH_CURVE_LINE,
                 "color", gwy_graph_get_preset_color(0),
                 "description", g_strdup(_("data")),
                 NULL);
    gwy_graph_model_add_curve(gui.gmodel, gcmodel);
    g_object_unref(gcmodel);

    gcmodel = gwy_graph_curve_model_new();
    g_object_set(gcmodel,
                 "mode", GWY_GRAPH_CURVE_LINE,
                 "color", gwy_graph_get_preset_color(1),
                 "description", g_strdup(_("fit")),
                 NULL);
    gwy_graph_model_add_curve(gui.gmodel, gcmodel);
    g_object_unref(gcmodel);

    graph = gwy_graph_new(gui.gmodel);
    area = gwy_graph_get_area(GWY_GRAPH(graph));
    gwy_graph_enable_user_input(GWY_GRAPH(graph), FALSE);
    gwy_graph_area_set_status(GWY_GRAPH_AREA(area), GWY_GRAPH_STATUS_XSEL);
    gwy_graph_area_set_selection_editable(GWY_GRAPH_AREA(area), TRUE);
    gui.graph_selection = gwy_graph_area_get_selection(GWY_GRAPH_AREA(area), GWY_GRAPH_STATUS_XSEL);
    gtk_widget_set_size_request(graph, PREVIEW_SIZE, PREVIEW_SIZE);
    gtk_box_pack_start(GTK_BOX(hbox), graph, TRUE, TRUE, 0);

    hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 20);
    gwy_dialog_add_content(GWY_DIALOG(gui.dialog), hbox, TRUE, TRUE, 4);

    table = gui.table = gwy_param_table_new(args->params);
    gwy_param_table_append_lawn_curve(table, PARAM_ABSCISSA, args->lawn);
    gwy_param_table_add_enabler(table, PARAM_ENABLE_ABSCISSA, PARAM_ABSCISSA);
    gwy_param_table_append_lawn_curve(table, PARAM_ORDINATE, args->lawn);
    if (args->nsegments) {
        gwy_param_table_append_lawn_segment(table, PARAM_SEGMENT, args->lawn);
        gwy_param_table_add_enabler(table, PARAM_ENABLE_SEGMENT, PARAM_SEGMENT);
    }
    gwy_param_table_append_slider(table, PARAM_RANGE_FROM);
    gwy_param_table_slider_set_factor(table, PARAM_RANGE_FROM, 100.0);
    gwy_param_table_set_unitstr(table, PARAM_RANGE_FROM, "%");
    gwy_param_table_append_slider(table, PARAM_RANGE_TO);
    gwy_param_table_slider_set_factor(table, PARAM_RANGE_TO, 100.0);
    gwy_param_table_set_unitstr(table, PARAM_RANGE_TO, "%");

    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    table = gui.table_fit = gwy_param_table_new(args->params);
    gwy_param_table_append_combo(table, PARAM_FUNCTION);
    gwy_param_table_append_info(table, INFO_FORMULA, "Formula:");
    gwy_param_table_append_foreign(table, WIDGET_FIT_PARAMETERS, create_fit_table, &gui, NULL);
    gwy_param_table_append_info(table, INFO_CHISQ, _("Residual sum:"));
    //gwy_param_table_set_label(table, INFO_CHISQ, _("χ<sup>2</sup> result:"));

    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    table = gui.table_optimize = gwy_param_table_new(args->params);
    gwy_param_table_append_checkbox(table, PARAM_ESTIMATE);
    gwy_param_table_append_checkbox(table, PARAM_REPORT_ERR);
    gwy_param_table_append_checkbox(table, PARAM_REPORT_CHI);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    g_signal_connect_swapped(gui.table, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.table_fit, "param-changed", G_CALLBACK(param_fit_changed), &gui);
    g_signal_connect_swapped(gui.selection, "changed", G_CALLBACK(point_selection_changed), &gui);
    g_signal_connect_swapped(dialog, "response", G_CALLBACK(dialog_response), &gui);
    g_signal_connect(gui.graph_selection, "changed", G_CALLBACK(graph_selected), &gui);
    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_IMMEDIATE, preview, &gui, NULL);

    gwy_param_table_param_changed(gui.table_fit, PARAM_FUNCTION);
    outcome = gwy_dialog_run(dialog);

    g_object_unref(gui.gmodel);

    return outcome;
}

static void
plot_result(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;
    gdouble from = gwy_params_get_double(params, PARAM_RANGE_FROM);
    gdouble to = gwy_params_get_double(params, PARAM_RANGE_TO);
    guint nparams = gwy_nlfit_preset_get_nparams(args->fitfunc);
    gdouble fit_param[nparams];
    gwy_fit_table_gather(GWY_FIT_TABLE(gui->fit_params), fit_param, NULL, NULL);

    GwyGraphCurveModel *gc = gwy_graph_model_get_curve(gui->gmodel, 0);
    gdouble xfrom, xto;
    gwy_graph_curve_model_get_x_range(gc, &xfrom, &xto);
    gdouble sel[2];
    sel[0] = xfrom + from*(xto-xfrom);
    sel[1] = xfrom + to*(xto-xfrom);
    gwy_selection_set_data(gui->graph_selection, 1, sel);

    guint nfit = 100;
    gdouble *xfit = gwy_math_linspace(NULL, nfit, xfrom, (xto - xfrom)/nfit);
    gdouble *yfit = g_new(gdouble, nfit);
    gboolean fres;
    for (guint i = 0; i < nfit; i++)
        yfit[i] = gwy_nlfit_preset_get_value(args->fitfunc, xfit[i], fit_param, &fres);

    gc = gwy_graph_model_get_curve(gui->gmodel, 1);
    gwy_graph_curve_model_set_data(gc, xfit, yfit, nfit);

    g_free(xfit);
    g_free(yfit);
}

static void
dialog_response(ModuleGUI *gui, gint response)
{
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;
    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    gint col = gwy_params_get_int(params, PARAM_XPOS);
    gint row = gwy_params_get_int(params, PARAM_YPOS);
    gboolean segment_enabled = args->nsegments ? gwy_params_get_boolean(params, PARAM_ENABLE_SEGMENT) : FALSE;
    gint segment = segment_enabled ? gwy_params_get_int(params, PARAM_SEGMENT) : -1;
    guint nparams = gwy_nlfit_preset_get_nparams(args->fitfunc);
    gdouble fit_param[nparams];

    gwy_fit_table_gather(table, args->fit_param, NULL, args->fixed);
    if (response == RESPONSE_ESTIMATE) {
        GwyGraphCurveModel *gc = gwy_graph_model_get_curve(gui->gmodel, 0);
        extract_one_curve(args->lawn, gc, col, row, segment, params);
        estimate_one_curve(gc, params, args->fitfunc, fit_param);
        gwy_fit_table_clear_errors(table);
        for (guint i = 0; i < nparams; i++)
            gwy_fit_table_set_value(table, i, fit_param[i]);
        plot_result(gui);
        gwy_param_table_info_set_valuestr(gui->table_fit, INFO_CHISQ, _("data not fitted"));
    }
    else if (response == RESPONSE_FIT) {
        gdouble error[nparams];
        gboolean fixed[nparams];
        gwy_fit_table_gather(table, fit_param, NULL, fixed);

        GwyGraphCurveModel *gc = gwy_graph_model_get_curve(gui->gmodel, 0);
        extract_one_curve(args->lawn, gc, col, row, segment, params);
        gboolean fitok;
        gdouble chisq = fit_one_curve(gc, params, args->fitfunc, fit_param, fixed, error, &fitok);
        for (guint i = 0; i < nparams; i++) {
            gwy_fit_table_set_value(table, i, fit_param[i]);
            gwy_fit_table_set_error(table, i, error[i]);
        }

        if (fitok)
            gwy_param_table_info_set_valuestr(gui->table_fit, INFO_CHISQ, g_strdup_printf("%g", chisq));
        else
            gwy_param_table_info_set_valuestr(gui->table_fit, INFO_CHISQ, _("fit failed"));

        plot_result(gui);
    }
}

static void
graph_selected(GwySelection* selection, gint i, ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    gdouble range[2];
    gdouble xfrom, xto, pfrom, pto;
    gboolean have_range = TRUE;

    g_return_if_fail(i <= 0);

    if (gwy_selection_get_data(selection, NULL) <= 0)
        have_range = FALSE;
    else {
        gwy_selection_get_object(selection, 0, range);
        if (range[0] == range[1])
            have_range = FALSE;
    }
    if (have_range) {
        xfrom = MIN(range[0], range[1]);
        xto = MAX(range[0], range[1]);
    }
    else {
        xfrom = args->xmin;
        xto = args->xmax;
    }

    pfrom = CLAMP((xfrom - args->xmin)/(args->xmax - args->xmin), 0.0, 1.0);
    pto = CLAMP((xto - args->xmin)/(args->xmax - args->xmin), 0.0, 1.0);

    gwy_param_table_set_double(gui->table, PARAM_RANGE_FROM, pfrom);
    gwy_param_table_set_double(gui->table, PARAM_RANGE_TO, pto);
}

static void
param_changed(ModuleGUI *gui, G_GNUC_UNUSED gint id)
{
    gwy_param_table_info_set_valuestr(gui->table_fit, INFO_CHISQ, _("data not fitted"));
    gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
param_fit_changed(ModuleGUI *gui, gint id)
{
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;

    if (id < 0 || id == PARAM_FUNCTION) {
        args->fitfunc = gwy_inventory_get_item(gwy_nlfit_presets(), gwy_params_get_string(params, PARAM_FUNCTION));
        gwy_param_table_info_set_valuestr(gui->table_fit, INFO_FORMULA, gwy_nlfit_preset_get_formula(args->fitfunc));
        fit_param_table_resize(gui);
    }

    gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
fit_param_table_resize(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    guint nparams = gwy_nlfit_preset_get_nparams(args->fitfunc);
    args->fit_param = g_renew(gdouble, args->fit_param, nparams);
    args->fixed = g_renew(gboolean, args->fixed, nparams);
    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    gwy_fit_table_resize(table, nparams);
    gwy_fit_table_clear_errors(table);
    fit_param_table_update_units(gui);
    /* TODO: Try to initialise parameter values. */
    for (guint i = 0; i < nparams; i++)
        gwy_fit_table_set_value(table, i, 0.0);
}

static void
fit_param_table_update_units(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    gboolean abscissa_enabled = gwy_params_get_boolean(args->params, PARAM_ENABLE_ABSCISSA);
    guint nparams = gwy_nlfit_preset_get_nparams(args->fitfunc);
    gint abscissa = gwy_params_get_int(args->params, PARAM_ABSCISSA);
    gint ordinate = gwy_params_get_int(args->params, PARAM_ORDINATE);
    GwyUnit *xunit, *yunit;

    if (abscissa_enabled)
        xunit = g_object_ref(gwy_lawn_get_unit_curve(args->lawn, abscissa));
    else
        xunit = gwy_unit_new(NULL);
    yunit = gwy_lawn_get_unit_curve(args->lawn, ordinate);

    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    for (guint i = 0; i < nparams; i++) {
        gwy_fit_table_set_checked(table, i, FALSE);
        gwy_fit_table_set_name(table, i, gwy_nlfit_preset_get_param_name(args->fitfunc, i));
        GwyUnit *unit = gwy_nlfit_preset_get_param_units(args->fitfunc, i, xunit, yunit);
        gwy_fit_table_set_unit(table, i, unit);
        g_object_unref(unit);
    }

    g_object_unref(xunit);
}

static void
set_selection(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    gint col = gwy_params_get_int(args->params, PARAM_XPOS);
    gint row = gwy_params_get_int(args->params, PARAM_YPOS);
    gdouble xy[2];

    xy[0] = (col + 0.5)*gwy_lawn_get_dx(args->lawn);
    xy[1] = (row + 0.5)*gwy_lawn_get_dy(args->lawn);
    gwy_selection_set_object(gui->selection, 0, xy);
}

static void
point_selection_changed(ModuleGUI *gui, gint id, GwySelection *selection)
{
    ModuleArgs *args = gui->args;
    GwyLawn *lawn = args->lawn;
    gint i, xres = gwy_lawn_get_xres(lawn), yres = gwy_lawn_get_yres(lawn);
    gdouble xy[2];

    gwy_selection_get_object(selection, id, xy);
    i = GWY_ROUND(floor(xy[0]/gwy_lawn_get_dx(lawn)));
    gwy_params_set_int(args->params, PARAM_XPOS, CLAMP(i, 0, xres-1));
    i = GWY_ROUND(floor(xy[1]/gwy_lawn_get_dy(lawn)));
    gwy_params_set_int(args->params, PARAM_YPOS, CLAMP(i, 0, yres-1));

    gwy_param_table_param_changed(gui->table, PARAM_XPOS);
    gwy_param_table_param_changed(gui->table, PARAM_YPOS);
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;
    gint col = gwy_params_get_int(params, PARAM_XPOS);
    gint row = gwy_params_get_int(params, PARAM_YPOS);
    gdouble from = gwy_params_get_double(params, PARAM_RANGE_FROM);
    gdouble to = gwy_params_get_double(params, PARAM_RANGE_TO);
    gboolean segment_enabled = args->nsegments ? gwy_params_get_boolean(params, PARAM_ENABLE_SEGMENT) : FALSE;
    gint segment = segment_enabled ? gwy_params_get_int(params, PARAM_SEGMENT) : -1;

    GwyGraphCurveModel *datacmodel = gwy_graph_model_get_curve(gui->gmodel, 0);
    extract_one_curve(args->lawn, datacmodel, col, row, segment, params);
    update_graph_model_props(gui->gmodel, args);
    fit_param_table_update_units(gui);

    gdouble xfrom, xto;
    gwy_graph_curve_model_get_x_range(datacmodel, &xfrom, &xto);
    args->xmin = xfrom;
    args->xmax = xto;
    gdouble sel[2] = { xfrom + from*(xto-xfrom), xfrom + to*(xto-xfrom) };
    GwyGraphCurveModel *fitcmodel = gwy_graph_model_get_curve(gui->gmodel, 1);
    gwy_graph_curve_model_set_data(fitcmodel, NULL, NULL, 0);

    gwy_selection_set_data(gui->graph_selection, 1, sel);
    update_sensitivity(gui);
}

static void
update_sensitivity(ModuleGUI *gui)
{
    GwyGraphCurveModel *gcmodel = gwy_graph_model_get_curve(gui->gmodel, 0);
    guint ndata = gwy_graph_curve_model_get_ndata(gcmodel);
    guint nparams = gwy_nlfit_preset_get_nparams(gui->args->fitfunc);
    /* FIXME: There are other things limiting the number of points (selected range, segments). The code actually
     * extracting the data to fit is scattered through too many functions. Do not try to replicate it here before
     * a substantial cleanup. */
    gboolean fittable = ndata > nparams;
    gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), RESPONSE_ESTIMATE, fittable);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), RESPONSE_FIT, fittable);
}

static gboolean
execute(ModuleArgs *args, GtkWindow *window)
{
    GwyParams *params = args->params;
    gint abscissa = gwy_params_get_int(params, PARAM_ABSCISSA);
    gint ordinate = gwy_params_get_int(params, PARAM_ORDINATE);
    gboolean abscissa_enabled = gwy_params_get_boolean(params, PARAM_ENABLE_ABSCISSA);
    gdouble from = gwy_params_get_double(params, PARAM_RANGE_FROM);
    gdouble to = gwy_params_get_double(params, PARAM_RANGE_TO);
    gboolean segment_enabled = args->nsegments ? gwy_params_get_boolean(params, PARAM_ENABLE_SEGMENT) : FALSE;
    gint segment = segment_enabled ? gwy_params_get_int(params, PARAM_SEGMENT) : -1;
    gboolean estimate = gwy_params_get_boolean(params, PARAM_ESTIMATE);
    gboolean report_err = gwy_params_get_boolean(params, PARAM_REPORT_ERR);
    gboolean report_chi = gwy_params_get_boolean(params, PARAM_REPORT_CHI);
    GwyUnit *unitx, *unity;
    GwyGraphCurveModel *gc = gwy_graph_curve_model_new();
    GwyNLFitPreset *fitfunc = GWY_NLFIT_PRESET(gwy_params_get_resource(params, PARAM_FUNCTION));

    gint nparams = gwy_nlfit_preset_get_nparams(fitfunc);
    gdouble fit_param[nparams], errors[nparams];
    gboolean fixed[nparams];
    GwyLawn *lawn = args->lawn;
    gdouble **rdata;
    gdouble chisq;
    gint xres = gwy_lawn_get_xres(lawn), yres = gwy_lawn_get_yres(lawn);
    gboolean fitok, cancelled = FALSE;
    gint nchannels;

    nchannels = nparams + report_err*nparams + report_chi;
    if (nchannels <= args->result_fields->len)
        g_ptr_array_set_size(args->result_fields, nchannels);
    else {
        while (args->result_fields->len < nchannels)
            g_ptr_array_add(args->result_fields, gwy_lawn_new_field_alike(lawn, -1, TRUE));
    }
    rdata = g_new0(gdouble*, nchannels);

    if (abscissa_enabled)
        unitx = g_object_ref(gwy_lawn_get_unit_curve(args->lawn, abscissa));
    else
        unitx = gwy_unit_new(NULL);
    unity = gwy_lawn_get_unit_curve(args->lawn, ordinate);

    guint n = 0;
    for (guint i = 0; i < nparams; i++) {
        GwyField *result_field = g_ptr_array_index(args->result_fields, n);
        GwyUnit *unit = gwy_nlfit_preset_get_param_units(fitfunc, n, unitx, unity);
        gwy_unit_assign(gwy_field_get_unit_z(result_field), unit);
        g_object_unref(unit);
        rdata[n] = gwy_field_get_data(result_field);
        n++;
    }
    if (report_err) {
        for (guint i = 0; i < nparams; i++) {
            GwyField *result_field = g_ptr_array_index(args->result_fields, n);
            GwyUnit *unit = gwy_nlfit_preset_get_param_units(fitfunc, i, unitx, unity);
            gwy_unit_assign(gwy_field_get_unit_z(result_field), unit);
            g_object_unref(unit);
            rdata[n] = gwy_field_get_data(result_field);
            n++;
        }
    }
    if (report_chi) {
        GwyField *result_field = g_ptr_array_index(args->result_fields, n);
        gwy_unit_clear(gwy_field_get_unit_z(result_field));
        rdata[n] = gwy_field_get_data(result_field);
    }
    g_assert(n == args->result_fields->len);
    g_object_unref(unitx);

    args->mask = gwy_nield_new(gwy_lawn_get_xres(lawn), gwy_lawn_get_yres(lawn));
    gint *mdata = gwy_nield_get_data(args->mask);

    gwy_app_wait_start(window, _("Fitting..."));

    for (guint k = 0; k < xres*yres; k++) {
        if (!gwy_app_wait_set_fraction((gdouble)k/(xres*yres))) {
            cancelled = TRUE;
            break;
        }

        guint col = k % xres;
        guint row = k/xres;

        extract_one_curve(lawn, gc, col, row, segment, params);
        guint ndata = gwy_graph_curve_model_get_ndata(gc);
        if (!ndata) {
            mdata[k] = 1;
            continue;
        }

        const gdouble *cdx = gwy_graph_curve_model_get_xdata(gc);
        const gdouble *cdy = gwy_graph_curve_model_get_ydata(gc);

        if (estimate)
            do_estimate(cdx, cdy, ndata, fitfunc, from, to, fit_param);
        else
            gwy_assign(fit_param, args->fit_param, nparams);

        chisq = do_fit(cdx, cdy, ndata, fitfunc, from, to, fit_param, fixed, errors, &fitok);

        for (guint j = 0; j < nparams; j++) {
             rdata[j][k] = fit_param[j];
             if (report_err)
                 rdata[j + nparams][k] = errors[j];
        }
        if (report_chi)
            rdata[nparams + nparams*report_err][k] = chisq;

        if (!fitok)
            mdata[k] = 1;
    }

    if (!cancelled && gwy_nield_max(args->mask) > 0) {
        for (guint i = 0; i < nparams; i++) {
            GwyField *result_field = g_ptr_array_index(args->result_fields, i);
            gwy_field_laplace_solve(result_field, args->mask, GWY_LAPLACE_MASKED, 1.0);
        }
    }

    gwy_app_wait_finish();

    return !cancelled;
}

static void
extract_one_curve(GwyLawn *lawn, GwyGraphCurveModel *gcmodel,
                  gint col, gint row, gint segment,
                  GwyParams *params)
{
    gint abscissa = gwy_params_get_int(params, PARAM_ABSCISSA);
    gint ordinate = gwy_params_get_int(params, PARAM_ORDINATE);
    gboolean abscissa_enabled = gwy_params_get_boolean(params, PARAM_ENABLE_ABSCISSA);
    const gdouble *xdata, *ydata;
    gdouble *samplenodata = NULL;
    gint ndata;
    gchar *s;

    s = g_strdup_printf("x: %d, y: %d", col, row);
    g_object_set(gcmodel, "description", s, NULL);
    g_free(s);

    ydata = gwy_lawn_get_curve_data_const(lawn, col, row, ordinate, &ndata);
    if (abscissa_enabled)
        xdata = gwy_lawn_get_curve_data_const(lawn, col, row, abscissa, NULL);
    else
        xdata = samplenodata = gwy_math_linspace(NULL, ndata, 0, 1);

    if (segment >= 0) {
        const gint *segments = gwy_lawn_get_segments(lawn, col, row, NULL);
        gint from = segments[2*segment];
        gint end = segments[2*segment + 1];
        xdata += from;
        ydata += from;
        ndata = end - from;
    }
    gwy_graph_curve_model_set_data(gcmodel, xdata, ydata, ndata);
    g_free(samplenodata);
}

static void
do_estimate(const gdouble *xdata, const gdouble *ydata,
            gint ndata, GwyNLFitPreset *fitfunc,
            gdouble from, gdouble to,
            gdouble *fitparams)
{
    gboolean fres;
    gint i, j, n;
    gdouble startval, endval, xmin, xmax, ymin, ymax;
    gdouble *xf, *yf;

    //get the total range and fit range
    xmin = xmax = xdata[0];
    ymin = ymax = ydata[0];
    for (i = 1; i < ndata; i++) {
        xmin = fmin(xmin, xdata[i]);
        xmax = fmax(xmax, xdata[i]);
        ymin = fmin(ymin, ydata[i]);
        ymax = fmax(ymax, ydata[i]);
    }
    startval = xmin + from*(xmax-xmin);
    endval = xmin + to*(xmax-xmin);

    //determine number of points to fit
    n = 0;
    for (i = 0; i < ndata; i++) {
        if (xdata[i] >= startval && xdata[i] < endval)
            n++;
    }

    //fill the data to fit
    xf = g_new(gdouble, n);
    yf = g_new(gdouble, n);
    j = 0;
    for (i = 0; i < ndata; i++) {
        if (xdata[i] >= startval && xdata[i] < endval) {
            xf[j] = xdata[i];
            yf[j] = ydata[i];
            j++;
        }
    }

    gwy_nlfit_preset_guess(fitfunc, n, xf, yf, fitparams, &fres);

    g_free(xf);
    g_free(yf);
}

static gdouble
do_fit(const gdouble *xdata, const gdouble *ydata,
       gint ndata, GwyNLFitPreset *fitfunc,
       gdouble from, gdouble to,
       gdouble *fitparams, gboolean *fix, gdouble *error,
       gboolean *fitok)
{
    gint i, j, n;
    gdouble startval, endval, xmin, xmax, ymin, ymax, chisq;
    gdouble *xf, *yf;
    GwyNLFitter *fitter;

    //get the total range and fit range
    xmin = xmax = xdata[0];
    ymin = ymax = ydata[0];
    for (i = 1; i < ndata; i++) {
        xmin = fmin(xmin, xdata[i]);
        xmax = fmax(xmax, xdata[i]);
        ymin = fmin(ymin, ydata[i]);
        ymax = fmax(ymax, ydata[i]);
    }
    startval = xmin + from*(xmax-xmin);
    endval = xmin + to*(xmax-xmin);

    //determine number of points to fit
    n = 0;
    for (i = 0; i < ndata; i++) {
        if (xdata[i] >= startval && xdata[i] < endval)
            n++;
    }

    //fill the data to fit
    xf = g_new(gdouble, n);
    yf = g_new(gdouble, n);
    j = 0;
    for (i = 0; i < ndata; i++) {
        if (xdata[i] >= startval && xdata[i] < endval) {
            xf[j] = xdata[i];
            yf[j] = ydata[i];
            j++;
        }
    }

    fitter = gwy_nlfit_preset_fit(fitfunc, NULL, n, xf, yf, fitparams, error, fix);

    *fitok = gwy_math_nlfit_succeeded(fitter);

    if (*fitok)
        chisq = gwy_math_nlfit_get_dispersion(fitter);
    else
        chisq = -1;

    g_free(xf);
    g_free(yf);
    gwy_math_nlfit_free(fitter);

    return chisq;
}

static gdouble
fit_one_curve(GwyGraphCurveModel *gcmodel, GwyParams *params, GwyNLFitPreset *fitfunc,
              gdouble *fitparams, gboolean *fix, gdouble *error,
              gboolean *fitok)
{
    gdouble from = gwy_params_get_double(params, PARAM_RANGE_FROM);
    gdouble to = gwy_params_get_double(params, PARAM_RANGE_TO);

    const gdouble *xdata = gwy_graph_curve_model_get_xdata(gcmodel);
    const gdouble *ydata = gwy_graph_curve_model_get_ydata(gcmodel);
    gint ndata = gwy_graph_curve_model_get_ndata(gcmodel);

    return do_fit(xdata, ydata, ndata, fitfunc, from, to, fitparams, fix, error, fitok);
}

static void
estimate_one_curve(GwyGraphCurveModel *gcmodel, GwyParams *params, GwyNLFitPreset *fitfunc,
                   gdouble *fitparams)
{
    gdouble from = gwy_params_get_double(params, PARAM_RANGE_FROM);
    gdouble to = gwy_params_get_double(params, PARAM_RANGE_TO);

    const gdouble *xdata = gwy_graph_curve_model_get_xdata(gcmodel);
    const gdouble *ydata = gwy_graph_curve_model_get_ydata(gcmodel);
    gint ndata = gwy_graph_curve_model_get_ndata(gcmodel);

    do_estimate(xdata, ydata, ndata, fitfunc, from, to, fitparams);
}

static void
update_graph_model_props(GwyGraphModel *gmodel, ModuleArgs *args)
{
    GwyLawn *lawn = args->lawn;
    GwyParams *params = args->params;
    gint abscissa = gwy_params_get_int(params, PARAM_ABSCISSA);
    gint ordinate = gwy_params_get_int(params, PARAM_ORDINATE);
    gboolean abscissa_enabled = gwy_params_get_boolean(params, PARAM_ENABLE_ABSCISSA);
    GwyUnit *xunit, *yunit;
    const gchar *xlabel, *ylabel;

    if (abscissa_enabled) {
        xunit = gwy_lawn_get_unit_curve(lawn, abscissa);
        xlabel = gwy_lawn_get_curve_label(lawn, abscissa);
    }
    else {
        xunit = gwy_unit_new(NULL);
        xlabel = g_strdup(_("sample"));
    }
    yunit = gwy_lawn_get_unit_curve(lawn, ordinate);
    ylabel = gwy_lawn_get_curve_label(lawn, ordinate);

    g_object_set(gmodel,
                 "unit-x", xunit,
                 "unit-y", yunit,
                 "axis-label-bottom", xlabel ? xlabel : _("Untitled"),
                 "axis-label-left", ylabel ? ylabel : _("Untitled"),
                 NULL);

    if (!abscissa_enabled)
        g_object_unref(xunit);
}

static GtkWidget*
create_fit_table(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;

    gui->fit_params = gwy_fit_table_new();
    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    gwy_fit_table_set_values_editable(table, TRUE);
    gwy_fit_table_set_has_checkboxes(table, TRUE);
    gwy_fit_table_set_header(table, _("Fix"), 0, 1);
    gwy_fit_table_set_header(table, _("Parameter"), 1, 3);
    gwy_fit_table_set_header(table, _("Error"), 6, 2);
    g_signal_connect_swapped(gui->fit_params, "activated", G_CALLBACK(param_value_activate), gui);

    return gui->fit_params;
}

static void
param_value_activate(ModuleGUI *gui, G_GNUC_UNUSED guint i)
{
    plot_result(gui);
}

static void
sanitise_one_param(GwyParams *params, gint id, gint min, gint max, gint defval)
{
    gint v;

    v = gwy_params_get_int(params, id);
    if (v >= min && v <= max) {
        gwy_debug("param #%d is %d, i.e. within range [%d..%d]", id, v, min, max);
        return;
    }
    gwy_debug("param #%d is %d, setting it to the default %d", id, v, defval);
    gwy_params_set_int(params, id, defval);
}

static void
sanitise_params(ModuleArgs *args)
{
    GwyParams *params = args->params;
    GwyLawn *lawn = args->lawn;

    sanitise_one_param(params, PARAM_XPOS, 0, gwy_lawn_get_xres(lawn)-1, gwy_lawn_get_xres(lawn)/2);
    sanitise_one_param(params, PARAM_YPOS, 0, gwy_lawn_get_yres(lawn)-1, gwy_lawn_get_yres(lawn)/2);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
