/*
 *  $Id: accurexii-txt.c 29477 2026-02-14 13:29:30Z yeti-dn $
 *  Copyright (C) 2014-2025 David Necas (Yeti).
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
 * [FILE-MAGIC-FREEDESKTOP]
 * <mime-type type="application/x-accurexii-txt">
 *   <comment>Accurex II text data</comment>
 *   <magic priority="80">
 *     <match type="string" offset="0" value="[Header Section]">
 *       <match type="string" offset="16:240" value="Stage Type"/>
 *     </match>
 *   </magic>
 * </mime-type>
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * Accurex II text data
 * .txt
 * Read
 **/

#include "config.h"
#include <glib/gi18n-lib.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <gwy.h>

#include "err.h"

#define MAGIC "[Header Section]"
#define MAGIC_SIZE (sizeof(MAGIC)-1)
#define MAGIC2 "Stage Type"
#define MAGIC3 "Probe Type"
#define DATA_MAGIC "[Data Section]"
#define EXTENSION ".txt"

static gboolean module_register(void);
static gint     detect_file    (const GwyFileDetectInfo *fileinfo,
                                gboolean only_name);
static GwyFile* load_file      (const gchar *filename,
                                GwyRunModeFlags mode,
                                GError **error);
static void     add_meta       (gpointer hkey,
                                gpointer hvalue,
                                gpointer user_data);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Imports Accurex II text files."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti)",
    "2014",
};

GWY_MODULE_QUERY2(module_info, accurexii_txt)

static gboolean
module_register(void)
{
    gwy_file_func_register("accurexii-txt",
                           N_("Accurex II text files (.txt)"),
                           detect_file, load_file, NULL, NULL);

    return TRUE;
}

static gint
detect_file(const GwyFileDetectInfo *fileinfo, gboolean only_name)
{
    if (only_name)
        return g_str_has_suffix(fileinfo->name_lowercase, EXTENSION) ? 10 : 0;

    if (fileinfo->file_size < MAGIC_SIZE || memcmp(fileinfo->head, MAGIC, MAGIC_SIZE) != 0)
        return 0;

    if (!strstr(fileinfo->head, MAGIC2) || !strstr(fileinfo->head, MAGIC3))
        return 0;

    return 90;
}

static GwyFile*
load_file(const gchar *filename,
          G_GNUC_UNUSED GwyRunModeFlags mode,
          GError **error)
{
    GwyFile *file = NULL;
    GwyDict *meta;
    GwyField *dfield = NULL;
    GwyUnit *xunit, *yunit, *zunit;
    gchar *end, *p, *value, *header = NULL, *buffer = NULL;
    gchar **dims;
    GwyTextHeaderParser parser;
    GHashTable *hash = NULL;
    gsize size;
    GError *err = NULL;
    gdouble xreal, yreal, q;
    guint xres, yres;
    gint power10x, power10y, power10z;

    if (!g_file_get_contents(filename, &buffer, &size, &err)) {
        err_GET_FILE_CONTENTS(error, &err);
        goto fail;
    }

    /* This one requires no string search. */
    if (memcmp(buffer, MAGIC, MAGIC_SIZE)) {
        err_FILE_TYPE(error, "Accurex II TXT");
        goto fail;
    }

    if (!(p = strstr(buffer + MAGIC_SIZE, DATA_MAGIC))) {
        err_FILE_TYPE(error, "Accurex II TXT");
        goto fail;
    }

    header = gwy_convert_to_utf8(buffer + MAGIC_SIZE, p - buffer - MAGIC_SIZE, "ISO-8859-1");
    if (!header) {
        /* XXX: Can this actually happen? */
        g_set_error(error, GWY_MODULE_FILE_ERROR, GWY_MODULE_FILE_ERROR_DATA,
                    _("File header cannot be converted from ISO-8859-1 character set."));
        goto fail;
    }

    gwy_clear1(parser);
    parser.key_value_separator = "=";
    hash = gwy_text_header_parse(header, &parser, NULL, NULL);

    if (!require_keys(hash, error, "Image Size", "Image Resolution", NULL))
        goto fail;

    value = (gchar*)g_hash_table_lookup(hash, "Image Resolution");
    gwy_debug("imgres <%s>", value);
    if (sscanf(value, "%u x %u", &xres, &yres) != 2) {
        err_INVALID(error, "Image Resolution");
        goto fail;
    }
    gwy_debug("xres %u, yres %u", xres, yres);
    if (err_DIMENSION(error, xres) || err_DIMENSION(error, yres))
        goto fail;

    value = (gchar*)g_hash_table_lookup(hash, "Image Size");
    dims = g_strsplit(value, "x", 0);
    if (!dims || g_strv_length(dims) != 2) {
        g_strfreev(dims);
        err_INVALID(error, "Image Size");
        goto fail;
    }
    xreal = g_ascii_strtod(dims[0], &end);
    xunit = gwy_unit_new_parse(end, &power10x);
    yreal = g_ascii_strtod(dims[1], &end);
    yunit = gwy_unit_new_parse(end, &power10y);
    g_strfreev(dims);
    sanitise_real_size(&xreal, "x size");
    sanitise_real_size(&yreal, "y size");
    if (!gwy_unit_equal(xunit, yunit)) {
        g_warning("X and Y units differ, using X");
    }

    xreal *= gwy_exp10(power10x);
    yreal *= gwy_exp10(power10y);

    dfield = gwy_field_new(xres, yres, xreal, yreal, FALSE);
    gwy_unit_assign(gwy_field_get_unit_xy(dfield), xunit);

    g_object_unref(xunit);
    g_object_unref(yunit);

    p += strlen(DATA_MAGIC);
    while (g_ascii_isspace(*p))
        p++;

    q = 1.0;
    if (g_str_has_prefix(p, "Z-unit:")) {
        value = gwy_str_next_line(&p);
        zunit = gwy_unit_new_parse(value + strlen("Z unit:"), &power10z);
        gwy_unit_assign(gwy_field_get_unit_z(dfield), zunit);
        g_object_unref(zunit);
        q *= gwy_exp10(power10z);
    }

    if (!gwy_parse_doubles(p, gwy_field_get_data(dfield), GWY_PARSE_DOUBLES_FREE_FORM, &yres, &xres, NULL, &err)) {
        err_PARSE_DOUBLES(error, &err);
        g_clear_object(&dfield);
        goto fail;
    }
    gwy_field_multiply(dfield, q);

    file = gwy_file_new_in_construction();

    gwy_file_pass_image(file, 0, dfield);
    if ((value = g_hash_table_lookup(hash, "Data Type")))
        gwy_file_set_title(file, GWY_FILE_IMAGE, 0, value, FALSE);
    meta = gwy_dict_new();
    g_hash_table_foreach(hash, add_meta, meta);
    gwy_file_pass_meta(file, GWY_FILE_IMAGE, 0, meta);
    gwy_log_add_import(file, GWY_FILE_IMAGE, 0, NULL, filename);

fail:
    g_free(header);
    g_free(buffer);
    if (hash)
        g_hash_table_destroy(hash);

    return file;
}

static void
add_meta(gpointer hkey, gpointer hvalue, gpointer user_data)
{
    gwy_dict_set_const_string_by_name(GWY_DICT(user_data), (gchar*)hkey, hvalue);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
