/*
 *  $Id: graph_cd.c 29477 2026-02-14 13:29:30Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <gwy.h>

#include "libgwyapp/sanity.h"
#include "../process/preview.h"

enum {
    PARAM_FITFUNC,
    PARAM_CURVE,
    PARAM_FULL_CIRCLE,
    PARAM_REPORT_STYLE,
    PARAM_RANGE_FROM,
    PARAM_RANGE_TO,

    MESSAGE_ERROR,
    WIDGET_DRAWING,
    WIDGET_FIT_RESULT,
};

typedef struct {
    GwyParams *params;
    GwyGraphModel *gmodel;
    GwyCDLine *fitfunc;
    GArray *xdata;
    GArray *ydata;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GwyFile *file;
    GtkWidget *dialog;
    GwyParamTable *table;
    GwyGraphModel *gmodel;
    GtkWidget *formula;
    GtkWidget *fit_params;
    GwyResults *results;
    gboolean fitted;
} ModuleGUI;

static gboolean         module_register        (void);
static GwyParamDef*     define_module_params   (void);
static void             module_main            (GwyGraph *graph);
static GwyDialogOutcome run_gui                (ModuleArgs *args,
                                                GwyFile *file);
static GtkWidget*       create_fit_table       (gpointer user_data);
static GtkWidget*       create_function_drawing(gpointer user_data);
static void             param_changed          (ModuleGUI *gui,
                                                gint id);
static void             fit_param_table_resize (ModuleGUI *gui);
static void             dialog_response        (ModuleGUI *gui,
                                                gint response);
static void             execute                (ModuleGUI *gui);
static void             plot_curve             (ModuleGUI *gui);
static void             update_function_image  (ModuleGUI *gui);
static void             update_fit_state       (ModuleGUI *gui,
                                                gboolean is_fitted);
static gint             extract_data           (ModuleArgs *args);
static void             create_results         (ModuleGUI *gui);
static void             fill_results           (ModuleGUI *gui);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Critical dimension measurements"),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "3.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, graph_cd)

static gboolean
module_register(void)
{
    gwy_graph_func_register("graph_cd",
                            &module_main,
                            N_("/Measure _Features/_Critical Dimension..."),
                            GWY_ICON_GRAPH_MEASURE,
                            GWY_MENU_FLAG_GRAPH_CURVE,
                            N_("Fit critical dimension"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_graph_func_current());
    gwy_param_def_add_resource(paramdef, PARAM_FITFUNC, "fitfunc", _("_Function"),
                               gwy_cdlines(), "Step height (positive)");
    gwy_param_def_add_graph_curve(paramdef, PARAM_CURVE, "curve", NULL);
    gwy_param_def_add_boolean(paramdef, PARAM_FULL_CIRCLE, "full-circle", _("_Draw full circle"), FALSE);
    gwy_param_def_add_report_type(paramdef, PARAM_REPORT_STYLE, "report_style", NULL,
                                  GWY_RESULTS_EXPORT_PARAMETERS, GWY_RESULTS_REPORT_COLON | GWY_RESULTS_REPORT_MACHINE);
    /* Range is GUI-only. */
    gwy_param_def_add_double(paramdef, PARAM_RANGE_FROM, NULL, _("Range"), -G_MAXDOUBLE, G_MAXDOUBLE, 0.0);
    gwy_param_def_add_double(paramdef, PARAM_RANGE_TO, NULL, NULL, -G_MAXDOUBLE, G_MAXDOUBLE, 0.0);

    return paramdef;
}

static void
module_main(GwyGraph *graph)
{
    ModuleArgs args;
    gwy_clear1(args);

    GwyFile *file;
    gwy_data_browser_get_current(GWY_APP_FILE, &file, 0);

    args.params = gwy_params_new_from_settings(define_module_params());
    args.gmodel = gwy_graph_get_model(graph);
    args.xdata = g_array_new(FALSE, FALSE, sizeof(gdouble));
    args.ydata = g_array_new(FALSE, FALSE, sizeof(gdouble));

    run_gui(&args, file);

    gwy_params_save_to_settings(args.params);

    g_object_unref(args.params);
    g_array_free(args.xdata, TRUE);
    g_array_free(args.ydata, TRUE);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyFile *file)
{
    ModuleGUI gui;

    gwy_clear1(gui);
    gui.args = args;
    gui.file = file;

    gui.dialog = gwy_dialog_new(_("Critical Dimension"));
    GwyDialog *dialog = GWY_DIALOG(gui.dialog);
    gwy_add_button_to_dialog(GTK_DIALOG(dialog), GWY_STOCK_COPY, GWY_ICON_GTK_COPY, RESPONSE_COPY);
    gwy_add_button_to_dialog(GTK_DIALOG(dialog), GWY_STOCK_SAVE, GWY_ICON_GTK_SAVE, RESPONSE_SAVE);
    gwy_add_button_to_dialog(GTK_DIALOG(dialog), C_("verb", "_Fit"), GWY_ICON_GTK_EXECUTE, RESPONSE_REFINE);
    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    GtkWidget *hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 4);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
    gwy_dialog_add_content(dialog, hbox, TRUE, TRUE, 0);

    gui.gmodel = gwy_graph_model_new_alike(args->gmodel);
    GtkWidget *graph = gwy_graph_new(gui.gmodel);
    gtk_widget_set_size_request(graph, 400, 300);
    gwy_graph_enable_user_input(GWY_GRAPH(graph), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), graph, TRUE, TRUE, 0);

    GwyParamTable *table = gui.table = gwy_param_table_new(args->params);
    gwy_param_table_append_graph_curve(table, PARAM_CURVE, args->gmodel);
    gwy_param_table_append_combo(table, PARAM_FITFUNC);
    gwy_param_table_append_foreign(table, WIDGET_DRAWING, create_function_drawing, &gui, NULL);
    gwy_param_table_append_foreign(table, WIDGET_FIT_RESULT, create_fit_table, &gui, NULL);
    gwy_param_table_append_message(table, MESSAGE_ERROR, NULL);
    gwy_param_table_append_separator(table);
    gwy_create_graph_xrange_with_params(table, PARAM_RANGE_FROM, PARAM_RANGE_TO, GWY_GRAPH(graph), args->gmodel);
    gwy_param_table_append_checkbox(table, PARAM_FULL_CIRCLE);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), FALSE, TRUE, 0);

    g_signal_connect_swapped(gui.table, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(dialog, "response", G_CALLBACK(dialog_response), &gui);

    GwyDialogOutcome outcome = gwy_dialog_run(dialog);

    if (outcome == GWY_DIALOG_PROCEED) {
        if (gwy_graph_model_get_n_curves(gui.gmodel) > 1)
            gwy_graph_model_add_curve(args->gmodel, gwy_graph_model_get_curve(gui.gmodel, 1));
    }

    g_object_unref(gui.gmodel);
    g_clear_object(&gui.results);

    return outcome;
}

static GtkWidget*
create_fit_table(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    gui->fit_params = gwy_fit_table_new();
    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    gwy_fit_table_set_header(table, _("Parameter"), 0, 3);
    gwy_fit_table_set_header(table, _("Error"), 4, 2);
    return gui->fit_params;
}

static GtkWidget*
create_function_drawing(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    gui->formula = gtk_image_new();
    return gui->formula;
}

static void
fit_param_table_resize(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    guint nparams = gwy_cdline_get_nparams(args->fitfunc);
    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    gwy_fit_table_resize(table, nparams);

    GwyUnit *xunit, *yunit;
    g_object_get(args->gmodel, "unit-x", &xunit, "unit-y", &yunit, NULL);

    for (guint i = 0; i < nparams; i++) {
        gwy_fit_table_set_name(table, i, gwy_cdline_get_param_name(args->fitfunc, i));
        GwyUnit *unit = gwy_cdline_get_param_units(args->fitfunc, i, xunit, yunit);
        gwy_fit_table_set_unit(table, i, unit);
        g_object_unref(unit);
    }
    g_object_unref(xunit);
    g_object_unref(yunit);
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;
    gboolean set_unfitted = FALSE;

    if (id < 0 || id == PARAM_FITFUNC) {
        args->fitfunc = GWY_CDLINE(gwy_params_get_resource(params, PARAM_FITFUNC));
        const gchar *funcname = gwy_params_get_string(args->params, PARAM_FITFUNC);
        gwy_param_table_set_sensitive(gui->table, PARAM_FULL_CIRCLE,
                                      gwy_stramong(funcname, "Circle (down)", "Circle (up)", NULL));
        update_function_image(gui);
        fit_param_table_resize(gui);
        create_results(gui);
        set_unfitted = TRUE;
    }

    if (id < 0 || id == PARAM_CURVE) {
        gwy_graph_model_remove_all_curves(gui->gmodel);
        gwy_graph_model_add_curve(gui->gmodel,
                                  gwy_graph_model_get_curve(args->gmodel,
                                                            gwy_params_get_int(params, PARAM_CURVE)));
        set_unfitted = TRUE;
    }

    if (id == PARAM_RANGE_FROM || id == PARAM_RANGE_TO)
        set_unfitted = TRUE;

    if (set_unfitted)
        update_fit_state(gui, FALSE);

    if (id < 0 || id == PARAM_FULL_CIRCLE) {
        if (gui->fitted)
            plot_curve(gui);
    }
}

static GwyGraphCurveModel*
ensure_model_curve(GwyGraphModel *gmodel, GwyGraphModel *parent_gmodel, gint cindex,
                   const gchar *description)
{
    GwyGraphCurveModel *gcmodel;
    if (gwy_graph_model_get_n_curves(gmodel) < cindex+1) {
        const GwyRGBA *color = gwy_graph_get_preset_color(gwy_graph_model_get_n_curves(parent_gmodel));
        gcmodel = gwy_graph_curve_model_new();
        g_object_set(gcmodel,
                     "mode", GWY_GRAPH_CURVE_LINE,
                     "color", color,
                     "line-style", GWY_GRAPH_LINE_DASH_DENSE,
                     "line-width", 2,
                     NULL);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
    }
    else
        gcmodel = gwy_graph_model_get_curve(gmodel, cindex);

    g_object_set(gcmodel, "description", description, NULL);
    return gcmodel;
}

static void
plot_curve(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;

    guint nparams = gwy_cdline_get_nparams(args->fitfunc);
    gdouble param[nparams];
    gwy_fit_table_gather(GWY_FIT_TABLE(gui->fit_params), param, NULL, NULL);

    gboolean full_circle = gwy_params_get_boolean(args->params, PARAM_FULL_CIRCLE);
    const gchar *funcname = gwy_params_get_string(args->params, PARAM_FITFUNC);
    if (!gwy_stramong(funcname, "Circle (down)", "Circle (up)", NULL))
        full_circle = FALSE;

    GwyGraphModel *gmodel = gui->gmodel;
    while (gwy_graph_model_get_n_curves(gmodel) > 2)
        gwy_graph_model_remove_curve(gmodel, gwy_graph_model_get_n_curves(gmodel)-1);

    guint n = extract_data(args);
    gdouble *xd = &g_array_index(args->xdata, gdouble, 0);
    gdouble *yd = &g_array_index(args->ydata, gdouble, 0);
    if (full_circle) {
        gdouble r = param[0], x0 = param[1], y0 = param[2];
        for (guint i = 0; i < n; i++) {
            gdouble phi = 2*G_PI*i/(n - 1.0);
            xd[i] = x0 + r*sin(phi);
            yd[i] = y0 + r*cos(phi);
        }
    }
    else {
        gboolean ok = TRUE;
        for (guint i = 0; i < n; i++) {
            gdouble v = gwy_cdline_get_value(args->fitfunc, xd[i], param, &ok);
            yd[i] = v;
        }
    }
    GwyGraphCurveModel *gcmodel = ensure_model_curve(gmodel, args->gmodel, 1, _("Fit"));
    gwy_graph_curve_model_set_data(gcmodel, xd, yd, n);
}

static void
dialog_response(ModuleGUI *gui, gint response)
{
    if (response == RESPONSE_COPY || response == RESPONSE_SAVE) {
        GwyResultsExportStyle report_style = gwy_params_get_report_type(gui->args->params, PARAM_REPORT_STYLE);
        gchar *report = gwy_results_create_report(gui->results, report_style);

        if (response == RESPONSE_SAVE)
            gwy_save_auxiliary_data(_("Save Fit Report"), GTK_WINDOW(gui->dialog), report, -1, FALSE);
        else {
            GdkDisplay *display = gtk_widget_get_display(gui->dialog);
            GtkClipboard *clipboard = gtk_clipboard_get_for_display(display, GDK_SELECTION_CLIPBOARD);
            gtk_clipboard_set_text(clipboard, report, -1);
        }
        g_free(report);
    }
    else if (response == RESPONSE_REFINE) {
        execute(gui);
    }
}

static void
execute(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    guint nparams = gwy_cdline_get_nparams(args->fitfunc);
    gdouble param[nparams], error[nparams];

    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    gwy_fit_table_clear_errors(table);

    guint npoints;
    npoints = extract_data(args);
    if (npoints <= nparams) {
        gwy_param_table_set_label(gui->table, MESSAGE_ERROR, _("Not enough data points."));
        return;
    }
    gwy_param_table_set_label(gui->table, MESSAGE_ERROR, NULL);

    gdouble *xd = &g_array_index(args->xdata, gdouble, 0);
    gdouble *yd = &g_array_index(args->ydata, gdouble, 0);
    gwy_cdline_fit(args->fitfunc, npoints, xd, yd, param, error);

    for (guint i = 0; i < nparams; i++) {
        gwy_fit_table_set_value(table, i, param[i]);
        if (error[i] >= 0.0)
            gwy_fit_table_set_error(table, i, error[i]);
    }

    plot_curve(gui);
    update_fit_state(gui, TRUE);
}

static void
update_function_image(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;

    gchar *filename = g_strconcat(gwy_cdline_get_definition(args->fitfunc), ".svg", NULL);
    gchar *filepath = gwy_find_self_path("data", "drawings", filename, NULL);
    gtk_image_set_from_file(GTK_IMAGE(gui->formula), filepath);
    g_free(filepath);
    g_free(filename);
}

static void
update_fit_state(ModuleGUI *gui, gboolean is_fitted)
{
    gui->fitted = is_fitted;

    gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), GTK_RESPONSE_OK, is_fitted);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), RESPONSE_COPY, is_fitted);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), RESPONSE_SAVE, is_fitted);

    if (is_fitted) {
        fill_results(gui);
        return;
    }

    gwy_fit_table_clear_values(GWY_FIT_TABLE(gui->fit_params));
    gwy_fit_table_clear_errors(GWY_FIT_TABLE(gui->fit_params));

    GwyGraphModel *gmodel = gui->gmodel;
    while (gwy_graph_model_get_n_curves(gmodel) > 1)
        gwy_graph_model_remove_curve(gmodel, gwy_graph_model_get_n_curves(gmodel)-1);
}

/* Extract relevant part of data. */
static gint
extract_data(ModuleArgs *args)
{
    GwyParams *params = args->params;
    gdouble from = gwy_params_get_double(params, PARAM_RANGE_FROM);
    gdouble to = gwy_params_get_double(params, PARAM_RANGE_TO);

    GwyGraphCurveModel *cmodel = gwy_graph_model_get_curve(args->gmodel, gwy_params_get_int(params, PARAM_CURVE));
    const gdouble *xs = gwy_graph_curve_model_get_xdata(cmodel);
    const gdouble *ys = gwy_graph_curve_model_get_ydata(cmodel);
    guint ns = gwy_graph_curve_model_get_ndata(cmodel);

    g_array_set_size(args->xdata, ns);
    g_array_set_size(args->ydata, ns);
    gdouble *xd = &g_array_index(args->xdata, gdouble, 0);
    gdouble *yd = &g_array_index(args->ydata, gdouble, 0);

    guint n = 0;
    for (guint i = 0; i < ns; i++) {
        if (xs[i] >= from && xs[i] <= to) {
            xd[n] = xs[i];
            yd[n] = ys[i];
            n++;
        }
    }
    if (!n)
        return 0;

    if (n < ns) {
        g_array_set_size(args->xdata, n);
        g_array_set_size(args->ydata, n);
    }

    return n;
}

static void
create_results(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    GwyCDLine *fitfunc = args->fitfunc;

    g_clear_object(&gui->results);
    GwyResults *results = gui->results = gwy_results_new();
    gwy_results_add_header(results, N_("Fit Results"));
    gwy_results_add_value_str(results, "file", N_("File"));
    gwy_results_add_value_str(results, "channel", N_("Curve"));
    gwy_results_add_format(results, "npts", N_("Number of points"), TRUE,
    /* TRANSLATORS: %{n}i and %{total}i are ids, do NOT translate them. */
                           N_("%{n}i of %{ntotal}i"), NULL);
    gwy_results_add_value_str(results, "func", N_("Fitted function"));
    //gwy_results_add_value_z(results, "rss", N_("Mean square difference"));

    gwy_results_add_separator(results);
    gwy_results_add_header(results, N_("Parameters"));
    guint nparams = gwy_cdline_get_nparams(fitfunc);
    const gchar *names[nparams];
    for (guint i = 0; i < nparams; i++) {
        const gchar *name = names[i] = gwy_cdline_get_param_name(fitfunc, i);
        gint power_x = gwy_cdline_get_param_power_x(fitfunc, i);
        gint power_z = gwy_cdline_get_param_power_y(fitfunc, i);
        gwy_results_add_value(results, name, "",
                              "symbol", name, "is-fitting-param", TRUE,
                              "power-x", power_x, "power-z", power_z,
                              NULL);
    }
}

static void
fill_results(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    GwyCDLine *fitfunc = args->fitfunc;
    guint nparams = gwy_cdline_get_nparams(args->fitfunc);
    gdouble param[nparams], error[nparams];
    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    gwy_fit_table_gather(table, param, error, NULL);

    GwyResults *results = gui->results;
    GwyGraphModel *gmodel = gui->gmodel;
    GwyUnit *xunit, *yunit;
    g_object_get(gmodel,
                 "unit-x", &xunit,
                 "unit-y", &yunit,
                 NULL);
    gwy_results_set_unit(results, "x", xunit);
    gwy_results_set_unit(results, "y", yunit);
    g_object_unref(xunit);
    g_object_unref(yunit);

    GwyGraphCurveModel *gcmodel_data = gwy_graph_model_get_curve(gmodel, 0);
    GwyGraphCurveModel *gcmodel_fit = gwy_graph_model_get_curve(gmodel, 1);
    guint ntotal = gwy_graph_curve_model_get_ndata(gcmodel_data);
    guint n = gwy_graph_curve_model_get_ndata(gcmodel_fit);
    gchar *desc;
    g_object_get(gcmodel_data, "description", &desc, NULL);

    gwy_results_fill_filename(results, "file", gui->file);
    gwy_results_fill_values(results,
                            "func", gwy_resource_get_name(GWY_RESOURCE(fitfunc)),
                            "channel", desc,
                            NULL);
    g_free(desc);
    gwy_results_fill_format(results, "npts",
                            "n", n,
                            "ntotal", ntotal,
                            NULL);

    for (guint i = 0; i < nparams; i++) {
        const gchar *name = gwy_cdline_get_param_name(fitfunc, i);

        if (gwy_fit_table_error_is_set(table, i))
            gwy_results_fill_values_with_errors(results, name, param[i], error[i], NULL);
        else
            gwy_results_fill_values(results, name, param[i], NULL);
    }
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
