/*
 *  $Id: graph_fit.c 29477 2026-02-14 13:29:30Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <gwy.h>

#include "libgwyapp/sanity.h"
#include "../process/preview.h"

/* Lower symmetric part indexing */
/* i MUST be greater or equal than j */
#define SLi(a, i, j) a[(i)*((i) + 1)/2 + (j)]

enum {
    PARAM_FITFUNC,
    PARAM_CURVE,
    PARAM_PLOT_FULL,
    PARAM_AUTO_ESTIMATE,
    PARAM_OUT_DIFF,
    PARAM_REPORT_STYLE,
    PARAM_RANGE_FROM,
    PARAM_RANGE_TO,

    MESSAGE_ERROR,
    INFO_RSS,
};

typedef enum {
    FIT_INITIALISED,
    FIT_ESTIMATED,
    FIT_FITTED,
    FIT_USER_MODIFIED,
    FIT_ESTIMATE_FAILED,
    FIT_FIT_FAILED,
} FitState;

typedef struct {
    GwyParamDef *paramdef;
    GwyInventory *inventory;
    const gchar *dialog_title;
    const gchar *default_function;
} Submodule;

typedef struct {
    Submodule *specs;
    GwyParams *params;
    GwyGraphModel *gmodel;
    GwyNLFitPreset *fitfunc;
    GArray *xdata;
    GArray *ydata;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GwyFile *file;
    GtkWidget *dialog;
    GwyGraphModel *gmodel;
    GwySelection *selection;
    GwyParamTable *table_above;
    GwyParamTable *table_below;
    GtkWidget *revert;
    GtkWidget *fit_params;
    GtkWidget *correl_table;
    GtkWidget *function;
    GwyResults *results;
    GwyNLFitter *fitter;
    gdouble *correl;
    gdouble rss;
    FitState state;
} ModuleGUI;

static gboolean         module_register        (void);
static GwyParamDef*     define_module_params   (Submodule *specs);
static void             module_main            (GwyGraph *graph);
static GwyDialogOutcome run_gui                (ModuleArgs *args,
                                                GwyFile *file);
static GtkWidget*       create_param_tab       (ModuleGUI *gui);
static GtkWidget*       create_correl_tab      (ModuleGUI *gui);
static GtkWidget*       create_function_tab    (ModuleGUI *gui);
static void             fit_param_table_resize (ModuleGUI *gui);
static void             fit_correl_table_resize(ModuleGUI *gui);
static void             execute                (ModuleGUI *gui);
static void             estimate               (ModuleGUI *gui);
static void             param_changed          (ModuleGUI *gui,
                                                gint id);
static void             param_value_activate   (ModuleGUI *gui,
                                                guint i);
static void             preview                (gpointer user_data);
static void             dialog_response        (ModuleGUI *gui,
                                                gint response);
static void             revert_parameters      (ModuleGUI *gui);
static void             guess_parameters       (ModuleGUI *gui);
static void             plot_curve             (ModuleGUI *gui);
static void             update_fit_state       (ModuleGUI *gui,
                                                FitState state);
static gint             extract_data           (ModuleArgs *args,
                                                gboolean for_plotting);
static void             create_difference_graph(ModuleGUI *gui,
                                                GwyFile *file);
static void             create_results         (ModuleGUI *gui);
static void             fill_results           (ModuleGUI *gui);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Fit graph with functions."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "3.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, graph_fit)

static gboolean
module_register(void)
{
    gwy_graph_func_register("graph_fit",
                            module_main,
                            N_("/Measure _Features/_Fit Function..."),
                            GWY_ICON_GRAPH_FUNCTION,
                            GWY_MENU_FLAG_GRAPH_CURVE,
                            N_("Fit a function on graph data"));
    gwy_graph_func_register("graph_fdfit",
                            module_main,
                            N_("/_Force Distance/_Fit FD Curve..."),
                            GWY_ICON_GRAPH_FD,
                            GWY_MENU_FLAG_GRAPH_CURVE,
                            N_("Fit a force-distance curve"));

    return TRUE;
}

static GwyParamDef*
define_module_params(Submodule *specs)
{
    if (specs->paramdef)
        return specs->paramdef;

    GwyParamDef *paramdef = specs->paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_graph_func_current());
    gwy_param_def_add_resource(paramdef, PARAM_FITFUNC, "fitfunc", _("_Function"),
                               specs->inventory, specs->default_function);
    gwy_param_def_add_graph_curve(paramdef, PARAM_CURVE, "curve", NULL);
    gwy_param_def_add_boolean(paramdef, PARAM_PLOT_FULL, "plot_full", _("Plot full range"), FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_AUTO_ESTIMATE, "auto_estimate", _("Instant esti_mate"), TRUE);
    gwy_param_def_add_boolean(paramdef, PARAM_OUT_DIFF, "out_diff", _("Create a difference graph"), FALSE);
    gwy_param_def_add_report_type(paramdef, PARAM_REPORT_STYLE, "report_style", NULL,
                                  GWY_RESULTS_EXPORT_PARAMETERS, GWY_RESULTS_REPORT_COLON | GWY_RESULTS_REPORT_MACHINE);
    /* Range is GUI-only. */
    gwy_param_def_add_double(paramdef, PARAM_RANGE_FROM, NULL, _("Range"), -G_MAXDOUBLE, G_MAXDOUBLE, 0.0);
    gwy_param_def_add_double(paramdef, PARAM_RANGE_TO, NULL, NULL, -G_MAXDOUBLE, G_MAXDOUBLE, 0.0);

    return paramdef;
}

static void
module_main(GwyGraph *graph)
{
    /* Keep them static for the container paramdef. */
    static Submodule fit_specs = {
        .default_function = "Polynomial (order 1)",
    };
    static Submodule fdfit_specs = {
        .default_function = "DMT: spherical",
    };
    const gchar *name = gwy_graph_func_current();
    ModuleArgs args;
    gwy_clear1(args);

    if (gwy_strequal(name, "graph_fit")) {
        args.specs = &fit_specs;
        args.specs->dialog_title = _("Fit Graph Curve");
        args.specs->inventory = gwy_nlfit_presets();
    }
    else if (gwy_strequal(name, "graph_fdfit")) {
        args.specs = &fdfit_specs;
        args.specs->dialog_title = _("Fit FD Curve");
        args.specs->inventory = gwy_fd_curve_presets();
    }

    GwyFile *file;
    gwy_data_browser_get_current(GWY_APP_FILE, &file, 0);

    args.params = gwy_params_new_from_settings(define_module_params(args.specs));
    args.gmodel = gwy_graph_get_model(graph);
    args.xdata = g_array_new(FALSE, FALSE, sizeof(gdouble));
    args.ydata = g_array_new(FALSE, FALSE, sizeof(gdouble));

    run_gui(&args, file);

    gwy_params_save_to_settings(args.params);

    g_object_unref(args.params);
    g_array_free(args.xdata, TRUE);
    g_array_free(args.ydata, TRUE);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyFile *file)
{
    ModuleGUI gui;

    gwy_clear1(gui);
    gui.args = args;
    gui.file = file;

    gui.dialog = gwy_dialog_new(args->specs->dialog_title);
    GwyDialog *dialog = GWY_DIALOG(gui.dialog);
    gwy_add_button_to_dialog(GTK_DIALOG(dialog), GWY_STOCK_COPY, GWY_ICON_GTK_COPY, RESPONSE_COPY);
    gwy_add_button_to_dialog(GTK_DIALOG(dialog), GWY_STOCK_SAVE, GWY_ICON_GTK_SAVE, RESPONSE_SAVE);
    gwy_add_button_to_dialog(GTK_DIALOG(dialog), C_("verb", "_Fit"), GWY_ICON_GTK_EXECUTE, RESPONSE_REFINE);
    gwy_add_button_to_dialog(GTK_DIALOG(dialog), C_("verb", "_Estimate"), NULL, RESPONSE_ESTIMATE);
    /* We now have instant plotting always on. */
    /* gwy_add_button_to_dialog(GTK_DIALOG(dialog), C_("verb", "_Plot"), NULL, RESPONSE_PLOT); */
    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    GtkWidget *hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 0);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
    gwy_dialog_add_content(dialog, hbox, TRUE, TRUE, 0);

    gui.gmodel = gwy_graph_model_new_alike(args->gmodel);
    GtkWidget *graph = gwy_graph_new(gui.gmodel);
    gtk_widget_set_size_request(graph, 400, 300);
    gwy_graph_enable_user_input(GWY_GRAPH(graph), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), graph, TRUE, TRUE, 0);

    GtkWidget *vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 2);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, TRUE, 0);

    GwyParamTable *table = gui.table_above = gwy_param_table_new(args->params);
    gwy_param_table_append_graph_curve(table, PARAM_CURVE, args->gmodel);
    gwy_param_table_append_combo(table, PARAM_FITFUNC);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(vbox), gwy_param_table_widget(table), FALSE, TRUE, 0);

    GtkWidget *notebook = gtk_notebook_new();
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), create_param_tab(&gui), gtk_label_new(_("Parameters")));
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), create_correl_tab(&gui), gtk_label_new(_("Correlation Matrix")));
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), create_function_tab(&gui), gtk_label_new(_("Function")));
    gtk_box_pack_start(GTK_BOX(vbox), notebook, FALSE, TRUE, 0);

    table = gui.table_below = gwy_param_table_new(args->params);
    gwy_create_graph_xrange_with_params(table, PARAM_RANGE_FROM, PARAM_RANGE_TO, GWY_GRAPH(graph), args->gmodel);
    gwy_param_table_append_info(table, INFO_RSS, _("Mean square difference"));
    gwy_param_table_append_message(table, MESSAGE_ERROR, NULL);
    gwy_param_table_append_checkbox(table, PARAM_PLOT_FULL);
    gwy_param_table_append_checkbox(table, PARAM_OUT_DIFF);
    gwy_param_table_append_checkbox(table, PARAM_AUTO_ESTIMATE);
    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(vbox), gwy_param_table_widget(table), FALSE, TRUE, 0);

    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_IMMEDIATE, preview, &gui, NULL);
    g_signal_connect_swapped(gui.table_above, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.table_below, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(dialog, "response", G_CALLBACK(dialog_response), &gui);

    GwyDialogOutcome outcome = gwy_dialog_run(dialog);

    if (outcome == GWY_DIALOG_PROCEED) {
        gwy_graph_model_add_curve(args->gmodel, gwy_graph_model_get_curve(gui.gmodel, 1));
        if (gwy_params_get_boolean(args->params, PARAM_OUT_DIFF))
            create_difference_graph(&gui, file);
    }

    g_free(gui.correl);
    g_object_unref(gui.gmodel);
    if (gui.fitter)
        gwy_math_nlfit_free(gui.fitter);
    g_clear_object(&gui.results);

    return outcome;
}

static GtkWidget*
create_param_tab(ModuleGUI *gui)
{
    GtkWidget *vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 4);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);

    gui->fit_params = gwy_fit_table_new();
    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    gwy_fit_table_set_values_editable(table, TRUE);
    gwy_fit_table_set_has_checkboxes(table, TRUE);
    gwy_fit_table_set_header(table, _("Fix"), 0, 1);
    gwy_fit_table_set_header(table, _("Parameter"), 1, 3);
    gwy_fit_table_set_header(table, _("Error"), 6, 2);
    gtk_box_pack_start(GTK_BOX(vbox), gui->fit_params, FALSE, TRUE, 0);
    g_signal_connect_swapped(gui->fit_params, "activated", G_CALLBACK(param_value_activate), gui);

    GtkWidget *hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 4);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    gui->revert = gtk_button_new_with_mnemonic(_("Revert to _Previous Values"));
    gtk_box_pack_start(GTK_BOX(hbox), gui->revert, TRUE, TRUE, 0);
    g_signal_connect_swapped(gui->revert, "clicked", G_CALLBACK(revert_parameters), gui);

    return vbox;
}

static GtkWidget*
create_correl_tab(ModuleGUI *gui)
{
    GtkWidget *scwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_container_set_border_width(GTK_CONTAINER(scwin), 4);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin), GTK_POLICY_AUTOMATIC, GTK_POLICY_NEVER);

    gui->correl_table = gwy_correl_table_new();
    gtk_container_set_border_width(GTK_CONTAINER(gui->correl_table), 4);
    gtk_container_add(GTK_CONTAINER(scwin), gui->correl_table);

    return scwin;
}

static GtkWidget*
create_function_tab(ModuleGUI *gui)
{
    GtkWidget *scwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_container_set_border_width(GTK_CONTAINER(scwin), 8);

    gui->function = gtk_label_new(NULL);
    gtk_label_set_xalign(GTK_LABEL(gui->function), 0.0);
    gtk_container_add(GTK_CONTAINER(scwin), gui->function);

    return scwin;
}

static void
fit_param_table_resize(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    guint nparams = gwy_nlfit_preset_get_nparams(args->fitfunc);
    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    gwy_fit_table_resize(table, nparams);

    GwyUnit *xunit, *yunit;
    g_object_get(args->gmodel, "unit-x", &xunit, "unit-y", &yunit, NULL);

    for (guint i = 0; i < nparams; i++) {
        gwy_fit_table_set_checked(table, i, FALSE);
        gwy_fit_table_set_name(table, i, gwy_nlfit_preset_get_param_name(args->fitfunc, i));
        GwyUnit *unit = gwy_nlfit_preset_get_param_units(args->fitfunc, i, xunit, yunit);
        gwy_fit_table_set_unit(table, i, unit);
        g_object_unref(unit);
    }
    g_object_unref(xunit);
    g_object_unref(yunit);

    gwy_fit_table_clear_errors(table);
}

static void
fit_correl_table_resize(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    guint nparams = gwy_nlfit_preset_get_nparams(args->fitfunc);
    gui->correl = g_renew(gdouble, gui->correl, nparams*(nparams + 1)/2);

    GwyCorrelTable *table = GWY_CORREL_TABLE(gui->correl_table);
    gwy_correl_table_resize(table, nparams);
    for (guint i = 0; i < nparams; i++) {
        const gchar *name = gwy_nlfit_preset_get_param_name(args->fitfunc, i);
        gwy_correl_table_set_name(table, i, name);
    }
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    ModuleArgs *args = gui->args;
    GwyParams *params = args->params;
    gboolean want_estimate = FALSE, want_replot = FALSE;
    gboolean auto_estimate = gwy_params_get_boolean(params, PARAM_AUTO_ESTIMATE);

    if (id < 0 || id == PARAM_FITFUNC) {
        args->fitfunc = GWY_NLFIT_PRESET(gwy_params_get_resource(params, PARAM_FITFUNC));
        gtk_label_set_markup(GTK_LABEL(gui->function), gwy_nlfit_preset_get_formula(args->fitfunc));

        fit_param_table_resize(gui);
        fit_correl_table_resize(gui);
        create_results(gui);
        want_replot = TRUE;
        want_estimate = want_estimate || auto_estimate;
        guess_parameters(gui);
    }

    if (id < 0 || id == PARAM_CURVE) {
        gwy_graph_model_remove_all_curves(gui->gmodel);
        gwy_graph_model_add_curve(gui->gmodel,
                                  gwy_graph_model_get_curve(args->gmodel,
                                                            gwy_params_get_int(params, PARAM_CURVE)));
        want_replot = TRUE;
        update_fit_state(gui, FIT_INITIALISED);
    }

    if (id < 0 || id == PARAM_AUTO_ESTIMATE) {
        if (auto_estimate && !(gui->state == FIT_FITTED || gui->state == FIT_ESTIMATED))
            want_estimate = TRUE;
    }

    if (id < 0 || id == PARAM_RANGE_FROM || id == PARAM_RANGE_TO) {
        /* TODO */
        want_replot = TRUE;
        want_estimate = want_estimate || auto_estimate;
    }

    if (id < 0 || id == PARAM_PLOT_FULL) {
        want_replot = TRUE;
        want_estimate = want_estimate || auto_estimate;
    }

    if (want_estimate) {
        estimate(gui);
    }
    else if (want_replot) {
        gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
    }
}

static void
param_value_activate(ModuleGUI *gui, guint i)
{
    gwy_fit_table_clear_error(GWY_FIT_TABLE(gui->fit_params), i);
    gwy_correl_table_clear(GWY_CORREL_TABLE(gui->correl_table));
    update_fit_state(gui, FIT_USER_MODIFIED);
    gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
revert_parameters(ModuleGUI *gui)
{
    gwy_fit_table_revert_values(GWY_FIT_TABLE(gui->fit_params));
    gwy_fit_table_clear_errors(GWY_FIT_TABLE(gui->fit_params));
    gwy_correl_table_clear(GWY_CORREL_TABLE(gui->correl_table));
    update_fit_state(gui, FIT_USER_MODIFIED);
    gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
update_rss_label(ModuleGUI *gui)
{
    GwyUnit *yunit;
    g_object_get(gui->args->gmodel, "unit-y", &yunit, NULL);
    GwyValueFormat *vf = gwy_unit_get_format(yunit, GWY_UNIT_FORMAT_VFMARKUP, gui->rss, NULL);
    gchar *s = g_strdup_printf("%.*f", vf->precision, gui->rss/vf->magnitude);
    gwy_param_table_info_set_valuestr(gui->table_below, INFO_RSS, s);
    g_free(s);
    gwy_param_table_set_unitstr(gui->table_below, INFO_RSS, vf->units);
    gwy_value_format_free(vf);
    g_object_unref(yunit);
}

static void
guess_parameters(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;

    guint nparams = gwy_nlfit_preset_get_nparams(args->fitfunc);
    gdouble param[nparams];

    guint n = extract_data(args, TRUE);
    gdouble *xd = &g_array_index(args->xdata, gdouble, 0);
    gdouble *yd = &g_array_index(args->ydata, gdouble, 0);

    gboolean ok = TRUE;
    gwy_nlfit_preset_guess(args->fitfunc, n, xd, yd, param, &ok);
    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    for (guint i = 0; i < nparams; i++)
        gwy_fit_table_set_value(table, i, param[i]);

    update_fit_state(gui, FIT_INITIALISED);
}

static GwyGraphCurveModel*
ensure_model_curve(GwyGraphModel *gmodel, GwyGraphModel *parent_gmodel, gint cindex,
                   const gchar *description)
{
    GwyGraphCurveModel *gcmodel;
    if (gwy_graph_model_get_n_curves(gmodel) < cindex+1) {
        const GwyRGBA *color = gwy_graph_get_preset_color(gwy_graph_model_get_n_curves(parent_gmodel));
        gcmodel = gwy_graph_curve_model_new();
        g_object_set(gcmodel,
                     "mode", GWY_GRAPH_CURVE_LINE,
                     "color", color,
                     "line-style", GWY_GRAPH_LINE_DASH_DENSE,
                     "line-width", 2,
                     NULL);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
    }
    else
        gcmodel = gwy_graph_model_get_curve(gmodel, cindex);

    g_object_set(gcmodel, "description", description, NULL);
    return gcmodel;
}

static void
plot_curve(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;

    guint nparams = gwy_nlfit_preset_get_nparams(args->fitfunc);
    gdouble param[nparams];
    gwy_fit_table_gather(GWY_FIT_TABLE(gui->fit_params), param, NULL, NULL);

    guint n = extract_data(args, TRUE);
    gdouble *xd = &g_array_index(args->xdata, gdouble, 0);
    gdouble *yd = &g_array_index(args->ydata, gdouble, 0);
    gboolean ok = TRUE;
    gdouble rss = 0.0;
    for (guint i = 0; i < n; i++) {
        gdouble v = gwy_nlfit_preset_get_value(args->fitfunc, xd[i], param, &ok);
        rss += (yd[i] - v)*(yd[i] - v);
        yd[i] = v;
    }
    gui->rss = sqrt(rss/n);

    const gchar *desc = _("Model");
    if (gui->state == FIT_FITTED || gui->state == FIT_FIT_FAILED)
        desc = _("Fit");
    else if (gui->state == FIT_ESTIMATED || gui->state == FIT_ESTIMATE_FAILED)
        desc = _("Estimate");

    GwyGraphCurveModel *gcmodel = ensure_model_curve(gui->gmodel, args->gmodel, 1, desc);
    gwy_graph_curve_model_set_data(gcmodel, xd, yd, n);

    update_rss_label(gui);
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;

    /* TODO: Plot calculated function. That's what we do here. */
    plot_curve(gui);
}

static void
dialog_response(ModuleGUI *gui, gint response)
{
    if (response == RESPONSE_COPY || response == RESPONSE_SAVE) {
        GwyResultsExportStyle report_style = gwy_params_get_report_type(gui->args->params, PARAM_REPORT_STYLE);
        gchar *report = gwy_results_create_report(gui->results, report_style);

        if (response == RESPONSE_SAVE)
            gwy_save_auxiliary_data(_("Save Fit Report"), GTK_WINDOW(gui->dialog), report, -1, FALSE);
        else {
            GdkDisplay *display = gtk_widget_get_display(gui->dialog);
            GtkClipboard *clipboard = gtk_clipboard_get_for_display(display, GDK_SELECTION_CLIPBOARD);
            gtk_clipboard_set_text(clipboard, report, -1);
        }
        g_free(report);
    }
    else if (response == RESPONSE_ESTIMATE) {
        estimate(gui);
    }
    else if (response == RESPONSE_REFINE) {
        execute(gui);
    }
}

static void
execute(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    guint nparams = gwy_nlfit_preset_get_nparams(args->fitfunc);
    gdouble param[nparams], error[nparams];
    gboolean fixed[nparams];
    guint nfree = nparams - gwy_fit_table_gather(GWY_FIT_TABLE(gui->fit_params), param, NULL, fixed);

    if (!nfree)
        return;

    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    gwy_fit_table_clear_errors(table);

    GwyCorrelTable *ctable = GWY_CORREL_TABLE(gui->correl_table);
    gwy_correl_table_clear(ctable);

    guint npoints;
    npoints = extract_data(args, FALSE);
    if (npoints <= nfree) {
        gwy_param_table_set_label(gui->table_below, MESSAGE_ERROR, _("Not enough data points."));
        return;
    }
    gwy_param_table_set_label(gui->table_below, MESSAGE_ERROR, NULL);

    if (gui->fitter)
        gwy_math_nlfit_free(gui->fitter);

    const gdouble *xd = &g_array_index(args->xdata, gdouble, 0);
    const gdouble *yd = &g_array_index(args->ydata, gdouble, 0);
    GwyNLFitter *fitter = gwy_nlfit_preset_fit(args->fitfunc, NULL, npoints, xd, yd, param, error, fixed);
    gui->fitter = fitter;
    gboolean is_fitted = gwy_math_nlfit_succeeded(fitter);

    if (is_fitted) {
        for (guint i = 0; i < nparams; i++) {
            gwy_fit_table_set_value(table, i, param[i]);
            gwy_fit_table_set_error(table, i, fixed[i] ? 0.0 : gwy_math_nlfit_get_sigma(fitter, i));
            gwy_correl_table_set_fixed(ctable, i, fixed[i]);
        }

        for (guint i = 0; i < nparams; i++) {
            for (guint j = 0; j <= i; j++) {
                gdouble c = gwy_math_nlfit_get_correlations(fitter, i, j);
                gwy_correl_table_set_correl(ctable, i, j, c);
                SLi(gui->correl, i, j) = c;
            }
        }
    }

    plot_curve(gui);
    update_fit_state(gui, is_fitted ? FIT_FITTED : FIT_FIT_FAILED);
    if (is_fitted)
        fill_results(gui);
}

static void
update_fit_state(ModuleGUI *gui, FitState state)
{
    if (gui->state == state)
        return;

    gui->state = state;
    gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), RESPONSE_SAVE, state == FIT_FITTED);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), RESPONSE_COPY, state == FIT_FITTED);

    if (state != FIT_FITTED) {
        gwy_fit_table_clear_errors(GWY_FIT_TABLE(gui->fit_params));
        gwy_correl_table_clear(GWY_CORREL_TABLE(gui->correl_table));
        gwy_param_table_set_label(gui->table_below, MESSAGE_ERROR, _("Fit failed."));
    }
}

static void
estimate(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    GwyNLFitPreset *fitfunc = args->fitfunc;
    guint nparams = gwy_nlfit_preset_get_nparams(fitfunc);
    gdouble param[nparams];
    gboolean fixed[nparams];
    gwy_fit_table_gather(GWY_FIT_TABLE(gui->fit_params), param, NULL, fixed);

    guint npoints;
    if (!(npoints = extract_data(args, FALSE)))
        return;

    gboolean ok;
    const gdouble *xd = &g_array_index(args->xdata, gdouble, 0);
    const gdouble *yd = &g_array_index(args->ydata, gdouble, 0);
    gwy_nlfit_preset_guess(fitfunc, npoints, xd, yd, param, &ok);

    GwyFitTable *table = GWY_FIT_TABLE(gui->fit_params);
    for (guint i = 0; i < nparams; i++) {
        gwy_debug("[%u] %g", i, param[i]);
        if (!fixed[i])
            gwy_fit_table_set_value(table, i, param[i]);
    }
    gwy_fit_table_clear_errors(table);
    /* TODO correlaton, fit state (together?) */

    update_fit_state(gui, ok ? FIT_ESTIMATED : FIT_ESTIMATE_FAILED);
    plot_curve(gui);
}

/* Extract relevant part of data. */
static gint
extract_data(ModuleArgs *args, gboolean for_plotting)
{
    GwyParams *params = args->params;
    gdouble from = gwy_params_get_double(params, PARAM_RANGE_FROM);
    gdouble to = gwy_params_get_double(params, PARAM_RANGE_TO);
    if (for_plotting && gwy_params_get_boolean(params, PARAM_PLOT_FULL)) {
        from = -G_MAXDOUBLE;
        to = G_MAXDOUBLE;
    }

    GwyGraphCurveModel *cmodel = gwy_graph_model_get_curve(args->gmodel, gwy_params_get_int(params, PARAM_CURVE));
    const gdouble *xs = gwy_graph_curve_model_get_xdata(cmodel);
    const gdouble *ys = gwy_graph_curve_model_get_ydata(cmodel);
    guint ns = gwy_graph_curve_model_get_ndata(cmodel);

    g_array_set_size(args->xdata, ns);
    g_array_set_size(args->ydata, ns);
    gdouble *xd = &g_array_index(args->xdata, gdouble, 0);
    gdouble *yd = &g_array_index(args->ydata, gdouble, 0);

    /* FIXME: Unhardcode and fix to actually check the support interval */
    gboolean skip_zero = FALSE;
    const gchar *func_name = gwy_resource_get_name(GWY_RESOURCE(args->fitfunc));
    if (g_str_has_suffix(func_name, "(PSDF)")) {
        skip_zero = !for_plotting;
    }
    else if (gwy_strequal(func_name, "Power")) {
        skip_zero = TRUE;
        from = 0.0;
    }

    guint n = 0;
    for (guint i = 0; i < ns; i++) {
        if (xs[i] >= from && xs[i] <= to && !(skip_zero && xs[i] == 0.0)) {
            xd[n] = xs[i];
            yd[n] = ys[i];
            n++;
        }
    }
    if (!n)
        return 0;

    if (n < ns) {
        g_array_set_size(args->xdata, n);
        g_array_set_size(args->ydata, n);
    }

    return n;
}

static void
create_difference_graph(ModuleGUI *gui, GwyFile *file)
{
    GwyGraphModel *gmodel = gwy_graph_model_new_alike(gui->gmodel);
    GwyGraphCurveModel *cmodel_data = gwy_graph_model_get_curve(gui->gmodel, 0);
    GwyGraphCurveModel *cmodel_fit = gwy_graph_model_get_curve(gui->gmodel, 1);
    GwyGraphCurveModel *cmodel = gwy_graph_curve_model_new_alike(cmodel_data);

    guint n = gwy_graph_curve_model_get_ndata(cmodel_fit);
    const gdouble *xs = gwy_graph_curve_model_get_xdata(cmodel_fit);
    const gdouble *ys = gwy_graph_curve_model_get_ydata(cmodel_data);
    const gdouble *fs = gwy_graph_curve_model_get_ydata(cmodel_fit);
    gdouble *ds = g_new(gdouble, n);
    for (guint i = 0; i < n; i++)
        ds[i] = ys[i] - fs[i];

    gwy_graph_curve_model_set_data(cmodel, xs, ds, n);
    g_free(ds);
    gwy_graph_model_add_curve(gmodel, cmodel);
    g_object_unref(cmodel);
    gint newid = gwy_file_add_graph(file, gmodel);
    gwy_file_set_visible(file, GWY_FILE_GRAPH, newid, TRUE);
    g_object_unref(gmodel);
}

static void
create_results(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    GwyNLFitPreset *fitfunc = args->fitfunc;

    g_clear_object(&gui->results);
    GwyResults *results = gui->results = gwy_results_new();
    gwy_results_add_header(results, N_("Fit Results"));
    gwy_results_add_value_str(results, "file", N_("File"));
    gwy_results_add_value_str(results, "channel", N_("Curve"));
    gwy_results_add_format(results, "npts", N_("Number of points"), TRUE,
    /* TRANSLATORS: %{n}i and %{total}i are ids, do NOT translate them. */
                           N_("%{n}i of %{ntotal}i"), NULL);
    gwy_results_add_value_str(results, "func", N_("Fitted function"));
    gwy_results_add_value_z(results, "rss", N_("Mean square difference"));

    gwy_results_add_separator(results);
    gwy_results_add_header(results, N_("Parameters"));
    guint nparams = gwy_nlfit_preset_get_nparams(fitfunc);
    const gchar *names[nparams];
    for (guint i = 0; i < nparams; i++) {
        const gchar *name = names[i] = gwy_nlfit_preset_get_param_name(fitfunc, i);
        gint power_x = gwy_nlfit_preset_get_param_power_x(fitfunc, i);
        gint power_z = gwy_nlfit_preset_get_param_power_y(fitfunc, i);
        gwy_results_add_value(results, name, "",
                              "symbol", name, "is-fitting-param", TRUE,
                              "power-x", power_x, "power-z", power_z,
                              NULL);
    }

    gwy_results_add_separator(results);
    gwy_results_add_correlation_matrixv(results, "correl", N_("Correlation Matrix"), nparams, names);
}

static void
fill_results(ModuleGUI *gui)
{
    ModuleArgs *args = gui->args;
    GwyNLFitPreset *fitfunc = args->fitfunc;
    guint nparams = gwy_nlfit_preset_get_nparams(args->fitfunc);
    gdouble param[nparams], error[nparams];
    gboolean fixed[nparams];
    gwy_fit_table_gather(GWY_FIT_TABLE(gui->fit_params), param, error, fixed);

    GwyResults *results = gui->results;
    GwyGraphModel *gmodel = gui->gmodel;
    GwyUnit *xunit, *yunit;
    g_object_get(gmodel,
                 "unit-x", &xunit,
                 "unit-y", &yunit,
                 NULL);
    gwy_results_set_unit(results, "x", xunit);
    gwy_results_set_unit(results, "y", yunit);
    g_object_unref(xunit);
    g_object_unref(yunit);

    GwyGraphCurveModel *gcmodel_data = gwy_graph_model_get_curve(gmodel, 0);
    GwyGraphCurveModel *gcmodel_fit = gwy_graph_model_get_curve(gmodel, 1);
    guint ntotal = gwy_graph_curve_model_get_ndata(gcmodel_data);
    guint n = gwy_graph_curve_model_get_ndata(gcmodel_fit);
    gchar *desc;
    g_object_get(gcmodel_data, "description", &desc, NULL);

    gwy_results_fill_filename(results, "file", gui->file);
    gwy_results_fill_values(results,
                            "func", gwy_resource_get_name(GWY_RESOURCE(fitfunc)),
                            "channel", desc,
                            "rss", sqrt(gwy_math_nlfit_get_dispersion(gui->fitter)),
                            NULL);
    g_free(desc);
    gwy_results_fill_format(results, "npts",
                            "n", n,
                            "ntotal", ntotal,
                            NULL);

    for (guint i = 0; i < nparams; i++) {
        const gchar *name = gwy_nlfit_preset_get_param_name(fitfunc, i);

        if (fixed[i])
            gwy_results_fill_values(results, name, param[i], NULL);
        else
            gwy_results_fill_values_with_errors(results, name, param[i], error[i], NULL);
    }

    gwy_results_fill_correlation_matrix(results, "correl", fixed, gui->correl);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
