/*
 *  $Id: grain_number.c 29559 2026-03-02 18:38:33Z yeti-dn $
 *  Copyright (C) 2026 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <gwy.h>
#include "preview.h"

#define RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

enum {
    PARAM_NUMBERING,

    LABEL_CURRENT_STATE,
};

typedef enum {
    RENUMBER_CONTIGUOUS = 0,
    RENUMBER_CONTIGUOUS_PERIODIC = 1,
    RENUMBER_FLATTEN = 2,
    RENUMBER_COMPACTIFY = 3,
    RENUMBER_REMOVE_NEGATIVE = 4,
    RENUMBER_SPLIT_NONCONTIGUOUS = 5,
} RenumberType;

typedef enum {
    MASK_IS_EMPTY      = 1u << 0,
    MASK_IS_FLAT       = 1u << 1,
    MASK_HAS_NEGATIVE  = 1u << 2,
    MASK_HAS_TOUCHING  = 1u << 3,
    MASK_HAS_SCATTERED = 1u << 4,
    MASK_HAS_GAPS      = 1u << 5,
} MaskFlags;

typedef struct {
    GwyParams *params;
    GwyNield *mask;
    GwyNield *result;
} ModuleArgs;

static gboolean         module_register     (void);
static GwyParamDef*     define_module_params(void);
static void             module_main         (GwyFile *data,
                                             GwyRunModeFlags mode);
static GwyDialogOutcome run_gui             (ModuleArgs *args);
static gboolean         execute             (ModuleArgs *args);
static MaskFlags        describe_mask       (GwyNield *mask,
                                             GwyNield *tmp);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Renumbering of masked areas in one of standard manners."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2026",
};

GWY_MODULE_QUERY2(module_info, grain_number)

static gboolean
module_register(void)
{
    gwy_process_func_register("grain_number",
                              module_main,
                              N_("/_Grains/_Renumber..."),
                              NULL,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE | GWY_MENU_FLAG_IMAGE_MASK,
                              N_("Renumber marked regions"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;
    static const GwyEnum methods[] = {
        { N_("Number contiguous"),              RENUMBER_CONTIGUOUS,          },
        { N_("Number peroidically contiguous"), RENUMBER_CONTIGUOUS_PERIODIC, },
        { N_("Flatten to 0 and 1"),             RENUMBER_FLATTEN,             },
        { N_("Compactify numbering"),           RENUMBER_COMPACTIFY,          },
        { N_("Remove negative"),                RENUMBER_REMOVE_NEGATIVE,     },
    };

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_process_func_current());
    gwy_param_def_add_gwyenum(paramdef, PARAM_NUMBERING, "numbering", _("Operation"),
                              methods, G_N_ELEMENTS(methods), RENUMBER_CONTIGUOUS);
    return paramdef;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    ModuleArgs args;
    GQuark quark;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_MASK_FIELD_KEY, &quark,
                                 GWY_APP_MASK_FIELD, &args.mask,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(args.mask && quark);

    args.result = gwy_nield_new_alike(args.mask);
    args.params = gwy_params_new_from_settings(define_module_params());
    if (mode != GWY_RUN_IMMEDIATE) {
        GwyDialogOutcome outcome = run_gui(&args);
        gwy_params_save_to_settings(args.params);
        if (outcome != GWY_DIALOG_PROCEED)
            goto end;
    }
    if (execute(&args)) {
        gwy_app_undo_qcheckpointv(GWY_DICT(data), 1, &quark);
        gwy_nield_copy_data(args.result, args.mask);
        gwy_log_add(data, GWY_FILE_IMAGE, id, id);
        gwy_nield_data_changed(args.mask);
    }

end:
    g_object_unref(args.params);
    g_clear_object(&args.result);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args)
{
    static const GwyEnum flagdesc[] = {
        { N_("empty"),                 MASK_IS_EMPTY,      },
        { N_("flat"),                  MASK_IS_FLAT,       },
        { N_("negative numbers"),      MASK_HAS_NEGATIVE,  },
        { N_("touching grains"),       MASK_HAS_TOUCHING,  },
        { N_("non-contiguous grains"), MASK_HAS_SCATTERED, },
        { N_("numbering gaps"),        MASK_HAS_GAPS,      },
    };

    GwyDialog *dialog = GWY_DIALOG(gwy_dialog_new(_("Number Grans")));
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    args->result = gwy_nield_new_alike(args->mask);
    MaskFlags flags = describe_mask(args->mask, args->result);

    GwyParamTable *table = gwy_param_table_new(args->params);
    gwy_param_table_append_radio(table, PARAM_NUMBERING);
    gchar *desc = gwy_flags_to_string(flags, flagdesc, G_N_ELEMENTS(flagdesc), ", ");
    gchar *state = g_strdup_printf("Current numbering: %s", strlen(desc) ? desc : _("unremarkable"));
    g_free(desc);
    gwy_param_table_append_separator(table);
    gwy_param_table_append_message(table, LABEL_CURRENT_STATE, state);
    g_free(state);
    gwy_dialog_add_content(dialog, gwy_param_table_widget(table), FALSE, FALSE, 0);
    gwy_dialog_add_param_table(dialog, table);

    return gwy_dialog_run(dialog);
}

static gboolean
execute(ModuleArgs *args)
{
    RenumberType operation = gwy_params_get_enum(args->params, PARAM_NUMBERING);
    GwyNield *result = args->result, *mask = args->mask;

    if (!result)
        result = args->result = gwy_nield_copy(mask);
    else
        gwy_nield_copy_data(mask, result);

    if (operation == RENUMBER_CONTIGUOUS)
        gwy_nield_number_contiguous(result);
    else if (operation == RENUMBER_CONTIGUOUS_PERIODIC)
        gwy_nield_number_contiguous_periodic(result);
    else if (operation == RENUMBER_FLATTEN)
        gwy_nield_flatten(result);
    else if (operation == RENUMBER_COMPACTIFY)
        gwy_nield_compactify(result);
    else if (operation == RENUMBER_REMOVE_NEGATIVE) {
        // Pass
    }
    else if (operation == RENUMBER_SPLIT_NONCONTIGUOUS)
        gwy_nield_split_noncontiguous(result);
    else {
        g_assert_not_reached();
    }

    gboolean changed = FALSE;
    gint xres = gwy_nield_get_xres(result), yres = gwy_nield_get_yres(result);
    gint *newgrains = gwy_nield_get_data(result);
    const gint *grains = gwy_nield_get_data_const(mask);
    for (gint i = 0; i < xres*yres; i++) {
        if (operation == RENUMBER_REMOVE_NEGATIVE) {
            if (newgrains[i] < 0) {
                newgrains[i] = 0;
                changed = TRUE;
            }
        }
        else if (newgrains[i] != grains[i]) {
            changed = TRUE;
            break;
        }
    }

    return changed;
}

static MaskFlags
describe_mask(GwyNield *mask, GwyNield *tmp)
{
    MaskFlags flags = 0;
    gint max = gwy_nield_max(mask);
    gint xres = gwy_nield_get_xres(mask), yres = gwy_nield_get_yres(mask);
    const gint *grains = gwy_nield_get_data_const(mask);

    if (!max)
        flags |= MASK_IS_EMPTY;
    else {
        if (max == 1)
            flags |= MASK_IS_FLAT;

        // NB: This is the number of distinct grains, not maximum grain number! (As needed below.)
        gint ngrains;
        gint *numbers = gwy_nield_grain_numbers(mask, &ngrains);
        for (gint i = 0; i < ngrains-1; i++) {
            if (numbers[i+1] != numbers[i]+1) {
                flags |= MASK_HAS_GAPS;
                break;
            }
        }
        g_free(numbers);

        for (gint i = 0; i < yres-1 && !(flags & MASK_HAS_TOUCHING); i++) {
            const gint *mrow = grains + i*xres, *nextrow = mrow + xres;
            for (gint j = 0; j < xres; j++) {
                if (mrow[j] > 0 && nextrow[j] > 0 && mrow[j] != nextrow[j]) {
                    flags |= MASK_HAS_TOUCHING;
                    break;
                }
            }
        }
        for (gint i = 0; i < yres && !(flags & MASK_HAS_TOUCHING); i++) {
            const gint *mrow = grains + i*xres;
            for (gint j = 0; j < xres-1; j++) {
                if (mrow[j] > 0 && mrow[j+1] > 0 && mrow[j] != mrow[j+1]) {
                    flags |= MASK_HAS_TOUCHING;
                    break;
                }
            }
        }

        gwy_nield_copy_data(mask, tmp);
        gint split_ngrains = gwy_nield_split_noncontiguous(tmp);
        if (split_ngrains != ngrains+1)
            flags |= MASK_HAS_SCATTERED;
    }

    for (gint i = 0; i < xres*yres; i++) {
        if (grains[i] < 0) {
            flags |= MASK_HAS_NEGATIVE;
            break;
        }
    }

    return flags;
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
