/*
 *  $Id: mark_disconn.c 29477 2026-02-14 13:29:30Z yeti-dn $
 *  Copyright (C) 2015-2026 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <string.h>
#include <gwy.h>
#include "preview.h"

#define RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

enum {
    PARAM_TYPE,
    PARAM_RADIUS,
    PARAM_THRESHOLD,
    PARAM_COMBINE_TYPE,
    PARAM_COMBINE,
    PARAM_MASK_COLOR,
};

typedef struct {
    GwyParams *params;
    GwyField *field;
    GwyNield *mask;
    GwyNield *result;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
} ModuleGUI;

static gboolean         module_register     (void);
static GwyParamDef*     define_module_params(void);
static void             module_main         (GwyFile *data,
                                             GwyRunModeFlags mode);
static gboolean         execute             (ModuleArgs *args,
                                             GtkWindow *wait_window);
static GwyDialogOutcome run_gui             (ModuleArgs *args,
                                             GwyFile *data,
                                             gint id);
static void             param_changed       (ModuleGUI *gui,
                                             gint id);
static void             preview             (gpointer user_data);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Creates mask of values disconnected to the rest."),
    "Yeti <yeti@gwyddion.net>",
    "3.0",
    "David Nečas (Yeti)",
    "2015",
};

GWY_MODULE_QUERY2(module_info, mark_disconn)

static gboolean
module_register(void)
{
    gwy_process_func_register("mark_disconn",
                              module_main,
                              N_("/_Correct Data/Mask of _Disconnected..."),
                              GWY_ICON_DISCONNECTED,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE,
                              N_("Mark data disconnected from other values"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum feature_types[] = {
        { N_("Positive"), GWY_SIGN_POSITIVE, },
        { N_("Negative"), GWY_SIGN_NEGATIVE, },
        { N_("Both"),     GWY_SIGN_BOTH,     },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_process_func_current());
    gwy_param_def_add_gwyenum(paramdef, PARAM_TYPE, "type", _("Defect type"),
                              feature_types, G_N_ELEMENTS(feature_types), GWY_SIGN_BOTH);
    gwy_param_def_add_double(paramdef, PARAM_THRESHOLD, "threshold", _("_Threshold"), 0.0, 1.0, 0.1);
    gwy_param_def_add_int(paramdef, PARAM_RADIUS, "radius", _("Defect _radius"), 1, 240, 5);
    gwy_param_def_add_enum(paramdef, PARAM_COMBINE_TYPE, "combine_type", NULL, GWY_TYPE_MERGE_TYPE, GWY_MERGE_UNION);
    gwy_param_def_add_boolean(paramdef, PARAM_COMBINE, "combine", NULL, FALSE);
    gwy_param_def_add_mask_color(paramdef, PARAM_MASK_COLOR, NULL, NULL);
    return paramdef;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    ModuleArgs args;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    GQuark mquark;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD, &args.field,
                                 GWY_APP_MASK_FIELD_KEY, &mquark,
                                 GWY_APP_MASK_FIELD, &args.mask,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(args.field && mquark);

    args.result = gwy_field_new_nield_alike(args.field);
    args.params = gwy_params_new_from_settings(define_module_params());

    if (mode == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, id);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (outcome != GWY_DIALOG_HAVE_RESULT) {
        if (!execute(&args, gwy_data_browser_get_window_for_data(data, GWY_FILE_IMAGE, id)))
            goto end;
    }

    gwy_app_undo_qcheckpointv(GWY_DICT(data), 1, &mquark);
    gint ngrains = gwy_nield_number_contiguous(args.result);
    gwy_file_set_image_mask(data, id, ngrains ? args.result : NULL);
    gwy_log_add(data, GWY_FILE_IMAGE, id, id);

end:
    g_object_unref(args.result);
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyFile *data, gint id)
{
    GtkWidget *hbox, *dataview;
    GwyDialog *dialog;
    GwyParamTable *table;
    ModuleGUI gui;

    gwy_clear1(gui);
    gui.args = args;

    gui.dialog = gwy_dialog_new(_("Mark Disconnected"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_UPDATE, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    dataview = gwy_create_preview(args->field, args->result, PREVIEW_SIZE);
    gwy_setup_data_view(GWY_DATA_VIEW(dataview), data, GWY_FILE_IMAGE, id,
                        GWY_FILE_ITEM_PALETTE | GWY_FILE_ITEM_RANGE | GWY_FILE_ITEM_REAL_SQUARE);
    hbox = gwy_create_dialog_preview_hbox(GTK_DIALOG(dialog), GWY_DATA_VIEW(dataview), FALSE);

    table = gui.table = gwy_param_table_new(args->params);

    gwy_param_table_append_radio(table, PARAM_TYPE);

    gwy_param_table_append_separator(table);
    gwy_param_table_append_slider(table, PARAM_RADIUS);
    gwy_param_table_set_unitstr(table, PARAM_RADIUS, _("px"));
    gwy_param_table_append_slider(table, PARAM_THRESHOLD);
    gwy_param_table_slider_set_steps(table, PARAM_THRESHOLD, 0.001, 0.1);
    gwy_param_table_slider_set_digits(table, PARAM_THRESHOLD, 4);

    gwy_param_table_append_header(table, -1, _("Options"));
    gwy_param_table_append_image_mask_color(table, PARAM_MASK_COLOR, GWY_DATA_VIEW(dataview), data, id);
    if (args->mask) {
        gwy_param_table_append_radio_buttons(table, PARAM_COMBINE_TYPE, NULL);
        gwy_param_table_add_enabler(table, PARAM_COMBINE, PARAM_COMBINE_TYPE);
    }

    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), TRUE, TRUE, 0);
    gwy_dialog_add_param_table(dialog, table);

    g_signal_connect_swapped(table, "param-changed", G_CALLBACK(param_changed), &gui);
    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_UPON_REQUEST, preview, &gui, NULL);

    return gwy_dialog_run(dialog);
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    if (id != PARAM_MASK_COLOR)
        gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    ModuleArgs *args = gui->args;

    if (execute(args, GTK_WINDOW(gui->dialog))) {
        gwy_nield_data_changed(args->result);
        gwy_dialog_have_result(GWY_DIALOG(gui->dialog));
    }
    else
        gwy_nield_clear(args->result);
}

/* Unmask pixels with image values that do not belong to the largest contiguous block of values in the height
 * distribution. */
static guint
unmark_disconnected_values(GwyField *field, GwyNield *inclmask,
                           guint n, gdouble threshold)
{
    guint xres = gwy_field_get_xres(field), yres = gwy_field_get_yres(field);
    guint lineres = (guint)floor(2.5*cbrt(xres*yres - n) + 0.5);
    GwyLine *dline = gwy_line_new(lineres, lineres, FALSE);

    gwy_field_area_height_dist(field, inclmask, GWY_MASK_INCLUDE, dline, 0, 0, xres, yres, lineres);
    gdouble rho_zero = gwy_line_max(dline)/sqrt(xres*yres - n)*threshold;
    const gdouble *d = gwy_line_get_data_const(dline);
    lineres = gwy_line_get_res(dline);

    guint blockstart = 0, bestblockstart = 0, bestblocklen = 0;
    gdouble blocksum = 0.0, bestblocksum = 0.0;
    for (gint i = 0; i <= lineres; i++) {
        if (i == lineres || (i && d[i] + d[i-1] < rho_zero)) {
            if (blocksum > bestblocksum) {
                bestblocksum = blocksum;
                bestblockstart = blockstart;
                bestblocklen = i - blockstart;
            }
            blockstart = i+1;
            blocksum = 0.0;
        }
        else
            blocksum += d[i];
    }

    if (bestblocklen == lineres) {
        g_object_unref(dline);
        return 0;
    }

    gdouble real = gwy_line_get_real(dline);
    gdouble off = gwy_line_get_offset(dline);
    gdouble min = off + real/lineres*bestblockstart;
    gdouble max = off + real/lineres*(bestblockstart + bestblocklen + 1);
    gint *m = gwy_nield_get_data(inclmask);
    d = gwy_field_get_data_const(field);
    gint nn = 0;
    for (gint i = 0; i < xres*yres; i++) {
        if (m[i] > 0 && (d[i] < min || d[i] > max)) {
            m[i] = 0;
            nn++;
        }
    }

    g_object_unref(dline);
    return nn;
}

static gboolean
execute(ModuleArgs *args, GtkWindow *wait_window)
{
    GwyParams *params = args->params;
    gboolean combine = gwy_params_get_boolean(params, PARAM_COMBINE);
    GwyMergeType combine_type = gwy_params_get_enum(params, PARAM_COMBINE_TYPE);
    gint size = 2*gwy_params_get_int(params, PARAM_RADIUS) + 1;
    gdouble threshold = gwy_params_get_double(params, PARAM_THRESHOLD);
    GwySignFlags type = gwy_params_get_enum(params, PARAM_TYPE);
    GwyField *field = args->field;
    GwyNield *mask = args->mask, *result = args->result, *maskbuf = NULL, *kernel = NULL;
    GwyField *difffield = NULL, *buffer = NULL;
    gint xres = gwy_field_get_xres(field);
    gint yres = gwy_field_get_yres(field);
    guint n, nn;
    gboolean ok = FALSE;

    gwy_app_wait_start(wait_window, _("Initializing..."));
    /* Remove the positive, negative (or both) defects using a filter.  This produces a defect-free field. */
    buffer = gwy_field_copy(field);
    if (!gwy_app_wait_set_message(_("Filtering...")))
        goto finish;

    kernel = gwy_nield_new(size, size);
    n = gwy_nield_elliptic_fill(kernel);
    if (type == GWY_SIGN_POSITIVE || type == GWY_SIGN_NEGATIVE) {
        GwyMinMaxFilterType filtertpe = (type == GWY_SIGN_POSITIVE
                                         ? GWY_MIN_MAX_FILTER_OPENING
                                         : GWY_MIN_MAX_FILTER_CLOSING);
        gwy_field_area_filter_min_max(buffer, kernel, filtertpe, 0, 0, xres, yres);
    }
    else {
        if (!gwy_field_area_filter_kth_rank(buffer, kernel, 0, 0, xres, yres, n/2, gwy_app_wait_set_fraction))
            goto finish;
    }

    /* Then find look at the difference and mark any outliers in it because these must be defects. */
    difffield = gwy_field_new_alike(field, FALSE);
    gwy_field_subtract_fields(difffield, field, buffer);

    maskbuf = gwy_field_new_nield_alike(field);
    gwy_nield_fill(maskbuf, 1);

    if (!gwy_app_wait_set_message(_("Marking outliers...")))
        goto finish;

    n = 0;
    while ((nn = unmark_disconnected_values(difffield, maskbuf, n, 4.0*threshold)))
        n += nn;

    gwy_nield_invert(maskbuf);
    if (combine)
        gwy_nield_merge(maskbuf, mask, combine_type);
    gwy_nield_copy_data(maskbuf, result);
    ok = TRUE;

finish:
    gwy_app_wait_finish();
    g_clear_object(&kernel);
    g_clear_object(&buffer);
    g_clear_object(&maskbuf);
    g_clear_object(&difffield);
    return ok;
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
