/*
 *  $Id: mask_edt.c 29447 2026-02-07 12:14:37Z yeti-dn $
 *  Copyright (C) 2014-2026 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <stdlib.h>
#include <gwy.h>
#include "preview.h"

#define RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

enum {
    PARAM_DIST_TYPE,
    PARAM_OUTPUT,
    PARAM_FROM_BORDER,
    PARAM_UPDATE,
};

typedef enum {
    MASKEDT_INTERIOR = 0,
    MASKEDT_EXTERIOR = 1,
    MASKEDT_SIGNED   = 2,
} MaskEdtType;

typedef struct {
    GwyParams *params;
    GwyField *field;
    GwyNield *mask;
    GwyField *result;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
} ModuleGUI;

static gboolean         module_register(void);
static GwyParamDef*     define_params  (void);
static void             module_main    (GwyFile *data,
                                        GwyRunModeFlags mode);
static void             execute        (ModuleArgs *args);
static GwyDialogOutcome run_gui        (ModuleArgs *args,
                                        GwyFile *data,
                                        gint id);
static void             param_changed  (ModuleGUI *gui,
                                        gint id);
static void             preview        (gpointer user_data);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Performs simple and true Euclidean distance transforms of masks."),
    "Yeti <yeti@gwyddion.net>",
    "3.1",
    "David Nečas (Yeti)",
    "2014",
};

GWY_MODULE_QUERY2(module_info, mask_edt)

static gboolean
module_register(void)
{
    gwy_process_func_register("mask_edt",
                              module_main,
                              N_("/_Mask/Distanc_e Transform..."),
                              GWY_ICON_DISTANCE_TRANSFORM,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_MASK | GWY_MENU_FLAG_IMAGE,
                              N_("Distance transform of mask"));

    return TRUE;
}

static GwyParamDef*
define_params(void)
{
    static const GwyEnum outputs[] = {
        { N_("Interior"),  MASKEDT_INTERIOR },
        { N_("Exterior"),  MASKEDT_EXTERIOR },
        { N_("Two-sided"), MASKEDT_SIGNED   },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_process_func_current());
    gwy_param_def_add_enum(paramdef, PARAM_DIST_TYPE, "dist_type", _("_Distance type"),
                           GWY_TYPE_DISTANCE_TRANSFORM_TYPE, GWY_DISTANCE_TRANSFORM_EUCLIDEAN);
    gwy_param_def_add_gwyenum(paramdef, PARAM_OUTPUT, "mask_type", _("Output type"),
                              outputs, G_N_ELEMENTS(outputs), MASKEDT_INTERIOR);
    gwy_param_def_add_boolean(paramdef, PARAM_FROM_BORDER, "from_border", _("Shrink from _border"), TRUE);
    gwy_param_def_add_instant_updates(paramdef, PARAM_UPDATE, "update", NULL, TRUE);
    return paramdef;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    ModuleArgs args;
    gint id, newid;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_MASK_FIELD, &args.mask,
                                 GWY_APP_FIELD, &args.field,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(args.mask && args.field);

    args.result = gwy_field_new_alike(args.field, TRUE);
    args.params = gwy_params_new_from_settings(define_params());

    if (mode == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, id);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (outcome != GWY_DIALOG_HAVE_RESULT)
        execute(&args);

    newid = gwy_file_add_image(data, args.result);

    gwy_file_set_visible(data, GWY_FILE_IMAGE, newid, TRUE);
    gwy_file_sync_items(data, GWY_FILE_IMAGE, id,
                        data, GWY_FILE_IMAGE, newid,
                        GWY_FILE_ITEM_PALETTE | GWY_FILE_ITEM_MASK_COLOR | GWY_FILE_ITEM_REAL_SQUARE, FALSE);
    gwy_file_set_title(data, GWY_FILE_IMAGE, newid, _("Distance Transform"), TRUE);
    gwy_log_add(data, GWY_FILE_IMAGE, id, newid);

end:
    g_object_unref(args.result);
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyFile *data, gint id)
{
    ModuleGUI gui;
    GwyDialog *dialog;
    GwyParamTable *table;
    GtkWidget *hbox, *dataview;

    gui.args = args;

    gui.dialog = gwy_dialog_new(_("Distance Transform"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_UPDATE, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    dataview = gwy_create_preview(args->result, NULL, PREVIEW_SIZE);
    gwy_setup_data_view(GWY_DATA_VIEW(dataview), data, GWY_FILE_IMAGE, id,
                        GWY_FILE_ITEM_PALETTE | GWY_FILE_ITEM_REAL_SQUARE);
    hbox = gwy_create_dialog_preview_hbox(GTK_DIALOG(dialog), GWY_DATA_VIEW(dataview), FALSE);

    table = gui.table = gwy_param_table_new(args->params);
    gwy_param_table_append_combo(table, PARAM_DIST_TYPE);
    gwy_param_table_append_radio(table, PARAM_OUTPUT);
    gwy_param_table_append_separator(table);
    gwy_param_table_append_checkbox(table, PARAM_FROM_BORDER);
    gwy_param_table_append_checkbox(table, PARAM_UPDATE);

    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), TRUE, TRUE, 0);
    gwy_dialog_add_param_table(dialog, table);

    g_signal_connect_swapped(table, "param-changed", G_CALLBACK(param_changed), &gui);
    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_IMMEDIATE, preview, &gui, NULL);

    return gwy_dialog_run(dialog);
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    if (id != PARAM_UPDATE)
        gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;

    execute(gui->args);
    gwy_field_data_changed(gui->args->result);
}

static void
execute(ModuleArgs *args)
{
    GwyField *field = args->field, *result = args->result;
    GwyNield *mask = args->mask;
    GwyDistanceTransformType dtype = gwy_params_get_enum(args->params, PARAM_DIST_TYPE);
    gboolean from_border = gwy_params_get_boolean(args->params, PARAM_FROM_BORDER);
    MaskEdtType output = gwy_params_get_enum(args->params, PARAM_OUTPUT);

    if (output == MASKEDT_INTERIOR) {
        gwy_nield_distance_transform(mask, result, dtype, TRUE, from_border);
    }
    else if (output == MASKEDT_EXTERIOR || output == MASKEDT_SIGNED) {
        GwyNield *invmask = gwy_nield_copy(mask);
        gwy_nield_invert(invmask);
        gwy_nield_distance_transform(invmask, result, dtype, TRUE, from_border);
        g_object_unref(invmask);

        if (output == MASKEDT_SIGNED) {
            GwyField *tmp = gwy_field_copy(result);
            gwy_nield_distance_transform(mask, tmp, dtype, TRUE, from_border);
            gwy_field_subtract_fields(result, result, tmp);
            g_object_unref(tmp);
        }
    }

    gwy_field_multiply(result, sqrt(gwy_field_get_dx(field)*gwy_field_get_dy(field)));
    gwy_unit_assign(gwy_field_get_unit_z(result), gwy_field_get_unit_xy(field));
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
