/*
 *  $Id: maskcor.c 29477 2026-02-14 13:29:30Z yeti-dn $
 *  Copyright (C) 2004-2026 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <gwy.h>
#include "preview.h"

#define RUN_MODES GWY_RUN_INTERACTIVE

enum {
    PARAM_RESULT,
    PARAM_THRESHOLD,
    PARAM_REGCOEFF,
    PARAM_METHOD,
    PARAM_USE_MASK,
    PARAM_PLOT_MASK,
    PARAM_KERNEL,
};

typedef enum {
    MASKCOR_OBJECTS = 0,
    MASKCOR_MAXIMA  = 1,
    MASKCOR_SCORE   = 2,
} MaskcorResult;

typedef struct {
    GwyParams *params;
    GwyField *field;
    GwyField *score;
    GwyNield *mask;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
    GtkWidget *dataview;
} ModuleGUI;

static gboolean         module_register     (void);
static GwyParamDef*     define_module_params(void);
static void             module_main         (GwyFile *data,
                                             GwyRunModeFlags mode);
static void             execute             (ModuleArgs *args);
static GwyDialogOutcome run_gui             (ModuleArgs *args,
                                             GwyFile *data,
                                             gint id);
static void             param_changed       (ModuleGUI *gui,
                                             gint id);
static void             preview             (gpointer user_data);
static gboolean         kernel_filter       (GwyFile *data,
                                             gint id,
                                             gpointer user_data);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Searches for a detail in another image using correlation."),
    "Petr Klapetek <klapetek@gwyddion.net>, Yeti <yeti@gwyddion.net>",
    "4.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, maskcor)

static gboolean
module_register(void)
{
    gwy_process_func_register("maskcor",
                              module_main,
                              N_("/M_ultidata/Correlation _Search..."),
                              GWY_ICON_CORRELATION_MASK,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE,
                              N_("Search for a detail using correlation"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum methods[] = {
        { N_("Correlation, raw"),           GWY_CORR_SEARCH_COVARIANCE_RAW    },
        { N_("Correlation, leveled"),       GWY_CORR_SEARCH_COVARIANCE        },
        { N_("Correlation score"),          GWY_CORR_SEARCH_COVARIANCE_SCORE  },
        { N_("Height difference, raw"),     GWY_CORR_SEARCH_HEIGHT_DIFF_RAW   },
        { N_("Height difference, leveled"), GWY_CORR_SEARCH_HEIGHT_DIFF       },
        { N_("Height difference score"),    GWY_CORR_SEARCH_HEIGHT_DIFF_SCORE },
        { N_("Phase-only score"),           GWY_CORR_SEARCH_PHASE_ONLY_SCORE  },
    };
    static const GwyEnum results[] = {
        { N_("Objects marked"),     MASKCOR_OBJECTS, },
        { N_("Correlation maxima"), MASKCOR_MAXIMA,  },
        { N_("Correlation score"),  MASKCOR_SCORE,   },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_process_func_current());
    gwy_param_def_add_gwyenum(paramdef, PARAM_RESULT, "result", _("Output _type"),
                              results, G_N_ELEMENTS(results), MASKCOR_OBJECTS);
    gwy_param_def_add_double(paramdef, PARAM_THRESHOLD, "threshold", _("T_hreshold"), 0.0, 1.0, 0.95);
    gwy_param_def_add_double(paramdef, PARAM_REGCOEFF, "regcoeff", _("_Regularization parameter"), 0.0, 1.0, 0.001);
    gwy_param_def_add_gwyenum(paramdef, PARAM_METHOD, "method", _("Correlation _method"),
                              methods, G_N_ELEMENTS(methods), GWY_CORR_SEARCH_COVARIANCE_SCORE);
    gwy_param_def_add_boolean(paramdef, PARAM_USE_MASK, "use_mask", _("Use _mask"), TRUE);
    gwy_param_def_add_boolean(paramdef, PARAM_PLOT_MASK, "plot_mask", _("_Plot mask"), TRUE);
    gwy_param_def_add_image_id(paramdef, PARAM_KERNEL, "kernel", _("_Detail to search"));
    return paramdef;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    GwyDialogOutcome outcome;
    MaskcorResult output;
    ModuleArgs args;
    GQuark mquark;
    gint id, newid;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD, &args.field,
                                 GWY_APP_FIELD_ID, &id,
                                 GWY_APP_MASK_FIELD_KEY, &mquark,
                                 0);
    g_return_if_fail(args.field);

    args.mask = gwy_field_new_nield_alike(args.field);
    args.score = gwy_field_new_alike(args.field, TRUE);
    args.params = gwy_params_new_from_settings(define_module_params());

    outcome = run_gui(&args, data, id);
    gwy_params_save_to_settings(args.params);
    if (outcome == GWY_DIALOG_CANCEL)
        goto end;

    /* Only one of args.score and args.mask is filled with useful data, depending on the output type! */
    execute(&args);

    output = gwy_params_get_enum(args.params, PARAM_RESULT);
    if (output == MASKCOR_SCORE) {
        newid = gwy_file_add_image(data, args.score);
        gwy_file_set_visible(data, GWY_FILE_IMAGE, newid, TRUE);
        gwy_file_sync_items(data, GWY_FILE_IMAGE, id,
                            data, GWY_FILE_IMAGE, newid,
                            GWY_FILE_ITEM_PALETTE, FALSE);
        gwy_file_set_title(data, GWY_FILE_IMAGE, newid, _("Correlation score"), TRUE);
        gwy_log_add(data, GWY_FILE_IMAGE, id, newid);
    }
    else {
        gwy_app_undo_qcheckpointv(GWY_DICT(data), 1, &mquark);
        gwy_dict_set_object(GWY_DICT(data), mquark, args.mask);
        gwy_log_add(data, GWY_FILE_IMAGE, id, id);
    }

end:
    g_object_unref(args.params);
    g_object_unref(args.score);
    g_object_unref(args.mask);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyFile *data, gint id)
{
    GwyDialog *dialog;
    GwyParamTable *table;
    ModuleGUI gui;
    GtkWidget *hbox;

    gui.args = args;

    gui.dialog = gwy_dialog_new(_("Correlation Search"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_UPDATE, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    gui.dataview = gwy_create_preview(args->score, NULL, PREVIEW_SIZE);
    gwy_setup_data_view(GWY_DATA_VIEW(gui.dataview), data, GWY_FILE_IMAGE, id,
                        GWY_FILE_ITEM_PALETTE | GWY_FILE_ITEM_REAL_SQUARE);
    hbox = gwy_create_dialog_preview_hbox(GTK_DIALOG(dialog), GWY_DATA_VIEW(gui.dataview), FALSE);

    table = gui.table = gwy_param_table_new(args->params);
    gwy_param_table_append_image_id(table, PARAM_KERNEL);
    gwy_param_table_data_id_set_filter(table, PARAM_KERNEL, kernel_filter, args->field, NULL);

    gwy_param_table_append_header(table, -1, _("Correlation Search"));
    gwy_param_table_append_checkbox(table, PARAM_USE_MASK);
    gwy_param_table_append_combo(table, PARAM_METHOD);
    gwy_param_table_append_slider(table, PARAM_THRESHOLD);
    gwy_param_table_append_slider(table, PARAM_REGCOEFF);

    gwy_param_table_append_header(table, -1, _("Output"));
    gwy_param_table_append_combo(table, PARAM_RESULT);
    gwy_param_table_append_checkbox(table, PARAM_PLOT_MASK);

    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), TRUE, TRUE, 0);
    gwy_dialog_add_param_table(dialog, table);

    g_signal_connect_swapped(table, "param-changed", G_CALLBACK(param_changed), &gui);
    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_UPON_REQUEST, preview, &gui, NULL);

    return gwy_dialog_run(dialog);
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    GwyParams *params = gui->args->params;
    GwyParamTable *table = gui->table;

    if (id < 0 || id == PARAM_KERNEL) {
        GwyField *kernel = gwy_params_get_image(params, PARAM_KERNEL);
        GwyNield *mask = NULL;
        GwyAppDataId dataid = gwy_params_get_data_id(params, PARAM_KERNEL);

        gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), GWY_RESPONSE_UPDATE, !!kernel);
        gtk_dialog_set_response_sensitive(GTK_DIALOG(gui->dialog), GTK_RESPONSE_OK, !!kernel);
        if (kernel) {
            GwyFile *data = gwy_data_browser_get_file(dataid.datano);
            mask = gwy_file_get_image_mask(data, dataid.id);
        }
        gwy_param_table_set_sensitive(table, PARAM_USE_MASK, !!mask && gwy_nield_max(mask) > 0);
    }
    if (id < 0 || id == PARAM_RESULT) {
        MaskcorResult output = gwy_params_get_enum(params, PARAM_RESULT);
        gwy_param_table_set_sensitive(table, PARAM_THRESHOLD, output != MASKCOR_SCORE);
        gwy_param_table_set_sensitive(table, PARAM_PLOT_MASK, output != MASKCOR_SCORE);
    }
    if (id < 0 || id == PARAM_METHOD) {
        GwyCorrSearchType method = gwy_params_get_enum(params, PARAM_METHOD);
        gboolean is_score = (method == GWY_CORR_SEARCH_COVARIANCE_SCORE
                             || method == GWY_CORR_SEARCH_HEIGHT_DIFF_SCORE
                             || method == GWY_CORR_SEARCH_PHASE_ONLY_SCORE);
        gwy_param_table_set_sensitive(table, PARAM_REGCOEFF, is_score);
    }

    if (id != PARAM_RESULT)
        gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    ModuleArgs *args = gui->args;

    execute(args);
    gwy_field_data_changed(args->score);
    gwy_dialog_have_result(GWY_DIALOG(gui->dialog));
}

static gboolean
kernel_filter(GwyFile *data, gint id, gpointer user_data)
{
    GwyField *kernel, *field = (GwyField*)user_data;

    if (!(kernel = gwy_file_get_image(data, id)))
        return FALSE;
    if (gwy_field_get_xreal(kernel) < gwy_field_get_xreal(field)
        && gwy_field_get_yreal(kernel) < gwy_field_get_yreal(field)
        && !gwy_field_is_incompatible(kernel, field, GWY_DATA_MISMATCH_LATERAL | GWY_DATA_MISMATCH_MEASURE))
        return TRUE;

    return FALSE;
}

static void
mark_only_maxima(GwyField *field, GwyNield *mask)
{
    gint n = gwy_field_get_xres(field) * gwy_field_get_yres(field);
    gint ngrains = gwy_nield_number_contiguous(mask);

    const gdouble *data = gwy_field_get_data_const(field);
    gint *grains = gwy_nield_get_data(mask);
    gint *grain_maxima = g_new(gint, ngrains + 1);

    for (gint i = 1; i <= ngrains; i++)
        grain_maxima[i] = -1;

    /* Sum grain sizes and find maxima. */
    for (gint i = 0; i < n; i++) {
        gint j = grains[i];
        if (j && (grain_maxima[j] < 0.0 || data[grain_maxima[j]] < data[i]))
            grain_maxima[j] = i;
    }

    /* Mark only maxima. */
    gwy_nield_clear(mask);
    for (gint i = 1; i <= ngrains; i++)
        grains[grain_maxima[i]] = 1;

    g_free(grain_maxima);
}

static void
execute(ModuleArgs *args)
{
    GwyField *score = args->score, *kernel, *kweight = NULL;
    GwyNield *mask = args->mask, *kmask = NULL;
    GwyParams *params = args->params;
    gboolean use_mask = gwy_params_get_boolean(params, PARAM_USE_MASK);
    gboolean plot_mask = gwy_params_get_boolean(params, PARAM_PLOT_MASK);
    gdouble threshold = gwy_params_get_double(params, PARAM_THRESHOLD);
    gdouble regcoeff = gwy_params_get_double(params, PARAM_REGCOEFF);
    GwyCorrSearchType method = gwy_params_get_enum(params, PARAM_METHOD);
    MaskcorResult output = gwy_params_get_enum(params, PARAM_RESULT);
    gdouble min, max;

    kernel = gwy_params_get_image(params, PARAM_KERNEL);
    if (use_mask) {
        kmask = gwy_params_get_image_mask(params, PARAM_KERNEL);
        kweight = gwy_field_new_alike(kernel, FALSE);
        gwy_field_fill_mask(kweight, kmask, 0.0, 1.0);
    }

    gwy_field_correlation_search(args->field, kernel, kweight, score, method, regcoeff, GWY_EXTERIOR_BORDER, 0.0);
    gwy_field_invalidate(score);
    g_clear_object(&kweight);

    if (output == MASKCOR_SCORE)
        return;

    if (method == GWY_CORR_SEARCH_COVARIANCE_SCORE) {
        /* pass */
    }
    else if (method == GWY_CORR_SEARCH_HEIGHT_DIFF_SCORE) {
        threshold = 2.0*(threshold - 1.0);
    }
    else {
        gwy_field_min_max(score, &min, &max);
        threshold = max*threshold + min*(1.0 - threshold);
    }

    /* Now it becomes convoluted.  There are the following possible outputs:
     * (a) plain thresholded score
     * (b) single-pixel maxima (in the thresholded score)
     * (c) plain thresholded score dilated by kmask (or rectangle if kmask is not used)
     * (d) single-pixel maxima plain dilated by kmask (or rectangle if kmask is not used)
     * So we interpret OBJECTS as (a) and (c).  The difference is PLOT_MASK.
     * And we interpret MAXIMA as (b) and (d).  Again, the difference is PLOT_MASK.
     *
     * This makes impossible to use kernel mask but plot rectangles.  Do we care?
     */
    gwy_nield_mark_by_threshold(mask, score, threshold, TRUE);
    if (output == MASKCOR_MAXIMA)
        mark_only_maxima(score, mask);

    if (plot_mask) {
        if (kmask)
            g_object_ref(kmask);
        else {
            kmask = gwy_field_new_nield_alike(kernel);
            gwy_nield_fill(kmask, 1);
        }
        gint xres = gwy_field_get_xres(score), yres = gwy_field_get_yres(score);
        /* Use score because the caller should only take the mask in this case. */
        gwy_field_fill_mask(score, mask, 0.0, 1.0);
        gwy_field_area_filter_min_max(score, kmask, GWY_MIN_MAX_FILTER_DILATION, 0, 0, xres, yres);
        g_object_unref(kmask);
        gwy_nield_mark_by_threshold(mask, score, 0.5, TRUE);
    }
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
