/*
 *  $Id: maskops.c 29477 2026-02-14 13:29:30Z yeti-dn $
 *  Copyright (C) 2003-2026 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gwy.h>

#define RUN_MODES GWY_RUN_IMMEDIATE

static gboolean module_register      (void);
static void     mask_remove          (GwyFile *data,
                                      GwyRunModeFlags mode);
static void     mask_invert          (GwyFile *data,
                                      GwyRunModeFlags mode);
static void     mask_extract         (GwyFile *data,
                                      GwyRunModeFlags mode);
static void     extract_grain_numbers(GwyFile *data,
                                      GwyRunModeFlags mode);
static void     remove_touching      (GwyFile *data,
                                      GwyRunModeFlags mode);
static void     mask_thin            (GwyFile *data,
                                      GwyRunModeFlags mode);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Basic operations with mask: inversion, removal, extraction."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, maskops)

static gboolean
module_register(void)
{
    gwy_process_func_register("mask_remove",
                              mask_remove,
                              N_("/_Mask/_Remove Mask"),
                              GWY_ICON_MASK_REMOVE,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_MASK | GWY_MENU_FLAG_IMAGE,
                              N_("Remove mask from data"));
    gwy_process_func_register("mask_invert",
                              mask_invert,
                              N_("/_Mask/_Invert Mask"),
                              GWY_ICON_MASK_INVERT,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_MASK | GWY_MENU_FLAG_IMAGE,
                              N_("Invert mask"));
    gwy_process_func_register("mask_extract",
                              mask_extract,
                              N_("/_Mask/_Extract Mask"),
                              GWY_ICON_MASK_EXTRACT,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_MASK | GWY_MENU_FLAG_IMAGE,
                              N_("Extract mask to a new image"));
    gwy_process_func_register("grain_extract_numbers",
                              extract_grain_numbers,
                              N_("/_Grains/_Extract Numbers"),
                              NULL,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_MASK | GWY_MENU_FLAG_IMAGE,
                              N_("Extract grain numbers to a new image"));
    gwy_process_func_register("grain_rem_touching",
                              remove_touching,
                              N_("/_Grains/_Remove Edge-Touching"),
                              GWY_ICON_GRAINS_EDGE_REMOVE,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE | GWY_MENU_FLAG_IMAGE_MASK,
                              N_("Remove grains touching image edges"));
    gwy_process_func_register("mask_thin",
                              mask_thin,
                              N_("/_Mask/Thi_n"),
                              GWY_ICON_MASK_THIN,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_MASK | GWY_MENU_FLAG_IMAGE,
                              N_("Thin mask"));

    return TRUE;
}

static void
mask_invert(GwyFile *data, GwyRunModeFlags mode)
{
    GwyNield *mask;
    GQuark mquark;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_MASK_FIELD, &mask,
                                 GWY_APP_MASK_FIELD_KEY, &mquark,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(mask && mquark);

    gwy_app_undo_qcheckpointv(GWY_DICT(data), 1, &mquark);
    gwy_nield_invert(mask);
    gwy_nield_data_changed(mask);
    gwy_log_add(data, GWY_FILE_IMAGE, id, id);
}

static void
mask_remove(GwyFile *data, GwyRunModeFlags mode)
{
    GQuark mquark;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_MASK_FIELD_KEY, &mquark,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(mquark);

    gwy_app_undo_qcheckpointv(GWY_DICT(data), 1, &mquark);
    gwy_dict_remove(GWY_DICT(data), mquark);
    gwy_log_add(data, GWY_FILE_IMAGE, id, id);
}

static void
mask_extract(GwyFile *data, GwyRunModeFlags mode)
{
    GwyField *field;
    GwyNield *mask;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD, &field,
                                 GWY_APP_MASK_FIELD, &mask,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(mask);

    field = gwy_field_new_alike(field, FALSE);
    gwy_unit_clear(gwy_field_get_unit_z(field));
    gwy_field_fill_mask(field, mask, 0.0, 1.0);

    gint newid = gwy_file_add_image(data, field);

    gwy_file_set_visible(data, GWY_FILE_IMAGE, newid, TRUE);
    g_object_unref(field);
    gwy_file_set_title(data, GWY_FILE_IMAGE, newid, _("Mask"), TRUE);
    gwy_file_sync_items(data, GWY_FILE_IMAGE, id,
                        data, GWY_FILE_IMAGE, newid,
                        GWY_FILE_ITEM_PALETTE | GWY_FILE_ITEM_REAL_SQUARE, FALSE);
    gwy_log_add(data, GWY_FILE_IMAGE, id, newid);
}

static void
extract_grain_numbers(GwyFile *data, GwyRunModeFlags mode)
{
    GwyField *field;
    GwyNield *mask;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD, &field,
                                 GWY_APP_MASK_FIELD, &mask,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(mask);

    /* We do *not* renumber the grains. Most functions renumber contiguous when creating the output. If something
     * numbered them differently, we want to preserve these numbers. */
    field = gwy_field_new_alike(field, FALSE);
    gwy_unit_clear(gwy_field_get_unit_z(field));

    guint n = gwy_nield_get_xres(mask)*gwy_nield_get_yres(mask);
    const gint *grains = gwy_nield_get_data_const(mask);
    gdouble *d = gwy_field_get_data(field);
    for (guint i = 0; i < n; i++)
        d[i] = grains[i];

    gint newid = gwy_file_add_image(data, field);

    gwy_file_set_visible(data, GWY_FILE_IMAGE, newid, TRUE);
    g_object_unref(field);
    gwy_file_set_title(data, GWY_FILE_IMAGE, newid, _("Grain Numbers"), TRUE);
    gwy_file_sync_items(data, GWY_FILE_IMAGE, id,
                        data, GWY_FILE_IMAGE, newid,
                        GWY_FILE_ITEM_PALETTE | GWY_FILE_ITEM_REAL_SQUARE, FALSE);
    gwy_log_add(data, GWY_FILE_IMAGE, id, newid);
}

static void
remove_touching(GwyFile *data, GwyRunModeFlags mode)
{
    GwyNield *mask;
    GQuark mquark;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_MASK_FIELD, &mask,
                                 GWY_APP_MASK_FIELD_KEY, &mquark,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(mask);

    gwy_app_undo_qcheckpointv(GWY_DICT(data), 1, &mquark);
    gwy_nield_clear_touching_border(mask, TRUE);
    gwy_nield_data_changed(mask);
    gwy_log_add(data, GWY_FILE_IMAGE, id, id);
}

static void
mask_thin(GwyFile *data, GwyRunModeFlags mode)
{
    GwyNield *mask;
    GQuark quark;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_MASK_FIELD, &mask,
                                 GWY_APP_MASK_FIELD_KEY, &quark,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(mask);

    gwy_app_undo_qcheckpointv(GWY_DICT(data), 1, &quark);
    gwy_nield_thin(mask);
    gwy_nield_data_changed(mask);
    gwy_log_add(data, GWY_FILE_IMAGE, id, id);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
