/*
 *  $Id: psdf_logphi.c 29393 2026-01-28 17:56:48Z yeti-dn $
 *  Copyright (C) 2015-2026 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <gwy.h>

#define RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

enum {
    PARAM_WINDOW,
    PARAM_SIGMA,
};

typedef struct {
    GwyParams *params;
    GwyField *field;
    GwyField *result;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
} ModuleGUI;

static gboolean         module_register     (void);
static GwyParamDef*     define_module_params(void);
static void             module_main         (GwyFile *data,
                                             GwyRunModeFlags mode);
static void             execute             (ModuleArgs *args);
static GwyDialogOutcome run_gui             (ModuleArgs *args);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Two-dimensional FFT (Fast Fourier Transform) transformed to coordinates (log-frequency, angle)."),
    "Yeti <yeti@gwyddion.net>",
    "2.1",
    "David Nečas (Yeti)",
    "2015",
};

GWY_MODULE_QUERY2(module_info, psdf_logphi)

static gboolean
module_register(void)
{
    gwy_process_func_register("psdf_logphi",
                              module_main,
                              N_("/_Statistics/_Log-Phi PSDF..."),
                              GWY_ICON_PSDF_LOG_PHI,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE,
                              N_("Compute PSDF in Log-Phi coordinates"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_process_func_current());
    gwy_param_def_add_double(paramdef, PARAM_SIGMA, "sigma",  _("Gaussian _smoothing"), 0.0, 40.0, 0.0);
    gwy_param_def_add_enum(paramdef, PARAM_WINDOW, "window", NULL, GWY_TYPE_WINDOWING_TYPE, GWY_WINDOWING_HANN);
    return paramdef;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    ModuleArgs args;
    gint id, newid;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD, &args.field,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(args.field);

    args.result = gwy_field_new(1, 1, 1.0, 1.0, FALSE);
    args.params = gwy_params_new_from_settings(define_module_params());

    if (mode == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    execute(&args);

    newid = gwy_file_add_image(data, args.result);

    gwy_file_set_visible(data, GWY_FILE_IMAGE, newid, TRUE);
    gwy_file_set_title(data, GWY_FILE_IMAGE, newid, "Log-phi PSDF", TRUE);
    gwy_log_add(data, GWY_FILE_IMAGE, id, newid);

end:
    g_object_unref(args.result);
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args)
{
    GwyDialog *dialog;
    GwyParamTable *table;
    ModuleGUI gui;

    gui.args = args;

    gui.dialog = gwy_dialog_new(_("Log-Phi PSDF"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    table = gui.table = gwy_param_table_new(args->params);
    gwy_param_table_append_combo(table, PARAM_WINDOW);
    gwy_param_table_append_slider(table, PARAM_SIGMA);

    gwy_dialog_add_content(dialog, gwy_param_table_widget(table), FALSE, FALSE, 0);
    gwy_dialog_add_param_table(dialog, table);

    return gwy_dialog_run(dialog);
}

static void
execute(ModuleArgs *args)
{
    enum { N = 4 };

    GwyField *field = args->field, *lpsdf = args->result;
    gdouble sigma = gwy_params_get_double(args->params, PARAM_SIGMA);
    GwyWindowingType window = gwy_params_get_enum(args->params, PARAM_WINDOW);

    GwyField *reout = gwy_field_new_alike(field, FALSE);
    GwyField *imout = gwy_field_new_alike(field, FALSE);
    gwy_field_fft_2d(field, NULL, reout, imout, window, GWY_TRANSFORM_DIRECTION_FORWARD, TRUE, 1);

    gint pxres = gwy_field_get_xres(reout), pyres = gwy_field_get_yres(reout);
    gdouble *redata = gwy_field_get_data(reout);
    gdouble *imdata = gwy_field_get_data(imout);
    for (gint i = 0; i < pxres*pyres; i++)
        redata[i] = redata[i]*redata[i] + imdata[i]*imdata[i];
    gwy_field_fft_2d_center(reout);
    gwy_field_filter_gaussian(reout, sigma);
    redata = gwy_field_get_data(reout);
    for (gint i = 0; i < pxres*pyres; i++)
        redata[i] = sqrt(redata[i]);

    gint fxres = pxres/2;
    gint fyres = pyres/2;
    gwy_field_resize(lpsdf, fxres, fyres);
    gdouble *ldata = gwy_field_get_data(lpsdf);

    gdouble xreal = gwy_field_get_xreal(field);
    gdouble yreal = gwy_field_get_yreal(field);
    gdouble f0 = 2.0/MIN(xreal, yreal);
    gdouble f_max = 0.5*MIN(pxres/xreal, pyres/yreal);
    if (f_max <= f0) {
        g_warning("Minimum frequency is not smaller than maximum frequency.");
    }
    gdouble b = log(f_max/f0)/fyres;

    /* Incorporate some prefactors to sinphi[] and cosphi[], knowing that
     * cosine is only ever used for x and sine for y frequencies. */
    gdouble *cosphi = g_new(gdouble, (N+1)*fxres);
    gdouble *sinphi = g_new(gdouble, (N+1)*fxres);
    for (gint j = 0; j < fxres; j++) {
        gdouble phi_from = 2.0*G_PI*j/fxres;
        gdouble phi_to = 2.0*G_PI*(j + 1.0)/fxres;

        for (gint pi = 0; pi <= N; pi++) {
            gdouble phi = ((pi + 0.5)*phi_from + (N - 0.5 - pi)*phi_to)/N;
            cosphi[j*(N+1) + pi] = cos(phi)*xreal;
            sinphi[j*(N+1) + pi] = sin(phi)*yreal;
        }
    }

#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            shared(fxres,fyres,pxres,pyres,cosphi,sinphi,f0,b,reout,ldata)
#endif
    for (gint i = 0; i < fyres; i++) {
        gdouble f_from = f0*exp(b*i);
        gdouble f_to = f0*exp(b*(i + 1.0));

        for (gint j = 0; j < fxres; j++) {
            const gdouble *cosphi_j = cosphi + j*(N+1);
            const gdouble *sinphi_j = sinphi + j*(N+1);
            guint n = 0;
            gdouble s = 0.0;

            for (gint fi = 0; fi <= N; fi++) {
                gdouble f = ((fi + 0.5)*f_from + (N - 0.5 - fi)*f_to)/N;
                for (gint pi = 0; pi <= N; pi++) {
                    gdouble x = f*cosphi_j[pi] + pxres/2.0;
                    gdouble y = f*sinphi_j[pi] + pyres/2.0;

                    if (G_UNLIKELY(x < 0.5 || y < 0.5 || x > pxres - 1.5 || y > pyres - 1.5))
                        continue;

                    gdouble p = gwy_field_get_dval(reout, x, y, GWY_INTERPOLATION_SCHAUM);
                    s += p;
                    n++;
                }
            }

            ldata[i*fxres + j] = 2.0*G_PI/fxres * s/MAX(n, 1)*(f_to - f_from);
        }
    }

    g_object_unref(imout);
    g_object_unref(reout);

    gwy_field_set_xreal(lpsdf, 2.0*G_PI);
    gwy_field_set_xoffset(lpsdf, 0.0);
    gwy_field_set_yreal(lpsdf, log(f_max/f0));
    gwy_field_set_yoffset(lpsdf, log(f0));
    gwy_unit_clear(gwy_field_get_unit_xy(lpsdf));
    gwy_unit_clear(gwy_field_get_unit_z(lpsdf));
    gwy_field_normalize(lpsdf);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
