/*
 *  $Id: mfm_current.c 29407 2026-01-30 15:34:09Z yeti-dn $
 *  Copyright (C) 2017-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gwy.h>
#include "preview.h"

#define RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

enum {
    PARAM_ACTIVE_PAGE,
    PARAM_UPDATE,
    PARAM_OUT,
    PARAM_PROBE,
    PARAM_HEIGHT,
    PARAM_CURRENT,
    PARAM_WIDTH,
    PARAM_POSITION,
    PARAM_MTIP,
    PARAM_BX,
    PARAM_BY,
    PARAM_LENGTH,

    PARAM_DIMS0
};

typedef enum {
    GWY_MFM_CURRENT_OUTPUT_HX        = 0,
    GWY_MFM_CURRENT_OUTPUT_HZ        = 1,
    GWY_MFM_CURRENT_OUTPUT_FORCE     = 2,
    GWY_MFM_CURRENT_OUTPUT_FORCE_DX  = 3,
    GWY_MFM_CURRENT_OUTPUT_FORCE_DDX = 4
} GwyMfmCurrentOutputType;

typedef struct {
    GwyParams *params;
    GwyField *field;
    GwyField *result;
    /* Cached input image parameters. */
    gdouble zscale;  /* Negative value means there is no input image. */
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GtkWidget *dataview;
    GwyParamTable *table_dimensions;
    GwyParamTable *table_generator;
    GwyField *template_;
} ModuleGUI;

static gboolean         module_register   (void);
static void             module_main       (GwyFile *data,
                                           GwyRunModeFlags mode);
static GwyDialogOutcome run_gui           (ModuleArgs *args,
                                           GwyFile *data,
                                           gint id);
static GtkWidget*       dimensions_tab_new(ModuleGUI *gui);
static GtkWidget*       generator_tab_new (ModuleGUI *gui);
static void             param_changed     (ModuleGUI *gui,
                                           gint id);
static void             dialog_response   (ModuleGUI *gui,
                                           gint response);
static void             preview           (gpointer user_data);
static void             execute           (ModuleArgs *args);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Simulation of current line magnetic field"),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2017",
};

GWY_MODULE_QUERY2(module_info, mfm_current)

static gboolean
module_register(void)
{
    gwy_synth_func_register("mfm_current",
                            module_main,
                            N_("/_Magnetic/_Current Line Field..."),
                            GWY_ICON_MFM_CURRENT_LINE,
                            RUN_MODES,
                            N_("Simulate stray field above current line"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum mfm_current_outputs[] = {
        {
            "H<sub>x</sub>",
            GWY_MFM_CURRENT_OUTPUT_HX,
        },
        {
            "H<sub>z</sub>",
            GWY_MFM_CURRENT_OUTPUT_HZ,
        },
        {
            "F<sub>z</sub>",
            GWY_MFM_CURRENT_OUTPUT_FORCE,
        },
        {
            "dF<sub>z</sub>/dz",
            GWY_MFM_CURRENT_OUTPUT_FORCE_DX,
        },
        {
            "d<sup>2</sup>F<sub>z</sub>/dz<sup>2</sup>",
            GWY_MFM_CURRENT_OUTPUT_FORCE_DDX,
        },
    };
    static const GwyEnum mfm_current_probes[] = {
        { N_("Point charge"), GWY_MFM_PROBE_CHARGE, },
        { N_("Bar"),          GWY_MFM_PROBE_BAR,   },
    };

    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_synth_func_current());
    gwy_param_def_add_gwyenum(paramdef, PARAM_OUT, "out", _("Output _type"),
                              mfm_current_outputs, G_N_ELEMENTS(mfm_current_outputs), GWY_MFM_CURRENT_OUTPUT_HZ);
    gwy_param_def_add_gwyenum(paramdef, PARAM_PROBE, "probe", _("P_robe type"), mfm_current_probes,
                              G_N_ELEMENTS(mfm_current_probes), GWY_MFM_PROBE_CHARGE);
    gwy_param_def_add_double(paramdef, PARAM_HEIGHT, "height", _("_Output plane height"), 1, 1000, 100);
    gwy_param_def_add_double(paramdef, PARAM_CURRENT, "current", _("Stripe _current"), -1000, 1000, 1);
    gwy_param_def_add_double(paramdef, PARAM_WIDTH, "width", _("_Stripe width"), 1, 1000, 100);
    gwy_param_def_add_double(paramdef, PARAM_POSITION, "position", _("_Position"), 1, 100, 50);
    gwy_param_def_add_double(paramdef, PARAM_MTIP, "mtip", _("Tip _magnetization"), 1, 10000, 1);
    gwy_param_def_add_double(paramdef, PARAM_BX, "bx", _("Bar width _x"), 1, 1000, 10);
    gwy_param_def_add_double(paramdef, PARAM_BY, "by", _("Bar width _y"), 1, 1000, 10);
    gwy_param_def_add_double(paramdef, PARAM_LENGTH, "length", _("Bar length (_z)"), 1, 10000, 500);
    gwy_param_def_add_instant_updates(paramdef, PARAM_UPDATE, "update", NULL, TRUE);
    gwy_param_def_add_active_page(paramdef, PARAM_ACTIVE_PAGE, "active_page", NULL);
    gwy_synth_define_dimensions_params(paramdef, PARAM_DIMS0);
    return paramdef;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    ModuleArgs args;
    GwyField *field;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_clear1(args);

    gwy_data_browser_get_current(GWY_APP_FIELD, &field,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    if (field && gwy_unit_equal_string(gwy_field_get_unit_xy(field), "m"))
       args.field = field;
    else
       args.field = field = NULL;

    args.zscale = field ? gwy_field_rms(field) : -1.0;

    args.params = gwy_params_new_from_settings(define_module_params());
    gwy_synth_sanitise_params(args.params, PARAM_DIMS0, field);

    if (mode == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, id);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    args.result = gwy_synth_make_result_field((args.field = field), args.params, FALSE);
    execute(&args);

    gwy_synth_add_image_to_file(args.result, data, id, args.params);

end:
    g_clear_object(&args.result);
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyFile *data, gint id)
{
    GwyDialogOutcome outcome;
    GwyDialog *dialog;
    GtkWidget *hbox;// *dataview;
    GtkNotebook *notebook;
    ModuleGUI gui;

    gwy_clear1(gui);
    gui.args = args;
    gui.template_ = args->field;

    args->result = gwy_synth_make_result_field(args->field, args->params, TRUE);

    gui.dialog = gwy_dialog_new(_("Current Line Stray Field"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_UPDATE, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    gui.dataview = gwy_create_preview(args->result, NULL, PREVIEW_SIZE);
    if (gui.template_)
        gwy_setup_data_view(GWY_DATA_VIEW(gui.dataview), data, GWY_FILE_IMAGE, id, GWY_FILE_ITEM_PALETTE);
    hbox = gwy_create_dialog_preview_hbox(GTK_DIALOG(dialog), GWY_DATA_VIEW(gui.dataview), FALSE);

    notebook = GTK_NOTEBOOK(gtk_notebook_new());
    gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(notebook), TRUE, TRUE, 0);

    gtk_notebook_append_page(notebook, dimensions_tab_new(&gui), gtk_label_new(_("Dimensions")));
    gtk_notebook_append_page(notebook, generator_tab_new(&gui), gtk_label_new(_("Generator")));
    gwy_param_active_page_link_to_notebook(args->params, PARAM_ACTIVE_PAGE, notebook);

    g_signal_connect_swapped(gui.table_dimensions, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(gui.table_generator, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_swapped(dialog, "response", G_CALLBACK(dialog_response), &gui);
    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_IMMEDIATE, preview, &gui, NULL);

    outcome = gwy_dialog_run(dialog);

    g_clear_object(&args->result);

    return outcome;
}

static GtkWidget*
dimensions_tab_new(ModuleGUI *gui)
{
    gui->table_dimensions = gwy_param_table_new(gui->args->params);
    gwy_synth_append_dimensions_to_param_table(gui->table_dimensions, GWY_SYNTH_FIXED_ZUNIT | GWY_SYNTH_NO_INITIALIZE);
    gwy_dialog_add_param_table(GWY_DIALOG(gui->dialog), gui->table_dimensions);

    return gwy_param_table_widget(gui->table_dimensions);
}

static GtkWidget*
generator_tab_new(ModuleGUI *gui)
{
    GwyParamTable *table;

    table = gui->table_generator = gwy_param_table_new(gui->args->params);

    gwy_param_table_append_header(table, -1, _("Output"));
    gwy_param_table_append_slider(table, PARAM_HEIGHT);
    gwy_param_table_set_unitstr(table, PARAM_HEIGHT, "nm");
    gwy_param_table_append_slider(table, PARAM_WIDTH);
    gwy_param_table_set_unitstr(table, PARAM_WIDTH, "nm");
    gwy_param_table_append_slider(table, PARAM_CURRENT);
    gwy_param_table_set_unitstr(table, PARAM_CURRENT, "mA");
    gwy_param_table_append_slider(table, PARAM_POSITION);
    gwy_param_table_set_unitstr(table, PARAM_POSITION, "%");
    gwy_param_table_append_combo(table, PARAM_OUT);

    gwy_param_table_append_header(table, -1, _("Probe"));
    gwy_param_table_append_combo(table, PARAM_PROBE);
    gwy_param_table_append_slider(table, PARAM_MTIP);
    gwy_param_table_set_unitstr(table, PARAM_MTIP, "kA/m");
    gwy_param_table_append_slider(table, PARAM_BX);
    gwy_param_table_set_unitstr(table, PARAM_BX, "nm");
    gwy_param_table_append_slider(table, PARAM_BY);
    gwy_param_table_set_unitstr(table, PARAM_BY, "nm");
    gwy_param_table_append_slider(table, PARAM_LENGTH);
    gwy_param_table_set_unitstr(table, PARAM_LENGTH, "nm");

    gwy_param_table_append_header(table, -1, _("Options"));
    gwy_param_table_append_checkbox(table, PARAM_UPDATE);

    gwy_dialog_add_param_table(GWY_DIALOG(gui->dialog), table);

    return gwy_param_table_widget(table);
}

static void //needs update from someone who knows what it is for
param_changed(ModuleGUI *gui, gint id)
{
    GwyParams *params = gui->args->params;
    gint power10, xyid = PARAM_DIMS0 + GWY_DIMS_PARAM_XYUNIT;

    if (gwy_synth_handle_param_changed(gui->table_dimensions, id))
        id = -1;

    if ((id < 0 || id == xyid) && !gwy_unit_equal_string(gwy_params_get_unit(params, xyid, &power10), "m")) {
        gwy_param_table_set_string(gui->table_dimensions, xyid, "µm");
        gwy_synth_handle_param_changed(gui->table_dimensions, xyid);
    }

    if (id != PARAM_ACTIVE_PAGE && id != PARAM_UPDATE)
        gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void //needs update from someone who knows what it is for
dialog_response(ModuleGUI *gui, gint response)
{
    ModuleArgs *args = gui->args;

    if (response == GWY_RESPONSE_SYNTH_INIT_Z) {
        gdouble zscale = args->zscale;
        gint power10z;

        if (zscale > 0.0) {
            gwy_params_get_unit(args->params, PARAM_DIMS0 + GWY_DIMS_PARAM_ZUNIT, &power10z);
            //gwy_param_table_set_double(gui->table_generator, PARAM_SIGMA, zscale/gwy_exp10(power10z));
            //same case as param_changed, presumably should do something but I don't know what
        }
    }
    else if (response == GWY_RESPONSE_SYNTH_TAKE_DIMS) {
        gwy_synth_use_dimensions_template(gui->table_dimensions);
    }
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    ModuleArgs *args = gui->args;

    g_clear_object(&args->result);
    args->result = gwy_synth_make_result_field(args->field, args->params, FALSE);
    gwy_data_view_set_field(GWY_DATA_VIEW(gui->dataview), args->result);
    gwy_set_data_preview_size(GWY_DATA_VIEW(gui->dataview), PREVIEW_SIZE);

    execute(args);
    gwy_field_data_changed(args->result);
}

static void
execute(ModuleArgs *args)
{
    GwyParams *params = args->params;
    GwyField *result = args->result;
    GwyMFMComponentType component;
    GwyField *tmp;
    gdouble height = gwy_params_get_double(params, PARAM_HEIGHT)*1e-9,
            width = gwy_params_get_double(params, PARAM_WIDTH)*1e-9,
            length = gwy_params_get_double(params, PARAM_LENGTH)*1e-9,
            bx = gwy_params_get_double(params, PARAM_BX)*1e-9,
            by = gwy_params_get_double(params, PARAM_BY)*1e-9,
            current = gwy_params_get_double(params, PARAM_CURRENT)*1e-3,
            mtip = gwy_params_get_double(params, PARAM_MTIP)*1e3,
            position = gwy_params_get_double(params, PARAM_POSITION)*gwy_field_get_xreal(result)/100;
    gint out = gwy_params_get_enum(params, PARAM_OUT);
    gint probe = gwy_params_get_enum(params, PARAM_PROBE);

    if (out == GWY_MFM_CURRENT_OUTPUT_HX)
        component = GWY_MFM_COMPONENT_HX;
    else if (out == GWY_MFM_CURRENT_OUTPUT_HZ || out == GWY_MFM_CURRENT_OUTPUT_FORCE)
        component = GWY_MFM_COMPONENT_HZ;
    else if (out == GWY_MFM_CURRENT_OUTPUT_FORCE_DX)
        component = GWY_MFM_COMPONENT_DHZ_DZ;
    else if (out == GWY_MFM_CURRENT_OUTPUT_FORCE_DDX)
        component = GWY_MFM_COMPONENT_D2HZ_DZ2;
    else {
        g_return_if_reached();
    }

    gwy_field_mfm_current_line(result, height, width, position, current, component);

    if (out == GWY_MFM_CURRENT_OUTPUT_FORCE
        || out == GWY_MFM_CURRENT_OUTPUT_FORCE_DX
        || out == GWY_MFM_CURRENT_OUTPUT_FORCE_DDX) {
        tmp = gwy_field_copy(result);
        gwy_field_mfm_perpendicular_medium_force(tmp, result, probe, mtip, bx, by, length);
        g_object_unref(tmp);
    }
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
