/*
 *  $Id: grainremover.c 29447 2026-02-07 12:14:37Z yeti-dn $
 *  Copyright (C) 2003-2026 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <gwy.h>

enum {
    PARAM_MODE,
    PARAM_METHOD,
};

typedef enum {
    GRAIN_REMOVE_MASK = 1 << 0,
    GRAIN_REMOVE_DATA = 1 << 1,
    GRAIN_REMOVE_BOTH = GRAIN_REMOVE_DATA | GRAIN_REMOVE_MASK
} RemoveMode;

typedef enum {
    GRAIN_REMOVE_LAPLACE         = 1,
    GRAIN_REMOVE_FRACTAL         = 2,
    GRAIN_REMOVE_FRACTAL_LAPLACE = 3,
    GRAIN_REMOVE_ZERO            = 4,
} RemoveAlgorithm;

#define GWY_TYPE_TOOL_GRAIN_REMOVER            (gwy_tool_grain_remover_get_type())
#define GWY_TOOL_GRAIN_REMOVER(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj), GWY_TYPE_TOOL_GRAIN_REMOVER, GwyToolGrainRemover))
#define GWY_IS_TOOL_GRAIN_REMOVER(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj), GWY_TYPE_TOOL_GRAIN_REMOVER))
#define GWY_TOOL_GRAIN_REMOVER_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj), GWY_TYPE_TOOL_GRAIN_REMOVER, GwyToolGrainRemoverClass))

typedef struct _GwyToolGrainRemover      GwyToolGrainRemover;
typedef struct _GwyToolGrainRemoverClass GwyToolGrainRemoverClass;

struct _GwyToolGrainRemover {
    GwyPlainTool parent_instance;

    GwyParams *params;
    GwyParamTable *table;
};

struct _GwyToolGrainRemoverClass {
    GwyPlainToolClass parent_class;
};

static gboolean     module_register                          (void);
static GwyParamDef* define_module_params                     (void);
static GType        gwy_tool_grain_remover_get_type          (void)                       G_GNUC_CONST;
static void         gwy_tool_grain_remover_finalize          (GObject *object);
static void         gwy_tool_grain_remover_init_dialog       (GwyToolGrainRemover *tool);
static void         gwy_tool_grain_remover_data_switched     (GwyTool *gwytool,
                                                              GwyDataView *data_view);
static void         gwy_tool_grain_remover_selection_finished(GwyPlainTool *plain_tool);
static void         param_changed                            (GwyToolGrainRemover *tool,
                                                              gint id);
static void         expand_bbox                              (GwyNield *nield,
                                                              gint *bbox);
static void         laplace_interpolation                    (GwyField *field,
                                                              GwyNield *grain,
                                                              const gint *bbox);
static void         fractal_laplace_interpolation            (GwyField *field,
                                                              GwyNield *grain,
                                                              const gint *bbox);
static void         fill_with_zero                           (GwyField *field,
                                                              GwyNield *grain,
                                                              const gint *bbox);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Grain removal tool, removes continuous parts of mask and/or underlying data."),
    "Petr Klapetek <klapetek@gwyddion.net>, Yeti <yeti@gwyddion.net>",
    "5.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2003",
};

GWY_MODULE_QUERY2(module_info, grainremover)

G_DEFINE_TYPE(GwyToolGrainRemover, gwy_tool_grain_remover, GWY_TYPE_PLAIN_TOOL)

static gboolean
module_register(void)
{
    gwy_tool_func_register(GWY_TYPE_TOOL_GRAIN_REMOVER);

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum modes[] = {
        { N_("_Mask"), GRAIN_REMOVE_MASK },
        { N_("_Data"), GRAIN_REMOVE_DATA },
        { N_("_Both"), GRAIN_REMOVE_BOTH },
    };
    static const GwyEnum methods[] = {
        { N_("Laplace solver"),        GRAIN_REMOVE_LAPLACE,         },
        { N_("Fractal correction"),    GRAIN_REMOVE_FRACTAL,         },
        { N_("Fractal-Laplace blend"), GRAIN_REMOVE_FRACTAL_LAPLACE, },
        { N_("Zero"),                  GRAIN_REMOVE_ZERO,            },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, "grainremover");
    gwy_param_def_add_gwyenum(paramdef, PARAM_MODE, "mode", _("Remove"),
                              modes, G_N_ELEMENTS(modes), GRAIN_REMOVE_BOTH);
    gwy_param_def_add_gwyenum(paramdef, PARAM_METHOD, "method", _("_Interpolation method"),
                              methods, G_N_ELEMENTS(methods), GRAIN_REMOVE_LAPLACE);

    return paramdef;
}

static void
gwy_tool_grain_remover_class_init(GwyToolGrainRemoverClass *klass)
{
    GwyPlainToolClass *ptool_class = GWY_PLAIN_TOOL_CLASS(klass);
    GwyToolClass *tool_class = GWY_TOOL_CLASS(klass);
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->finalize = gwy_tool_grain_remover_finalize;

    tool_class->icon_name = GWY_ICON_GRAINS_REMOVE;
    tool_class->title = _("Grain Remove");
    tool_class->tooltip = _("Remove individual grains (continuous parts of mask)");
    tool_class->prefix = "/module/grainremover";
    tool_class->data_switched = gwy_tool_grain_remover_data_switched;

    ptool_class->selection_finished = gwy_tool_grain_remover_selection_finished;
}

static void
gwy_tool_grain_remover_finalize(GObject *object)
{
    GwyToolGrainRemover *tool = GWY_TOOL_GRAIN_REMOVER(object);

    gwy_params_save_to_settings(tool->params);
    g_clear_object(&tool->params);

    G_OBJECT_CLASS(gwy_tool_grain_remover_parent_class)->finalize(object);
}

static void
gwy_tool_grain_remover_init(GwyToolGrainRemover *tool)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);

    tool->params = gwy_params_new_from_settings(define_module_params());

    gwy_plain_tool_connect_selection(plain_tool, GWY_TYPE_LAYER_POINT, "pointer");

    gwy_tool_grain_remover_init_dialog(tool);
}

static void
gwy_tool_grain_remover_init_dialog(GwyToolGrainRemover *tool)
{
    GtkDialog *dialog = GTK_DIALOG(GWY_TOOL(tool)->dialog);
    GwyParamTable *table;

    table = tool->table = gwy_param_table_new(tool->params);
    gwy_param_table_append_radio(table, PARAM_MODE);
    gwy_param_table_append_combo(table, PARAM_METHOD);
    gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(dialog)), gwy_param_table_widget(table), FALSE, FALSE, 0);
    gwy_plain_tool_add_param_table(GWY_PLAIN_TOOL(tool), table);

    gwy_tool_add_hide_button(GWY_TOOL(tool), TRUE);
    gwy_help_add_to_tool_dialog(dialog, GWY_TOOL(tool), GWY_HELP_DEFAULT);

    g_signal_connect_swapped(tool->table, "param-changed", G_CALLBACK(param_changed), tool);

    param_changed(tool, -1);

    gtk_widget_show_all(gtk_dialog_get_content_area(dialog));
}

static void
gwy_tool_grain_remover_data_switched(GwyTool *gwytool,
                                     GwyDataView *data_view)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(gwytool);
    gboolean ignore = (data_view == plain_tool->data_view);

    GWY_TOOL_CLASS(gwy_tool_grain_remover_parent_class)->data_switched(gwytool, data_view);

    if (ignore || plain_tool->init_failed)
        return;

    if (data_view) {
        gwy_object_set_or_reset(plain_tool->layer, GWY_TYPE_LAYER_POINT,
                                "draw-marker", FALSE,
                                "editable", TRUE,
                                "focus", -1,
                                NULL);
        gwy_selection_set_max_objects(plain_tool->selection, 1);
    }
}

static void
param_changed(GwyToolGrainRemover *tool, gint id)
{
    if (id < 0 || id == PARAM_MODE) {
        RemoveMode mode = gwy_params_get_enum(tool->params, PARAM_MODE);

        gwy_param_table_set_sensitive(tool->table, PARAM_METHOD,
                                      mode == GRAIN_REMOVE_DATA || mode == GRAIN_REMOVE_BOTH);
    }
}

static void
gwy_tool_grain_remover_selection_finished(GwyPlainTool *plain_tool)
{
    GwyToolGrainRemover *tool = GWY_TOOL_GRAIN_REMOVER(plain_tool);
    RemoveMode mode = gwy_params_get_enum(tool->params, PARAM_MODE);
    RemoveAlgorithm method = gwy_params_get_enum(tool->params, PARAM_METHOD);
    GwyField *field = plain_tool->field;
    GwyNield *mask = plain_tool->mask_field;
    gint col, row;
    gdouble point[2];
    GQuark quarks[2];

    if (!mask || !gwy_selection_get_object(plain_tool->selection, 0, point))
        return;

    row = floor(gwy_field_rtoi(field, point[1]));
    col = floor(gwy_field_rtoj(field, point[0]));
    if (!gwy_nield_get_val(mask, col, row))
        return;

    quarks[0] = quarks[1] = 0;
    if (mode & GRAIN_REMOVE_DATA)
        quarks[0] = gwy_file_key_image(plain_tool->id);
    if (mode & GRAIN_REMOVE_MASK)
        quarks[1] = gwy_file_key_image_mask(plain_tool->id);

    gwy_app_undo_qcheckpointv(plain_tool->container, 2, quarks);
    if (mode & GRAIN_REMOVE_DATA) {
        GwyNield *tmp = gwy_nield_copy(mask);
        gint bbox[4];
        gwy_nield_isolate_at(tmp, col, row, bbox);
        expand_bbox(mask, bbox);

        if (method == GRAIN_REMOVE_LAPLACE)
            laplace_interpolation(field, tmp, bbox);
        else if (method == GRAIN_REMOVE_FRACTAL)
            gwy_field_fractal_correction(field, tmp, GWY_INTERPOLATION_LINEAR);
        else if (method == GRAIN_REMOVE_FRACTAL_LAPLACE)
            fractal_laplace_interpolation(field, tmp, bbox);
        else if (method == GRAIN_REMOVE_ZERO)
            fill_with_zero(field, tmp, bbox);
        else {
            g_assert_not_reached();
        }
        g_object_unref(tmp);
        gwy_field_data_changed(field);
    }
    if (mode & GRAIN_REMOVE_MASK) {
        gwy_nield_clear_at(mask, col, row);
        gwy_nield_data_changed(mask);
    }
    gwy_params_save_to_settings(tool->params);   /* Ensure correct parameters in the log. */
    gwy_plain_tool_log_add(plain_tool);
    gwy_selection_clear(plain_tool->selection);
}

static void
expand_bbox(GwyNield *nield, gint *bbox)
{
    if (bbox[0] > 0) {
        bbox[0]--;
        bbox[2]++;
    }
    if (bbox[1] > 0) {
        bbox[1]--;
        bbox[3]++;
    }
    if (bbox[0] + bbox[2] < gwy_nield_get_xres(nield))
        bbox[2]++;
    if (bbox[1] + bbox[3] < gwy_nield_get_yres(nield))
        bbox[3]++;
}

static void
laplace_interpolation(GwyField *field, GwyNield *grain, const gint *bbox)
{
    gint col = bbox[0], row = bbox[1], w = bbox[2], h = bbox[3];

    /* Work on extracted area for better memory locality. */
    GwyField *area = gwy_field_area_extract(field, col, row, w, h);
    GwyNield *mask = gwy_nield_area_extract(grain, col, row, w, h);
    gwy_field_laplace_solve(area, mask, 1, 2.0);
    g_object_unref(mask);
    gwy_field_area_copy(area, field, 0, 0, w, h, col, row);
    g_object_unref(area);
}

/* XXX: Common with spotremove.c */
static void
blend_fractal_and_laplace(GwyField *field,
                          GwyField *laplace_area,
                          GwyField *distances,
                          gint col, gint row)
{
    gint xres = gwy_field_get_xres(field);
    gint w = gwy_field_get_xres(laplace_area);
    gint h = gwy_field_get_yres(laplace_area);
    const gdouble *a = gwy_field_get_data_const(laplace_area);
    const gdouble *e = gwy_field_get_data_const(distances);
    gdouble *d = gwy_field_get_data(field) + row*xres + col;

#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            shared(a,e,d,xres,h,w)
#endif
    for (gint i = 0; i < h; i++) {
        gdouble *drow = d + xres*i;
        const gdouble *arow = a + w*i;
        const gdouble *erow = e + w*i;
        for (gint j = 0; j < w; j++) {
            if (erow[j] <= 0.0)
                continue;

            gdouble t = exp(0.167*(1.0 - erow[j]));
            drow[j] *= (1.0 - t);
            drow[j] += t*arow[j];
        }
    }
}

static void
fractal_laplace_interpolation(GwyField *field, GwyNield *grain, const gint *bbox)
{
    gint col = bbox[0], row = bbox[1], w = bbox[2], h = bbox[3];

    /* Extract the area for Laplace.  Then overwrite it with fractal interpolation. */
    GwyField *laplace = gwy_field_area_extract(field, col, row, w, h);
    GwyNield *mask = gwy_nield_area_extract(grain, col, row, w, h);
    gwy_field_laplace_solve(laplace, mask, 1, 1.0);
    GwyField *distances = gwy_field_new_alike(laplace, FALSE);
    gwy_nield_distance_transform(mask, distances, GWY_DISTANCE_TRANSFORM_EUCLIDEAN, TRUE, FALSE);
    g_object_unref(mask);

    gwy_field_fractal_correction(field, grain, GWY_INTERPOLATION_LINEAR);
    blend_fractal_and_laplace(field, laplace, distances, col, row);
    g_object_unref(laplace);
    g_object_unref(distances);
}

static void
fill_with_zero(GwyField *field, GwyNield *grain, const gint *bbox)
{
    gint col = bbox[0], row = bbox[1], w = bbox[2], h = bbox[3];
    gwy_field_area_fill(field, grain, GWY_MASK_INCLUDE, col, row, w, h, 0.0);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
