/*
 *  $Id: volume_scars.c 29521 2026-02-23 10:24:50Z yeti-dn $
 *  Copyright (C) 2015-2021 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gwy.h>
#include "libgwyddion/omp.h"

#define RUN_MODES (GWY_RUN_INTERACTIVE)

enum {
    PREVIEW_SIZE = 360,
    MAX_LENGTH = 1024
};

enum {
    PARAM_TYPE,
    PARAM_THRESHOLD_HIGH,
    PARAM_THRESHOLD_LOW,
    PARAM_MIN_LENGTH,
    PARAM_MAX_WIDTH,
    PARAM_UPDATE,
    PARAM_MASK_COLOR,
    PARAM_Z,
};

typedef struct {
    GwyParams *params;
    GwyBrick *brick;
    GwyBrick *result;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GtkWidget *dataview;
    GwyParamTable *table_options;
} ModuleGUI;

static gboolean         module_register     (void);
static GwyParamDef*     define_module_params(void);
static void             module_main         (GwyFile *data,
                                             GwyRunModeFlags mode);
static gboolean         execute             (ModuleArgs *args,
                                             GtkWindow *wait_window);
static GwyDialogOutcome run_gui             (ModuleArgs *args,
                                             GwyFile *data,
                                             gint id);
static void             param_changed       (ModuleGUI *gui,
                                             gint id);
static void             dialog_response     (GwyDialog *dialog,
                                             gint response,
                                             ModuleGUI *gui);
static void             preview             (gpointer user_data);
static void             update_image        (ModuleGUI *gui,
                                             gint z);
static void             sanitise_params     (ModuleArgs *args);


static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Performs scars removal for all the volume data levels."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.1",
    "Petr Klapetek & David Nečas (Yeti)",
    "2023",
};

GWY_MODULE_QUERY2(module_info, volume_scars)

static gboolean
module_register(void)
{
    gwy_volume_func_register("volume_scars",
                             module_main,
                             N_("/_Correct Data/_Scars..."),
                             NULL,
                             RUN_MODES,
                             GWY_MENU_FLAG_VOLUME,
                             N_("Remove scars in all levels"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum feature_types[] = {
        { N_("Positive"), GWY_SIGN_POSITIVE, },
        { N_("Negative"), GWY_SIGN_NEGATIVE, },
        { N_("Both"),     GWY_SIGN_BOTH,     },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, "scars");
    gwy_param_def_add_int(paramdef, PARAM_Z, "z", "Preview level", 0, G_MAXINT, 0);
    gwy_param_def_add_gwyenum(paramdef, PARAM_TYPE, "type", _("Scars type"),
                              feature_types, G_N_ELEMENTS(feature_types), GWY_SIGN_BOTH);
    gwy_param_def_add_double(paramdef, PARAM_THRESHOLD_HIGH, "threshold_high", _("_Hard threshold"), 0.0, 2.0, 0.666);
    gwy_param_def_add_double(paramdef, PARAM_THRESHOLD_LOW, "threshold_low", _("_Soft threshold"), 0.0, 2.0, 0.25);
    gwy_param_def_add_int(paramdef, PARAM_MIN_LENGTH, "min_len", _("Minimum _length"), 1, MAX_LENGTH, 16);
    gwy_param_def_add_int(paramdef, PARAM_MAX_WIDTH, "max_width", _("Maximum _width"), 1, 16, 4);
    gwy_param_def_add_instant_updates(paramdef, PARAM_UPDATE, "update", NULL, TRUE);
    gwy_param_def_add_mask_color(paramdef, PARAM_MASK_COLOR, NULL, NULL);
    return paramdef;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    ModuleArgs args;
    GwyBrick *brick = NULL;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    gint oldid, newid;

    g_return_if_fail(mode & RUN_MODES);

    gwy_data_browser_get_current(GWY_APP_BRICK, &brick,
                                 GWY_APP_BRICK_ID, &oldid,
                                 0);
    g_return_if_fail(GWY_IS_BRICK(brick));
    args.result = NULL;
    args.brick = brick;
    args.params = gwy_params_new_from_settings(define_module_params());
    sanitise_params(&args);

    if (mode == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, oldid);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (execute(&args, gwy_data_browser_get_window_for_data(data, GWY_FILE_VOLUME, oldid))) {
        newid = gwy_file_add_volume(data, args.result, NULL);
        gwy_file_set_visible(data, GWY_FILE_VOLUME, newid, TRUE);
        gwy_file_set_title(data, GWY_FILE_VOLUME, newid, _("Scars corrected"), TRUE);
        gwy_file_sync_items(data, GWY_FILE_VOLUME, oldid,
                            data, GWY_FILE_VOLUME, newid,
                            GWY_FILE_ITEM_PALETTE, FALSE);
        gwy_log_add(data, GWY_FILE_VOLUME, -1, newid);
    }

end:
    g_object_unref(args.params);
    g_object_unref(args.result);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyFile *data, gint id)
{
    GtkWidget *hbox, *dataview;
    GwyParamTable *table;
    GwyDialog *dialog;
    ModuleGUI gui;
    GwyDialogOutcome outcome;
    GwyBrick *brick = args->brick;
    GwyField *field = gwy_brick_new_field_like_xy_plane(brick, TRUE);
    GwyNield *mask = gwy_field_new_nield_alike(field);

    gwy_clear1(gui);
    gui.args = args;

    args->result = gwy_brick_copy(brick);

    gui.dialog = gwy_dialog_new(_("Scars"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_UPDATE, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    dataview = gui.dataview = gwy_create_preview(field, mask, PREVIEW_SIZE);
    gwy_setup_data_view(GWY_DATA_VIEW(dataview), data, GWY_FILE_VOLUME, id, GWY_FILE_ITEM_PALETTE);
    hbox = gwy_create_dialog_preview_hbox(GTK_DIALOG(dialog), GWY_DATA_VIEW(dataview), FALSE);

    table = gui.table_options = gwy_param_table_new(args->params);
    gwy_param_table_append_slider(table, PARAM_Z);
    gwy_param_table_slider_restrict_range(table, PARAM_Z, 0, gwy_brick_get_zres(brick)-1);
    gwy_param_table_append_slider(table, PARAM_MAX_WIDTH);
    gwy_param_table_set_unitstr(table, PARAM_MAX_WIDTH, _("px"));
    gwy_param_table_slider_set_mapping(table, PARAM_MAX_WIDTH, GWY_SCALE_MAPPING_LINEAR);
    gwy_param_table_append_slider(table, PARAM_MIN_LENGTH);
    gwy_param_table_set_unitstr(table, PARAM_MIN_LENGTH, _("px"));

    gwy_param_table_append_slider(table, PARAM_THRESHOLD_HIGH);
    gwy_param_table_set_unitstr(table, PARAM_THRESHOLD_HIGH, _("RMS"));
    gwy_param_table_append_slider(table, PARAM_THRESHOLD_LOW);
    gwy_param_table_set_unitstr(table, PARAM_THRESHOLD_LOW, _("RMS"));

    gwy_param_table_append_separator(table);
    gwy_param_table_append_radio(table, PARAM_TYPE);

    gwy_param_table_append_header(table, -1, _("Options"));
    gwy_param_table_append_image_mask_color(table, PARAM_MASK_COLOR, GWY_DATA_VIEW(dataview), data, id);
    gwy_param_table_append_checkbox(table, PARAM_UPDATE);

    gwy_dialog_add_param_table(dialog, table);
    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), TRUE, TRUE, 0);

    g_signal_connect_swapped(gui.table_options, "param-changed", G_CALLBACK(param_changed), &gui);
    g_signal_connect_after(dialog, "response", G_CALLBACK(dialog_response), &gui);
    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_IMMEDIATE, preview, &gui, NULL);

    outcome = gwy_dialog_run(dialog);

    g_object_unref(mask);
    g_object_unref(field);

    return outcome;
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    GwyParams *params = gui->args->params;
    GwyParamTable *table = gui->table_options;

    if (id == PARAM_THRESHOLD_HIGH || id == PARAM_THRESHOLD_LOW) {
        gdouble threshold_low = gwy_params_get_double(params, PARAM_THRESHOLD_LOW);
        gdouble threshold_high = gwy_params_get_double(params, PARAM_THRESHOLD_HIGH);
        if (threshold_high < threshold_low) {
            if (id == PARAM_THRESHOLD_HIGH)
                gwy_param_table_set_double(table, PARAM_THRESHOLD_LOW, threshold_high);
            else
                gwy_param_table_set_double(table, PARAM_THRESHOLD_HIGH, threshold_low);
        }
    }
    if (id != PARAM_MASK_COLOR && id != PARAM_UPDATE)
        gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
dialog_response(G_GNUC_UNUSED GwyDialog *dialog, gint response, ModuleGUI *gui)
{
    if (response == GWY_RESPONSE_RESET)
        gwy_brick_copy_data(gui->args->brick, gui->args->result);

    preview(gui);
}

static void
update_image(ModuleGUI *gui, gint z)
{
    GwyBrick *brick = gui->args->result;
    GwyParams *params = gui->args->params;
    GwySignFlags type = gwy_params_get_enum(params, PARAM_TYPE);
    gdouble threshold_high = gwy_params_get_double(params, PARAM_THRESHOLD_HIGH);
    gdouble threshold_low = gwy_params_get_double(params, PARAM_THRESHOLD_LOW);
    gint min_len = gwy_params_get_int(params, PARAM_MIN_LENGTH);
    gint max_width = gwy_params_get_int(params, PARAM_MAX_WIDTH);
    gint zres = gwy_brick_get_zres(brick);

    GwyField *field = gwy_data_view_get_field(GWY_DATA_VIEW(gui->dataview));
    GwyNield *mask = gwy_data_view_get_mask(GWY_DATA_VIEW(gui->dataview));

    gwy_brick_extract_xy_plane(brick, field, CLAMP(z, 0, zres-1));
    gwy_field_mark_scars(field, mask, threshold_high, threshold_low, min_len, max_width, type);
    gwy_field_data_changed(field);
    gwy_nield_data_changed(mask);
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    gint z = gwy_params_get_int(gui->args->params, PARAM_Z);

    update_image(gui, z);
}

static gboolean
execute(ModuleArgs *args, GtkWindow *wait_window)
{
    GwyParams *params = args->params;
    GwyBrick *original = args->brick;
    GwyBrick *brick = args->result;
    GwySignFlags type = gwy_params_get_enum(params, PARAM_TYPE);
    gdouble threshold_high = gwy_params_get_double(params, PARAM_THRESHOLD_HIGH);
    gdouble threshold_low = gwy_params_get_double(params, PARAM_THRESHOLD_LOW);
    gint min_len = gwy_params_get_int(params, PARAM_MIN_LENGTH);
    gint max_width = gwy_params_get_int(params, PARAM_MAX_WIDTH);
    gint xres = gwy_brick_get_xres(brick), yres = gwy_brick_get_yres(brick), zres = gwy_brick_get_zres(brick);
    GwyField *field = gwy_field_new(xres, yres, gwy_brick_get_xreal(brick), gwy_brick_get_yreal(brick), FALSE);
    GwyNield *mask = gwy_field_new_nield_alike(field);
    gboolean cancelled = FALSE;

    gwy_app_wait_start(wait_window, _("Removing scars..."));

    for (gint k = 0; k < zres; k++) {
        gwy_brick_extract_xy_plane(original, field, k);
        gwy_field_mark_scars(field, mask, threshold_high, threshold_low, min_len, max_width, type);
        gwy_field_laplace_solve(field, mask, GWY_LAPLACE_MASKED, 1.0);
        gwy_brick_set_xy_plane(brick, field, k);

        if (!gwy_app_wait_set_fraction((gdouble)k/zres)) {
            cancelled = TRUE;
            break;
        }
    }
    gwy_app_wait_finish();

    g_object_unref(field);
    g_object_unref(mask);

    return !cancelled;
}

static void
sanitise_params(ModuleArgs *args)
{
    GwyParams *params = args->params;
    gdouble threshold_high = gwy_params_get_double(params, PARAM_THRESHOLD_HIGH);
    gdouble threshold_low = gwy_params_get_double(params, PARAM_THRESHOLD_LOW);

    if (threshold_low > threshold_high)
        gwy_params_set_double(params, PARAM_THRESHOLD_HIGH, threshold_low);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
