/*
 *  $Id: volumeops.c 29477 2026-02-14 13:29:30Z yeti-dn $
 *  Copyright (C) 2003 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <gwy.h>

#define RUN_MODES (GWY_RUN_IMMEDIATE)

static gboolean module_register(void);
static void     extract_preview(GwyFile *data,
                                GwyRunModeFlags mode);
static void     volume_invert  (GwyFile *data,
                                GwyRunModeFlags mode);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Basic volume data operations."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti)",
    "2017",
};

GWY_MODULE_QUERY2(module_info, volumeops)

static gboolean
module_register(void)
{
    gwy_volume_func_register("extract_preview",
                             &extract_preview,
                             N_("/_Basic Operations/Extract _Preview"),
                             NULL,
                             RUN_MODES,
                             GWY_MENU_FLAG_VOLUME,
                             N_("Extract volume data preview to an image"));
    gwy_volume_func_register("volume_invert",
                             volume_invert,
                             N_("/_Basic Operations/_Invert Value"),
                             GWY_ICON_VOLUME_INVERT,
                             RUN_MODES,
                             GWY_MENU_FLAG_VOLUME,
                             N_("Invert value in volume data"));

    return TRUE;
}

static void
extract_preview(GwyFile *data, GwyRunModeFlags mode)
{
    GwyField *dfield;
    gint id, newid;

    g_return_if_fail(mode & RUN_MODES);

    gwy_data_browser_get_current(GWY_APP_BRICK_ID, &id, 0);

    dfield = gwy_field_copy(gwy_dict_get_object(GWY_DICT(data), gwy_file_key_volume_picture(id)));

    g_return_if_fail(GWY_IS_FIELD(dfield));

    newid = gwy_file_add_image(data, dfield);
    gwy_file_set_visible(data, GWY_FILE_IMAGE, newid, TRUE);
    g_object_unref(dfield);

    gwy_file_pass_title(data, GWY_FILE_IMAGE, newid, gwy_file_get_display_title(data, GWY_FILE_VOLUME, id));
    gwy_log_add_full(data, GWY_FILE_IMAGE, -1, newid, "volume::extract_preview", NULL);
}

static void
volume_invert(GwyFile *data, GwyRunModeFlags mode)
{
    GwyBrick *brick = NULL;
    GwyField *dfield = NULL;
    gint id, newid;

    g_return_if_fail(mode & RUN_MODES);

    gwy_data_browser_get_current(GWY_APP_BRICK, &brick,
                                 GWY_APP_BRICK_ID, &id,
                                 0);

    dfield = gwy_field_copy(gwy_dict_get_object(GWY_DICT(data), gwy_file_key_volume_picture(id)));
    brick = gwy_brick_copy(brick);

    g_return_if_fail(GWY_IS_BRICK(brick));
    g_return_if_fail(GWY_IS_FIELD(dfield));

    gwy_field_multiply(dfield, -1.0);
    gwy_brick_multiply(brick, -1.0);

    newid = gwy_file_add_volume(data, brick, dfield);

    gwy_file_set_visible(data, GWY_FILE_VOLUME, newid, TRUE);
    gwy_file_set_title(data, GWY_FILE_VOLUME, newid, _("Inverted"), TRUE);
    gwy_file_sync_items(data, GWY_FILE_VOLUME, id,
                        data, GWY_FILE_VOLUME, newid,
                        GWY_FILE_ITEM_PALETTE, FALSE);

    g_object_unref(brick);
    g_object_unref(dfield);

    gwy_log_add(data, GWY_FILE_VOLUME, id, newid);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
